import { createContext, useContext, useState, useEffect, useRef, useCallback, ReactNode } from 'react';
import { useAuth } from '@/hooks/useAuth';
import { useToast } from '@/hooks/use-toast';
import { favoritesApi } from '@/lib/apiServices';

interface Favorite {
  id: string;
  item_id: string;
  item_type: 'food' | 'cosmetic';
}

interface FavoritesContextType {
  favorites: Favorite[];
  loading: boolean;
  addFavorite: (itemId: string, itemType: 'food' | 'cosmetic') => Promise<boolean>;
  removeFavorite: (itemId: string, itemType: 'food' | 'cosmetic') => Promise<boolean>;
  isFavorite: (itemId: string, itemType: 'food' | 'cosmetic') => boolean;
  toggleFavorite: (itemId: string, itemType: 'food' | 'cosmetic') => Promise<boolean>;
  refetch: () => Promise<void>;
}

const FavoritesContext = createContext<FavoritesContextType | undefined>(undefined);

export function FavoritesProvider({ children }: { children: ReactNode }) {
  const { user } = useAuth();
  const { toast } = useToast();
  const [favorites, setFavorites] = useState<Favorite[]>([]);
  const [loading, setLoading] = useState(false);
  const userIdRef = useRef<string | null>(null);

  const fetchFavorites = useCallback(async () => {
    const userId = userIdRef.current;
    if (!userId) return;
    
    setLoading(true);
    try {
      const data = await favoritesApi.getAll();
      setFavorites(data as Favorite[]);
    } catch (error) {
      console.error('Error fetching favorites:', error);
    }
    setLoading(false);
  }, []);

  useEffect(() => {
    userIdRef.current = user?.id ?? null;
    
    if (user) {
      fetchFavorites();
    } else {
      setFavorites([]);
    }
  }, [user, fetchFavorites]);

  const addFavorite = useCallback(async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to save favorites",
        variant: "destructive",
      });
      return false;
    }

    try {
      const data = await favoritesApi.add(itemId, itemType) as Favorite & { message?: string };
      
      // Add to local state: use returned row, or synthetic entry when server says "Already favorited"
      if (data && data.id) {
        setFavorites(prev => {
          if (prev.some(f => f.id === data.id)) return prev;
          return [...prev, data as Favorite];
        });
      } else if (data?.message?.toLowerCase().includes('already')) {
        setFavorites(prev => {
          if (prev.some(f => f.item_id === itemId && f.item_type === itemType)) return prev;
          return [...prev, { id: `existing-${itemId}-${itemType}`, item_id: itemId, item_type: itemType }];
        });
      }

      toast({
        title: "Added to favorites",
        description: `${itemType === 'food' ? 'Food' : 'Cosmetic'} added to your favorites`,
      });
      return true;
    } catch (error: any) {
      if (error.message?.includes('duplicate') || error.message?.includes('23505')) {
        setFavorites(prev => {
          if (prev.some(f => f.item_id === itemId && f.item_type === itemType)) return prev;
          return [...prev, { id: `existing-${itemId}-${itemType}`, item_id: itemId, item_type: itemType }];
        });
        toast({
          title: "Already favorited",
          description: "This item is already in your favorites",
        });
        return true;
      } else {
        toast({
          title: "Error",
          description: "Failed to add favorite",
          variant: "destructive",
        });
      }
      return false;
    }
  }, [user, toast]);

  const removeFavorite = useCallback(async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (!user) return false;

    // Optimistic update - remove from local state immediately
    setFavorites(prev => prev.filter(f => !(f.item_id === itemId && f.item_type === itemType)));

    try {
      await favoritesApi.remove(itemType, itemId);
      
      toast({
        title: "Removed from favorites",
        description: "Item removed from your favorites",
      });
      return true;
    } catch (error) {
      // Revert on error - refetch
      fetchFavorites();
      toast({
        title: "Error",
        description: "Failed to remove favorite",
        variant: "destructive",
      });
      return false;
    }
  }, [user, toast, fetchFavorites]);

  const isFavorite = useCallback((itemId: string, itemType: 'food' | 'cosmetic') => {
    return favorites.some(f => f.item_id === itemId && f.item_type === itemType);
  }, [favorites]);

  const toggleFavorite = useCallback(async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (isFavorite(itemId, itemType)) {
      return removeFavorite(itemId, itemType);
    } else {
      return addFavorite(itemId, itemType);
    }
  }, [isFavorite, addFavorite, removeFavorite]);

  return (
    <FavoritesContext.Provider value={{
      favorites,
      loading,
      addFavorite,
      removeFavorite,
      isFavorite,
      toggleFavorite,
      refetch: fetchFavorites,
    }}>
      {children}
    </FavoritesContext.Provider>
  );
}

export function useFavorites() {
  const context = useContext(FavoritesContext);
  if (context === undefined) {
    throw new Error('useFavorites must be used within a FavoritesProvider');
  }
  return context;
}
