import { createContext, useContext, useState, useEffect, useRef, useCallback, ReactNode } from 'react';
import { useAuth } from '@/hooks/useAuth';
import { useToast } from '@/hooks/use-toast';
import { shoppingListApi } from '@/lib/apiServices';

interface ShoppingListItem {
  id: string;
  item_id: string;
  item_type: 'food' | 'cosmetic';
  quantity: number;
  notes: string | null;
  is_purchased: boolean;
  created_at: string;
}

interface ShoppingListContextType {
  items: ShoppingListItem[];
  loading: boolean;
  addItem: (itemId: string, itemType: 'food' | 'cosmetic', notes?: string) => Promise<boolean>;
  removeItem: (itemId: string, itemType: 'food' | 'cosmetic') => Promise<boolean>;
  togglePurchased: (id: string, isPurchased: boolean) => Promise<boolean>;
  updateQuantity: (id: string, quantity: number) => Promise<boolean>;
  clearPurchased: () => Promise<boolean>;
  isInList: (itemId: string, itemType: 'food' | 'cosmetic') => boolean;
  toggleItem: (itemId: string, itemType: 'food' | 'cosmetic') => Promise<boolean>;
  refetch: () => Promise<void>;
}

const ShoppingListContext = createContext<ShoppingListContextType | undefined>(undefined);

export function ShoppingListProvider({ children }: { children: ReactNode }) {
  const { user } = useAuth();
  const { toast } = useToast();
  const [items, setItems] = useState<ShoppingListItem[]>([]);
  const [loading, setLoading] = useState(false);
  const userIdRef = useRef<string | null>(null);

  const fetchItems = useCallback(async () => {
    const userId = userIdRef.current;
    if (!userId) return;
    
    setLoading(true);
    try {
      const data = await shoppingListApi.getAll();
      setItems(data as ShoppingListItem[]);
    } catch (error) {
      console.error('Error fetching shopping list:', error);
    }
    setLoading(false);
  }, []);

  useEffect(() => {
    userIdRef.current = user?.id ?? null;
    
    if (user) {
      fetchItems();
    } else {
      setItems([]);
    }
  }, [user, fetchItems]);

  const addItem = useCallback(async (itemId: string, itemType: 'food' | 'cosmetic', notes?: string) => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to add items to your shopping list",
        variant: "destructive",
      });
      return false;
    }

    try {
      const data = await shoppingListApi.add(itemId, itemType, 1, notes) as ShoppingListItem;
      
      // Optimistic update - add to local state immediately
      if (data && data.id) {
        setItems(prev => {
          if (prev.some(i => i.id === data.id)) return prev;
          return [data, ...prev];
        });
      }

      toast({
        title: "Added to shopping list",
        description: "Item added to your shopping list",
      });
      return true;
    } catch (error: any) {
      if (error.message?.includes('duplicate') || error.message?.includes('23505')) {
        toast({
          title: "Already in list",
          description: "This item is already in your shopping list",
        });
      } else {
        toast({
          title: "Error",
          description: "Failed to add item to shopping list",
          variant: "destructive",
        });
      }
      return false;
    }
  }, [user, toast]);

  const removeItem = useCallback(async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (!user) return false;

    // Find the item to get its ID
    const item = items.find(i => i.item_id === itemId && i.item_type === itemType);
    if (!item) return false;

    // Optimistic update - remove from local state immediately
    setItems(prev => prev.filter(i => !(i.item_id === itemId && i.item_type === itemType)));

    try {
      await shoppingListApi.remove(item.id);
      
      toast({
        title: "Removed",
        description: "Item removed from shopping list",
      });
      return true;
    } catch (error) {
      // Revert on error - refetch
      fetchItems();
      toast({
        title: "Error",
        description: "Failed to remove item",
        variant: "destructive",
      });
      return false;
    }
  }, [user, items, toast, fetchItems]);

  const togglePurchased = useCallback(async (id: string, isPurchased: boolean) => {
    if (!user) return false;

    // Optimistic update
    setItems(prev => prev.map(i => i.id === id ? { ...i, is_purchased: isPurchased } : i));

    try {
      await shoppingListApi.update(id, { is_purchased: isPurchased });
      return true;
    } catch (error) {
      fetchItems();
      toast({
        title: "Error",
        description: "Failed to update item",
        variant: "destructive",
      });
      return false;
    }
  }, [user, toast, fetchItems]);

  const updateQuantity = useCallback(async (id: string, quantity: number) => {
    if (!user || quantity < 1) return false;

    // Optimistic update
    setItems(prev => prev.map(i => i.id === id ? { ...i, quantity } : i));

    try {
      await shoppingListApi.update(id, { quantity });
      return true;
    } catch (error) {
      fetchItems();
      toast({
        title: "Error",
        description: "Failed to update quantity",
        variant: "destructive",
      });
      return false;
    }
  }, [user, toast, fetchItems]);

  const clearPurchased = useCallback(async () => {
    if (!user) return false;

    // Optimistic update
    setItems(prev => prev.filter(i => !i.is_purchased));

    try {
      await shoppingListApi.clearPurchased();
      
      toast({
        title: "Cleared",
        description: "Purchased items removed from list",
      });
      return true;
    } catch (error) {
      fetchItems();
      toast({
        title: "Error",
        description: "Failed to clear purchased items",
        variant: "destructive",
      });
      return false;
    }
  }, [user, toast, fetchItems]);

  const isInList = useCallback((itemId: string, itemType: 'food' | 'cosmetic') => {
    return items.some(i => i.item_id === itemId && i.item_type === itemType);
  }, [items]);

  const toggleItem = useCallback(async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (isInList(itemId, itemType)) {
      return removeItem(itemId, itemType);
    } else {
      return addItem(itemId, itemType);
    }
  }, [isInList, addItem, removeItem]);

  return (
    <ShoppingListContext.Provider value={{
      items,
      loading,
      addItem,
      removeItem,
      togglePurchased,
      updateQuantity,
      clearPurchased,
      isInList,
      toggleItem,
      refetch: fetchItems,
    }}>
      {children}
    </ShoppingListContext.Provider>
  );
}

export function useShoppingList() {
  const context = useContext(ShoppingListContext);
  if (context === undefined) {
    throw new Error('useShoppingList must be used within a ShoppingListProvider');
  }
  return context;
}
