// Chemical hazard categories from CDC Fourth National Report on Human Exposure to Environmental Chemicals
// Focus on key categories: phthalates, BPA, parabens, heavy metals, pesticides

export interface ChemicalHazard {
  id: string;
  name: string;
  category: "phthalate" | "bisphenol" | "paraben" | "heavy_metal" | "pesticide" | "pfas" | "phytoestrogen" | "other";
  type: "phytoestrogen" | "xenoestrogen" | "endocrine_disruptor" | "carcinogen" | "neurotoxin";
  sources: string[];
  healthEffects: string[];
  cdcBiomarker?: string; // CDC biomonitoring chemical name if applicable
  potency?: "weak" | "moderate" | "strong"; // Estrogenic potency
  recommendation?: "avoid" | "limit" | "caution";
}

export interface EstrogenicProfile {
  hasPhytoestrogens: boolean;
  phytoestrogens?: string[]; // e.g., isoflavones, lignans, coumestans
  hasXenoestrogens: boolean;
  xenoestrogens?: string[]; // e.g., BPA, phthalates, pesticides
  estrogenicPotency: "none" | "low" | "moderate" | "high";
  healthConcerns?: string[];
}

// Comprehensive list of ALL known phytoestrogens - ALL marked as "Avoid"
export interface PhytoestrogensData {
  id: string;
  name: string;
  class: "isoflavone" | "lignan" | "coumestan" | "stilbene" | "flavonoid" | "mycoestrogen" | "other";
  potency: "weak" | "moderate" | "strong";
  foodSources: string[];
  recommendation: "avoid";
  healthConcerns: string[];
  notes?: string;
}

// COMPREHENSIVE PHYTOESTROGEN DATABASE - ALL IN "AVOID" CATEGORY
export const phytoestrogensDatabase: PhytoestrogensData[] = [
  // ==================== ISOFLAVONES (Strongest phytoestrogens) ====================
  {
    id: "genistein",
    name: "Genistein",
    class: "isoflavone",
    potency: "strong",
    foodSources: ["Soybeans", "Tofu", "Tempeh", "Soy milk", "Edamame", "Miso", "Soy protein isolate"],
    recommendation: "avoid",
    healthConcerns: [
      "Strong estrogen receptor binding",
      "Thyroid function interference",
      "May stimulate estrogen-sensitive cancers",
      "Endometrial cell proliferation",
      "Fertility concerns"
    ],
    notes: "Most potent dietary phytoestrogen; binds ERβ with high affinity"
  },
  {
    id: "daidzein",
    name: "Daidzein",
    class: "isoflavone",
    potency: "strong",
    foodSources: ["Soybeans", "Tofu", "Soy milk", "Tempeh", "Natto", "Soy sauce"],
    recommendation: "avoid",
    healthConcerns: [
      "Converts to equol in some individuals (more estrogenic)",
      "Hormonal disruption",
      "Thyroid concerns",
      "Reproductive system effects"
    ],
    notes: "Metabolized to equol by gut bacteria in ~30-50% of people"
  },
  {
    id: "glycitein",
    name: "Glycitein",
    class: "isoflavone",
    potency: "moderate",
    foodSources: ["Soybeans", "Soy germ", "Fermented soy products"],
    recommendation: "avoid",
    healthConcerns: [
      "Estrogenic activity",
      "Endocrine disruption potential"
    ],
    notes: "Third major soy isoflavone; found mainly in soy germ"
  },
  {
    id: "equol",
    name: "Equol",
    class: "isoflavone",
    potency: "strong",
    foodSources: ["Metabolite of daidzein", "Produced by gut bacteria from soy consumption"],
    recommendation: "avoid",
    healthConcerns: [
      "Highest estrogenic activity among isoflavone metabolites",
      "Anti-androgen effects",
      "May affect male hormones",
      "Fertility implications"
    ],
    notes: "Not directly consumed; produced by intestinal bacteria from daidzein"
  },
  {
    id: "formononetin",
    name: "Formononetin",
    class: "isoflavone",
    potency: "moderate",
    foodSources: ["Red clover", "Chickpeas", "Lentils", "Alfalfa", "Kudzu", "Licorice root"],
    recommendation: "avoid",
    healthConcerns: [
      "Converts to daidzein in the body",
      "Estrogenic metabolites",
      "Endocrine effects"
    ],
    notes: "Precursor to daidzein; found in legumes and herbs"
  },
  {
    id: "biochanin-a",
    name: "Biochanin A",
    class: "isoflavone",
    potency: "moderate",
    foodSources: ["Red clover", "Chickpeas", "Peanuts", "Alfalfa", "Soy"],
    recommendation: "avoid",
    healthConcerns: [
      "Converts to genistein in the body",
      "Estrogenic effects",
      "Aromatase modulation"
    ],
    notes: "Metabolized to genistein; used in menopause supplements"
  },
  {
    id: "puerarin",
    name: "Puerarin",
    class: "isoflavone",
    potency: "moderate",
    foodSources: ["Kudzu root", "Pueraria lobata"],
    recommendation: "avoid",
    healthConcerns: [
      "Estrogenic activity",
      "May affect cardiovascular system",
      "Hormone modulation"
    ],
    notes: "Main isoflavone in kudzu; used in traditional medicine"
  },
  {
    id: "irilone",
    name: "Irilone",
    class: "isoflavone",
    potency: "weak",
    foodSources: ["Red clover", "Iris species"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "Limited research"]
  },
  {
    id: "prunetin",
    name: "Prunetin",
    class: "isoflavone",
    potency: "weak",
    foodSources: ["Prunes", "Licorice", "Red clover"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "May affect estrogen metabolism"]
  },
  {
    id: "orobol",
    name: "Orobol",
    class: "isoflavone",
    potency: "weak",
    foodSources: ["Various legumes", "Herbal preparations"],
    recommendation: "avoid",
    healthConcerns: ["Estrogenic metabolite of genistein"]
  },
  {
    id: "santal",
    name: "Santal",
    class: "isoflavone",
    potency: "weak",
    foodSources: ["Sandalwood", "Some legumes"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic properties"]
  },

  // ==================== LIGNANS ====================
  {
    id: "secoisolariciresinol",
    name: "Secoisolariciresinol (SECO)",
    class: "lignan",
    potency: "moderate",
    foodSources: ["Flaxseeds", "Sesame seeds", "Sunflower seeds", "Pumpkin seeds", "Whole grains", "Berries"],
    recommendation: "avoid",
    healthConcerns: [
      "Converted to enterodiol and enterolactone (estrogenic)",
      "Hormone modulation",
      "May affect estrogen-sensitive conditions"
    ],
    notes: "Primary lignan in flaxseed; most abundant dietary lignan"
  },
  {
    id: "matairesinol",
    name: "Matairesinol",
    class: "lignan",
    potency: "moderate",
    foodSources: ["Sesame seeds", "Flaxseeds", "Rye", "Oats", "Barley", "Wheat bran"],
    recommendation: "avoid",
    healthConcerns: [
      "Estrogenic metabolites",
      "Converted to enterolactone by gut bacteria",
      "Hormone interference"
    ],
    notes: "Second most common dietary lignan"
  },
  {
    id: "pinoresinol",
    name: "Pinoresinol",
    class: "lignan",
    potency: "weak",
    foodSources: ["Sesame seeds", "Olive oil", "Rye", "Oats", "Brassica vegetables"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "Metabolized to enterolignans"
    ]
  },
  {
    id: "lariciresinol",
    name: "Lariciresinol",
    class: "lignan",
    potency: "weak",
    foodSources: ["Sesame seeds", "Kale", "Broccoli", "Brussels sprouts"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic potential",
      "Precursor to enterolignans"
    ]
  },
  {
    id: "enterodiol",
    name: "Enterodiol",
    class: "lignan",
    potency: "moderate",
    foodSources: ["Metabolite of plant lignans", "Produced from flax, sesame, whole grains"],
    recommendation: "avoid",
    healthConcerns: [
      "Estrogenic and anti-estrogenic effects",
      "May affect hormone balance",
      "Breast tissue effects"
    ],
    notes: "Mammalian lignan produced by gut bacteria from dietary lignans"
  },
  {
    id: "enterolactone",
    name: "Enterolactone",
    class: "lignan",
    potency: "moderate",
    foodSources: ["Metabolite of plant lignans", "From flaxseed, sesame, whole grains"],
    recommendation: "avoid",
    healthConcerns: [
      "Estrogen receptor binding",
      "May affect estrogen metabolism",
      "Potential effects on hormone-sensitive cancers"
    ],
    notes: "Most abundant mammalian lignan in plasma; gut bacteria metabolite"
  },
  {
    id: "sesamin",
    name: "Sesamin",
    class: "lignan",
    potency: "weak",
    foodSources: ["Sesame seeds", "Sesame oil"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic effects",
      "May affect lipid metabolism and hormones"
    ]
  },
  {
    id: "sesamolin",
    name: "Sesamolin",
    class: "lignan",
    potency: "weak",
    foodSources: ["Sesame seeds", "Sesame oil"],
    recommendation: "avoid",
    healthConcerns: ["Converted to sesamol; weak estrogenic potential"]
  },
  {
    id: "arctigenin",
    name: "Arctigenin",
    class: "lignan",
    potency: "weak",
    foodSources: ["Burdock root", "Arctium lappa"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "May affect hormone signaling"]
  },
  {
    id: "trachelogenin",
    name: "Trachelogenin",
    class: "lignan",
    potency: "weak",
    foodSources: ["Various medicinal plants"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic properties"]
  },
  {
    id: "hydroxymatairesinol",
    name: "Hydroxymatairesinol (HMR)",
    class: "lignan",
    potency: "weak",
    foodSources: ["Norway spruce knots", "Some grains"],
    recommendation: "avoid",
    healthConcerns: ["Estrogenic metabolites after gut conversion"]
  },
  {
    id: "syringaresinol",
    name: "Syringaresinol",
    class: "lignan",
    potency: "weak",
    foodSources: ["Grain bran", "Sesame", "Olive oil"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic potential"]
  },
  {
    id: "medioresinol",
    name: "Medioresinol",
    class: "lignan",
    potency: "weak",
    foodSources: ["Sesame seeds", "Rye bran"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity"]
  },

  // ==================== COUMESTANS ====================
  {
    id: "coumestrol",
    name: "Coumestrol",
    class: "coumestan",
    potency: "strong",
    foodSources: ["Alfalfa sprouts", "Clover sprouts", "Split peas", "Pinto beans", "Lima beans", "Soybean sprouts"],
    recommendation: "avoid",
    healthConcerns: [
      "Highest estrogenic potency among phytoestrogens",
      "Strong estrogen receptor binding",
      "Reproductive toxicity in animals",
      "May cause infertility at high doses",
      "Uterine effects"
    ],
    notes: "10-100x more potent than isoflavones; significant concerns"
  },
  {
    id: "4-methoxycoumestrol",
    name: "4'-Methoxycoumestrol",
    class: "coumestan",
    potency: "moderate",
    foodSources: ["Alfalfa", "Clover"],
    recommendation: "avoid",
    healthConcerns: ["Estrogenic activity similar to coumestrol"]
  },
  {
    id: "wedelolactone",
    name: "Wedelolactone",
    class: "coumestan",
    potency: "weak",
    foodSources: ["Eclipta alba (bhringraj)", "Wedelia calendulacea"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "Used in Ayurvedic medicine"]
  },

  // ==================== STILBENES ====================
  {
    id: "resveratrol",
    name: "Resveratrol",
    class: "stilbene",
    potency: "weak",
    foodSources: ["Red grapes", "Red wine", "Peanuts", "Berries", "Dark chocolate"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "May have mixed agonist/antagonist effects",
      "Affects aromatase enzyme",
      "Hormone modulation"
    ],
    notes: "Often marketed for health; still has estrogenic properties"
  },
  {
    id: "pterostilbene",
    name: "Pterostilbene",
    class: "stilbene",
    potency: "weak",
    foodSources: ["Blueberries", "Grapes", "Almonds"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic effects", "Related to resveratrol"]
  },
  {
    id: "piceatannol",
    name: "Piceatannol",
    class: "stilbene",
    potency: "weak",
    foodSources: ["Passion fruit", "Grapes", "Rhubarb"],
    recommendation: "avoid",
    healthConcerns: ["Metabolite of resveratrol with estrogenic properties"]
  },
  {
    id: "rhapontigenin",
    name: "Rhapontigenin",
    class: "stilbene",
    potency: "weak",
    foodSources: ["Rhubarb", "Rheum rhaponticum"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity"]
  },

  // ==================== FLAVONOIDS WITH ESTROGENIC ACTIVITY ====================
  {
    id: "quercetin",
    name: "Quercetin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Onions", "Apples", "Berries", "Grapes", "Broccoli", "Citrus", "Tea"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "May inhibit aromatase",
      "Mixed hormonal effects"
    ],
    notes: "Common antioxidant with weak phytoestrogenic properties"
  },
  {
    id: "kaempferol",
    name: "Kaempferol",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Kale", "Beans", "Tea", "Spinach", "Broccoli"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic receptor binding",
      "May affect estrogen metabolism"
    ]
  },
  {
    id: "luteolin",
    name: "Luteolin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Celery", "Parsley", "Broccoli", "Onion leaves", "Carrots", "Peppers", "Chamomile"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "Aromatase inhibition",
      "Hormone modulation"
    ]
  },
  {
    id: "apigenin",
    name: "Apigenin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Chamomile", "Parsley", "Celery", "Thyme", "Oregano", "Basil"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic effects",
      "Progesterone receptor interaction"
    ]
  },
  {
    id: "naringenin",
    name: "Naringenin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Grapefruit", "Oranges", "Lemons", "Tomatoes"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "May affect drug metabolism",
      "Hormone receptor binding"
    ]
  },
  {
    id: "hesperetin",
    name: "Hesperetin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Oranges", "Lemons", "Limes", "Citrus peel"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic properties"]
  },
  {
    id: "eriodictyol",
    name: "Eriodictyol",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Lemons", "Oregano", "Yerba santa"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity"]
  },
  {
    id: "catechin",
    name: "Catechin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Green tea", "Black tea", "Cocoa", "Apples", "Berries"],
    recommendation: "avoid",
    healthConcerns: [
      "Very weak estrogenic effects",
      "May affect estrogen metabolism"
    ]
  },
  {
    id: "epicatechin",
    name: "Epicatechin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Dark chocolate", "Green tea", "Cocoa", "Grapes"],
    recommendation: "avoid",
    healthConcerns: ["Very weak estrogenic activity"]
  },
  {
    id: "epigallocatechin-gallate",
    name: "Epigallocatechin gallate (EGCG)",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Green tea", "White tea", "Matcha"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "May affect thyroid at high doses",
      "Hormone modulation"
    ],
    notes: "Main catechin in green tea; popular supplement with hormonal effects"
  },
  {
    id: "myricetin",
    name: "Myricetin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Berries", "Grapes", "Walnuts", "Tea", "Vegetables"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic receptor binding"]
  },
  {
    id: "isorhamnetin",
    name: "Isorhamnetin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Onions", "Pears", "Olive oil", "Tomatoes"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic effects"]
  },
  {
    id: "chrysin",
    name: "Chrysin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Honey", "Propolis", "Passionflower"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "Aromatase inhibition claims (poorly absorbed)"
    ]
  },
  {
    id: "galangin",
    name: "Galangin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Propolis", "Galangal", "Honey"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity"]
  },
  {
    id: "fisetin",
    name: "Fisetin",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Strawberries", "Apples", "Persimmons", "Onions", "Cucumbers"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic effects"]
  },
  {
    id: "taxifolin",
    name: "Taxifolin (Dihydroquercetin)",
    class: "flavonoid",
    potency: "weak",
    foodSources: ["Onions", "Milk thistle", "French maritime pine"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic properties"]
  },

  // ==================== MYCOESTROGENS (Fungal estrogens) ====================
  {
    id: "zearalenone",
    name: "Zearalenone",
    class: "mycoestrogen",
    potency: "strong",
    foodSources: ["Contaminated corn", "Wheat", "Barley", "Oats", "Rice", "Sorghum"],
    recommendation: "avoid",
    healthConcerns: [
      "Potent estrogen mimic",
      "Reproductive toxicity",
      "Infertility risk",
      "Estrogenic cancer promotion",
      "Early puberty risk"
    ],
    notes: "Mycotoxin from Fusarium mold; often found in stored grains"
  },
  {
    id: "alpha-zearalenol",
    name: "α-Zearalenol",
    class: "mycoestrogen",
    potency: "strong",
    foodSources: ["Metabolite of zearalenone", "Contaminated grains"],
    recommendation: "avoid",
    healthConcerns: ["More estrogenic than parent compound", "Reproductive disruption"]
  },
  {
    id: "beta-zearalenol",
    name: "β-Zearalenol",
    class: "mycoestrogen",
    potency: "moderate",
    foodSources: ["Metabolite of zearalenone", "Contaminated grains"],
    recommendation: "avoid",
    healthConcerns: ["Estrogenic activity", "Hormone disruption"]
  },
  {
    id: "zearalanone",
    name: "Zearalanone",
    class: "mycoestrogen",
    potency: "moderate",
    foodSources: ["Metabolite of zearalenone", "Moldy grains"],
    recommendation: "avoid",
    healthConcerns: ["Estrogenic effects similar to zearalenone"]
  },

  // ==================== OTHER PHYTOESTROGENS ====================
  {
    id: "8-prenylnaringenin",
    name: "8-Prenylnaringenin",
    class: "other",
    potency: "strong",
    foodSources: ["Hops", "Beer"],
    recommendation: "avoid",
    healthConcerns: [
      "Most potent known phytoestrogen",
      "Derived from hops in beer",
      "Strong estrogen receptor binding",
      "May affect fertility"
    ],
    notes: "Found in hops; regular beer consumption leads to exposure"
  },
  {
    id: "isoxanthohumol",
    name: "Isoxanthohumol",
    class: "other",
    potency: "weak",
    foodSources: ["Beer", "Hops"],
    recommendation: "avoid",
    healthConcerns: [
      "Converted to 8-prenylnaringenin by gut bacteria",
      "Precursor to potent phytoestrogen"
    ]
  },
  {
    id: "xanthohumol",
    name: "Xanthohumol",
    class: "other",
    potency: "weak",
    foodSources: ["Hops", "Beer"],
    recommendation: "avoid",
    healthConcerns: ["Weak direct estrogenic activity", "Converted to more potent forms"]
  },
  {
    id: "glabridin",
    name: "Glabridin",
    class: "other",
    potency: "moderate",
    foodSources: ["Licorice root", "Glycyrrhiza glabra"],
    recommendation: "avoid",
    healthConcerns: [
      "Estrogenic activity",
      "May affect hormone-sensitive conditions",
      "Found in licorice supplements"
    ]
  },
  {
    id: "liquiritigenin",
    name: "Liquiritigenin",
    class: "other",
    potency: "weak",
    foodSources: ["Licorice root", "Glycyrrhiza species"],
    recommendation: "avoid",
    healthConcerns: ["Selective estrogen receptor binding", "Hormone effects"]
  },
  {
    id: "isoliquiritigenin",
    name: "Isoliquiritigenin",
    class: "other",
    potency: "weak",
    foodSources: ["Licorice root", "Bean sprouts"],
    recommendation: "avoid",
    healthConcerns: ["Estrogenic and anti-estrogenic properties"]
  },
  {
    id: "beta-sitosterol",
    name: "β-Sitosterol",
    class: "other",
    potency: "weak",
    foodSources: ["Vegetable oils", "Nuts", "Seeds", "Avocados", "Wheat germ"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic effects",
      "May affect testosterone metabolism",
      "Plant sterol with hormonal activity"
    ]
  },
  {
    id: "stigmasterol",
    name: "Stigmasterol",
    class: "other",
    potency: "weak",
    foodSources: ["Soybeans", "Vegetable oils", "Nuts", "Legumes"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic properties", "Steroid precursor"]
  },
  {
    id: "campesterol",
    name: "Campesterol",
    class: "other",
    potency: "weak",
    foodSources: ["Vegetable oils", "Nuts", "Seeds", "Wheat germ"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "Plant sterol"]
  },
  {
    id: "ursolic-acid",
    name: "Ursolic Acid",
    class: "other",
    potency: "weak",
    foodSources: ["Apple peels", "Rosemary", "Basil", "Cranberries", "Prunes"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "May affect hormone pathways"]
  },
  {
    id: "oleuropein",
    name: "Oleuropein",
    class: "other",
    potency: "weak",
    foodSources: ["Olive oil", "Olives", "Olive leaves"],
    recommendation: "avoid",
    healthConcerns: ["Very weak estrogenic effects", "Hormone modulation potential"]
  },
  {
    id: "curcumin",
    name: "Curcumin",
    class: "other",
    potency: "weak",
    foodSources: ["Turmeric", "Curry powder"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak phytoestrogenic activity",
      "May affect estrogen metabolism",
      "Aromatase modulation"
    ],
    notes: "Popular supplement; has weak but documented estrogenic effects"
  },
  {
    id: "ginsenosides",
    name: "Ginsenosides",
    class: "other",
    potency: "weak",
    foodSources: ["Ginseng", "Panax ginseng", "American ginseng"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogenic activity",
      "May affect hormone-sensitive conditions",
      "Multiple ginsenoside types with varying effects"
    ]
  },
  {
    id: "actein",
    name: "Actein (Triterpene glycosides)",
    class: "other",
    potency: "weak",
    foodSources: ["Black cohosh"],
    recommendation: "avoid",
    healthConcerns: [
      "Controversial estrogenic activity",
      "Used for menopause symptoms",
      "May have hormone effects"
    ]
  },
  {
    id: "hop-acids",
    name: "Hop Acids (Humulone, Lupulone)",
    class: "other",
    potency: "weak",
    foodSources: ["Hops", "Beer", "Herbal supplements"],
    recommendation: "avoid",
    healthConcerns: ["Weak estrogenic activity", "Related to hop phytoestrogens"]
  },
  {
    id: "indole-3-carbinol",
    name: "Indole-3-Carbinol (I3C)",
    class: "other",
    potency: "weak",
    foodSources: ["Broccoli", "Brussels sprouts", "Cabbage", "Cauliflower", "Kale"],
    recommendation: "avoid",
    healthConcerns: [
      "Mixed estrogen agonist/antagonist effects",
      "May affect estrogen metabolism",
      "Converts to DIM with complex hormonal effects"
    ],
    notes: "Often promoted as anti-estrogenic but has documented estrogenic properties"
  },
  {
    id: "diindolylmethane",
    name: "Diindolylmethane (DIM)",
    class: "other",
    potency: "weak",
    foodSources: ["Metabolite of I3C from cruciferous vegetables"],
    recommendation: "avoid",
    healthConcerns: [
      "Weak estrogen receptor binding",
      "Affects estrogen metabolism",
      "Can have both pro- and anti-estrogenic effects"
    ]
  },
  {
    id: "boron",
    name: "Boron (as dietary mineral)",
    class: "other",
    potency: "weak",
    foodSources: ["Almonds", "Raisins", "Dried fruits", "Avocados", "Wine"],
    recommendation: "avoid",
    healthConcerns: [
      "Increases estrogen and testosterone levels",
      "May affect hormone balance",
      "Essential mineral with hormonal effects"
    ],
    notes: "Trace mineral that affects sex hormone levels"
  }
];

// Get phytoestrogen by ID
export const getPhytoestrogenById = (id: string): PhytoestrogensData | undefined => {
  return phytoestrogensDatabase.find(p => p.id === id);
};

// Get phytoestrogens by class
export const getPhytoestrogensByClass = (className: PhytoestrogensData["class"]): PhytoestrogensData[] => {
  return phytoestrogensDatabase.filter(p => p.class === className);
};

// Get phytoestrogens by potency
export const getPhytoestrogensByPotency = (potency: PhytoestrogensData["potency"]): PhytoestrogensData[] => {
  return phytoestrogensDatabase.filter(p => p.potency === potency);
};

// Get all phytoestrogen names as a flat list for searching
export const getAllPhytoestrogenNames = (): string[] => {
  return phytoestrogensDatabase.map(p => p.name);
};

// Check if a compound is a phytoestrogen
export const isPhytoestrogen = (name: string): boolean => {
  const lowerName = name.toLowerCase();
  return phytoestrogensDatabase.some(p => 
    p.name.toLowerCase().includes(lowerName) || 
    lowerName.includes(p.name.toLowerCase())
  );
};

// Get phytoestrogens found in a specific food
export const getPhytoestrogensInFood = (foodName: string): PhytoestrogensData[] => {
  const lowerFood = foodName.toLowerCase();
  return phytoestrogensDatabase.filter(p => 
    p.foodSources.some(source => source.toLowerCase().includes(lowerFood))
  );
};


// Key CDC Fourth Report chemicals affecting food and cosmetics
export const chemicalHazards: ChemicalHazard[] = [
  // PHTHALATES
  {
    id: "dehp",
    name: "Di(2-ethylhexyl) phthalate (DEHP)",
    category: "phthalate",
    type: "xenoestrogen",
    sources: ["Food packaging", "Plastic containers", "Vinyl flooring", "Personal care products"],
    healthEffects: [
      "Reproductive toxicity",
      "Reduced testosterone",
      "Liver damage",
      "Thyroid disruption",
      "Increased obesity risk"
    ],
    cdcBiomarker: "Mono(2-ethylhexyl) phthalate (MEHP)"
  },
  {
    id: "dbp",
    name: "Dibutyl phthalate (DBP)",
    category: "phthalate",
    type: "xenoestrogen",
    sources: ["Nail polish", "Hair spray", "Perfume", "Plastics", "Medications"],
    healthEffects: [
      "Male reproductive disorders",
      "Decreased sperm quality",
      "Developmental toxicity",
      "Endocrine disruption"
    ],
    cdcBiomarker: "Mono-n-butyl phthalate (MnBP)"
  },
  {
    id: "dep",
    name: "Diethyl phthalate (DEP)",
    category: "phthalate",
    type: "xenoestrogen",
    sources: ["Fragrances", "Deodorants", "Lotions", "Shampoos", "Soaps"],
    healthEffects: [
      "Hormone disruption",
      "Allergic reactions",
      "Potential developmental effects"
    ],
    cdcBiomarker: "Mono-ethyl phthalate (MEP)"
  },
  
  // BISPHENOLS
  {
    id: "bpa",
    name: "Bisphenol A (BPA)",
    category: "bisphenol",
    type: "xenoestrogen",
    sources: ["Canned food linings", "Plastic bottles", "Receipts", "Food containers"],
    healthEffects: [
      "Estrogen mimicry",
      "Obesity and diabetes risk",
      "Heart disease risk",
      "Reproductive problems",
      "Brain development issues",
      "Increased cancer risk"
    ],
    cdcBiomarker: "Bisphenol A (BPA)"
  },
  {
    id: "bps",
    name: "Bisphenol S (BPS)",
    category: "bisphenol",
    type: "xenoestrogen",
    sources: ["BPA-free plastics", "Thermal paper", "Personal care products"],
    healthEffects: [
      "Similar endocrine disruption as BPA",
      "Hormonal imbalances",
      "Potential obesity link"
    ]
  },
  
  // PARABENS
  {
    id: "methylparaben",
    name: "Methylparaben",
    category: "paraben",
    type: "xenoestrogen",
    sources: ["Lotions", "Makeup", "Shampoo", "Shaving gel", "Toothpaste"],
    healthEffects: [
      "Weak estrogen mimicry",
      "Skin aging from UV exposure",
      "Potential breast cancer link",
      "Reproductive concerns"
    ],
    cdcBiomarker: "Methyl paraben"
  },
  {
    id: "propylparaben",
    name: "Propylparaben",
    category: "paraben",
    type: "xenoestrogen",
    sources: ["Cosmetics", "Pharmaceuticals", "Food preservatives"],
    healthEffects: [
      "Stronger estrogenic activity than methylparaben",
      "Reduced sperm production",
      "Decreased testosterone",
      "Skin sensitization"
    ],
    cdcBiomarker: "Propyl paraben"
  },
  
  // HEAVY METALS
  {
    id: "lead",
    name: "Lead",
    category: "heavy_metal",
    type: "neurotoxin",
    sources: ["Old paint", "Contaminated water", "Some imported spices", "Lipstick", "Rice"],
    healthEffects: [
      "Neurodevelopmental damage",
      "Cognitive impairment",
      "Hypertension",
      "Kidney damage",
      "Reproductive toxicity"
    ],
    cdcBiomarker: "Lead"
  },
  {
    id: "mercury",
    name: "Mercury",
    category: "heavy_metal",
    type: "neurotoxin",
    sources: ["Large predatory fish", "Dental amalgams", "Skin lightening creams", "Contaminated rice"],
    healthEffects: [
      "Neurological damage",
      "Memory impairment",
      "Kidney damage",
      "Fetal brain damage",
      "Vision and hearing loss"
    ],
    cdcBiomarker: "Total Mercury"
  },
  {
    id: "arsenic",
    name: "Arsenic",
    category: "heavy_metal",
    type: "carcinogen",
    sources: ["Rice", "Chicken", "Contaminated water", "Apple juice"],
    healthEffects: [
      "Increased cancer risk",
      "Cardiovascular disease",
      "Diabetes",
      "Skin lesions",
      "Developmental effects"
    ],
    cdcBiomarker: "Arsenic, Total"
  },
  {
    id: "cadmium",
    name: "Cadmium",
    category: "heavy_metal",
    type: "carcinogen",
    sources: ["Tobacco smoke", "Shellfish", "Leafy vegetables", "Grains", "Offal"],
    healthEffects: [
      "Kidney damage",
      "Bone disease",
      "Lung cancer",
      "Cardiovascular disease",
      "Reproductive toxicity"
    ],
    cdcBiomarker: "Cadmium"
  },
  
  // PESTICIDES
  {
    id: "glyphosate",
    name: "Glyphosate",
    category: "pesticide",
    type: "endocrine_disruptor",
    sources: ["Non-organic wheat", "Oats", "Corn", "Soy", "Legumes"],
    healthEffects: [
      "Gut microbiome disruption",
      "Potential carcinogen (IARC)",
      "Endocrine disruption",
      "Liver and kidney stress"
    ]
  },
  {
    id: "atrazine",
    name: "Atrazine",
    category: "pesticide",
    type: "endocrine_disruptor",
    sources: ["Corn", "Sorghum", "Sugarcane", "Contaminated water"],
    healthEffects: [
      "Hormonal disruption",
      "Feminization effects",
      "Birth defects risk",
      "Immune system effects"
    ],
    cdcBiomarker: "Atrazine mercapturate"
  },
  {
    id: "chlorpyrifos",
    name: "Chlorpyrifos",
    category: "pesticide",
    type: "neurotoxin",
    sources: ["Apples", "Citrus", "Grapes", "Tomatoes", "Broccoli"],
    healthEffects: [
      "Neurodevelopmental damage in children",
      "Reduced IQ",
      "ADHD-like symptoms",
      "Tremors"
    ],
    cdcBiomarker: "3,5,6-Trichloro-2-pyridinol"
  },
  
  // PFAS (Forever Chemicals)
  {
    id: "pfoa",
    name: "Perfluorooctanoic acid (PFOA)",
    category: "pfas",
    type: "endocrine_disruptor",
    sources: ["Non-stick cookware", "Food packaging", "Stain-resistant fabrics", "Microwave popcorn bags"],
    healthEffects: [
      "Thyroid disease",
      "Increased cholesterol",
      "Immune suppression",
      "Kidney and testicular cancer",
      "Liver damage"
    ],
    cdcBiomarker: "Perfluorooctanoic acid (PFOA)"
  },
  {
    id: "pfos",
    name: "Perfluorooctane sulfonate (PFOS)",
    category: "pfas",
    type: "endocrine_disruptor",
    sources: ["Firefighting foam", "Stain repellents", "Food packaging", "Cookware"],
    healthEffects: [
      "Immune system damage",
      "Thyroid disruption",
      "Reproductive issues",
      "Developmental delays"
    ],
    cdcBiomarker: "Perfluorooctane sulfonic acid (PFOS)"
  }
];

// Helper to get chemical by ID
export const getChemicalById = (id: string): ChemicalHazard | undefined => {
  return chemicalHazards.find(c => c.id === id);
};

// Helper to get chemicals by category
export const getChemicalsByCategory = (category: ChemicalHazard["category"]): ChemicalHazard[] => {
  return chemicalHazards.filter(c => c.category === category);
};

// Color coding for chemical types
export const getChemicalTypeColor = (type: ChemicalHazard["type"]): string => {
  switch (type) {
    case "xenoestrogen": return "text-pink-600 bg-pink-100";
    case "phytoestrogen": return "text-purple-600 bg-purple-100";
    case "endocrine_disruptor": return "text-orange-600 bg-orange-100";
    case "carcinogen": return "text-red-600 bg-red-100";
    case "neurotoxin": return "text-yellow-600 bg-yellow-100";
    default: return "text-gray-600 bg-gray-100";
  }
};

export const getEstrogenicPotencyColor = (potency: EstrogenicProfile["estrogenicPotency"]): string => {
  switch (potency) {
    case "none": return "text-green-600 bg-green-100";
    case "low": return "text-yellow-600 bg-yellow-100";
    case "moderate": return "text-orange-600 bg-orange-100";
    case "high": return "text-red-600 bg-red-100";
    default: return "text-gray-600 bg-gray-100";
  }
};
