// Cosmetics and personal care products with chemical hazard information
// Based on CDC Fourth Report chemicals and EWG Skin Deep data

import { EstrogenicProfile } from "./chemicalDatabase";

// TOBIN Score interface for cosmetics
export interface TobinScore {
  T: number; // Toxins (0-10)
  O: number; // Oxidative Stress (0-10)
  B: number; // Biological/Hormonal (0-10)
  I: number; // Inflammatory (0-10)
  N: number; // Nutrient Depleting (0-10)
  total: number; // Sum (0-50)
  notes?: string;
}

// Helper function to get TOBIN rating
export const getTobinRating = (total: number): { label: string; color: string; bgColor: string } => {
  if (total <= 10) return { label: 'Excellent', color: 'text-green-600', bgColor: 'bg-green-100' };
  if (total <= 20) return { label: 'Good', color: 'text-emerald-600', bgColor: 'bg-emerald-100' };
  if (total <= 30) return { label: 'Moderate', color: 'text-yellow-600', bgColor: 'bg-yellow-100' };
  if (total <= 40) return { label: 'Concern', color: 'text-orange-600', bgColor: 'bg-orange-100' };
  return { label: 'High Risk', color: 'text-red-600', bgColor: 'bg-red-100' };
};

export interface CosmeticItem {
  id: string;
  name: string;
  category: "skincare" | "makeup" | "haircare" | "fragrance" | "sunscreen" | "oral_care" | "deodorant" | "nail_care";
  emoji: string;
  
  // Risk assessment
  riskLevel: 1 | 2 | 3 | 4; // 1=High risk, 2=Moderate, 3=Low, 4=Safe/Clean
  riskName: string;
  
  // TOBIN Score (optional)
  tobinScore?: TobinScore;
  
  // Chemical concerns
  commonChemicals: string[];
  chemicalDetails: {
    chemical: string;
    purpose: string;
    concern: string;
  }[];
  
  // Estrogenic profile
  estrogenicProfile: EstrogenicProfile;
  
  // Health effects
  healthEffects: string[];
  
  // Alternatives
  saferAlternatives?: string[];
  
  // Additional info
  usageWarning?: string;
  regulatoryNotes?: string;
}

export const cosmeticsDatabase: CosmeticItem[] = [
  // HIGH RISK (Category 1)
  {
    id: "conventional-perfume",
    name: "Conventional Perfume/Fragrance",
    category: "fragrance",
    emoji: "🌸",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 7, O: 5, B: 9, I: 8, N: 3, total: 32, notes: "High phthalate and musk exposure, severe hormone disruption" },
    commonChemicals: ["Phthalates (DEP, DBP)", "Synthetic musks", "Benzyl compounds", "Aldehydes"],
    chemicalDetails: [
      { chemical: "Diethyl phthalate (DEP)", purpose: "Fragrance fixative", concern: "Endocrine disruption, reproductive effects" },
      { chemical: "Synthetic musks", purpose: "Long-lasting scent", concern: "Bioaccumulation, hormone disruption" },
      { chemical: "Benzaldehyde", purpose: "Almond scent", concern: "Respiratory irritation, CNS effects" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Phthalates", "Synthetic musks"],
      estrogenicPotency: "high",
      healthConcerns: ["Hormone disruption", "Allergies", "Headaches", "Respiratory issues"]
    },
    healthEffects: [
      "Endocrine disruption from phthalates",
      "Allergic reactions and contact dermatitis",
      "Respiratory sensitization",
      "Potential reproductive effects",
      "Bioaccumulation of synthetic musks"
    ],
    saferAlternatives: ["Essential oil-based perfumes", "Phthalate-free fragrances", "Solid perfumes with natural ingredients"],
    usageWarning: "'Fragrance' on labels can hide dozens of undisclosed chemicals"
  },
  {
    id: "nail-polish",
    name: "Nail Polish (Conventional)",
    category: "makeup",
    emoji: "💅",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 9, O: 6, B: 8, I: 5, N: 2, total: 30, notes: "Formaldehyde, toluene, DBP - severe toxin exposure" },
    commonChemicals: ["Dibutyl phthalate (DBP)", "Toluene", "Formaldehyde", "Triphenyl phosphate (TPHP)"],
    chemicalDetails: [
      { chemical: "Dibutyl phthalate", purpose: "Prevents chipping", concern: "Reproductive toxicity, hormone disruption" },
      { chemical: "Toluene", purpose: "Smooth application", concern: "Neurological damage, developmental toxicity" },
      { chemical: "Formaldehyde", purpose: "Nail hardener", concern: "Carcinogen, respiratory irritant" },
      { chemical: "TPHP", purpose: "Plasticizer", concern: "Hormone disruption, enters bloodstream within hours" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["DBP", "TPHP"],
      estrogenicPotency: "high",
      healthConcerns: ["Reproductive toxicity", "Endocrine disruption", "Carcinogenic exposure"]
    },
    healthEffects: [
      "TPHP detected in body within 10-14 hours of application",
      "Reproductive and developmental toxicity",
      "Respiratory irritation from vapors",
      "Known carcinogen exposure (formaldehyde)",
      "Neurological effects from toluene"
    ],
    saferAlternatives: ["10-free or 13-free nail polish brands", "Water-based nail polish", "Natural nail buffing"],
    regulatoryNotes: "DBP banned in EU cosmetics but still used in some US products"
  },
  {
    id: "anti-aging-cream",
    name: "Anti-Aging Cream (with Retinoids)",
    category: "skincare",
    emoji: "✨",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 5, O: 6, B: 7, I: 6, N: 3, total: 27, notes: "Parabens and BHT hormone disruption, sun sensitivity" },
    commonChemicals: ["Parabens", "BHT", "Retinyl palmitate", "Synthetic fragrances", "Oxybenzone"],
    chemicalDetails: [
      { chemical: "Parabens", purpose: "Preservative", concern: "Estrogen mimicry, found in breast tumors" },
      { chemical: "BHT", purpose: "Antioxidant preservative", concern: "Potential endocrine disruptor, organ toxicity" },
      { chemical: "Retinyl palmitate", purpose: "Anti-aging", concern: "May accelerate skin cancer with sun exposure" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Parabens", "BHT"],
      estrogenicPotency: "moderate",
      healthConcerns: ["Estrogen mimicry", "Skin sensitivity", "Potential carcinogenic interaction with UV"]
    },
    healthEffects: [
      "Parabens detected in breast cancer tissue",
      "Increased sun sensitivity",
      "Potential for accelerated photoaging",
      "Allergic reactions",
      "Hormone disruption"
    ],
    saferAlternatives: ["Bakuchiol (natural retinol alternative)", "Vitamin C serums", "Paraben-free formulations"],
    usageWarning: "Avoid sun exposure when using retinoids; use SPF protection"
  },
  {
    id: "conventional-lipstick",
    name: "Lipstick (Conventional)",
    category: "makeup",
    emoji: "💄",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 8, O: 4, B: 6, I: 5, N: 4, total: 27, notes: "Lead and heavy metal ingestion, cumulative toxicity" },
    commonChemicals: ["Lead", "Cadmium", "Chromium", "Aluminum", "Parabens", "BHA"],
    chemicalDetails: [
      { chemical: "Lead", purpose: "Color additive impurity", concern: "Neurotoxin, no safe level, accumulates in body" },
      { chemical: "Cadmium", purpose: "Pigment impurity", concern: "Carcinogen, kidney damage" },
      { chemical: "BHA", purpose: "Preservative", concern: "Potential carcinogen and endocrine disruptor" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["BHA", "Parabens"],
      estrogenicPotency: "moderate",
      healthConcerns: ["Heavy metal ingestion", "Cumulative toxicity", "Neurodevelopmental risks"]
    },
    healthEffects: [
      "Average woman ingests 4 pounds of lipstick in lifetime",
      "Lead accumulates in bones and organs",
      "Heavy metals cross blood-brain barrier",
      "Endocrine disruption from preservatives"
    ],
    saferAlternatives: ["Lead-tested clean beauty brands", "Mineral-based lipsticks", "Tinted lip balms"],
    regulatoryNotes: "FDA found lead in 400 lipsticks tested; no limit set for cosmetics"
  },
  
  // MODERATE RISK (Category 2)
  {
    id: "conventional-shampoo",
    name: "Shampoo (Conventional)",
    category: "haircare",
    emoji: "🧴",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 4, O: 3, B: 4, I: 4, N: 2, total: 17, notes: "Formaldehyde releasers, SLS irritation, paraben exposure" },
    commonChemicals: ["Sodium lauryl sulfate (SLS)", "Parabens", "Cocamide DEA", "Fragrance", "Formaldehyde releasers"],
    chemicalDetails: [
      { chemical: "Sodium lauryl sulfate", purpose: "Foaming agent", concern: "Skin irritation, strips natural oils" },
      { chemical: "Cocamide DEA", purpose: "Foam booster", concern: "Possible carcinogen, forms nitrosamines" },
      { chemical: "DMDM Hydantoin", purpose: "Preservative", concern: "Releases formaldehyde, carcinogen" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Parabens", "DEA compounds"],
      estrogenicPotency: "low",
      healthConcerns: ["Scalp irritation", "Hair loss", "Potential carcinogen exposure"]
    },
    healthEffects: [
      "Scalp irritation and dryness",
      "Formaldehyde exposure from preservatives",
      "Potential hormone disruption",
      "Allergic reactions"
    ],
    saferAlternatives: ["Sulfate-free shampoos", "Solid shampoo bars", "Natural/organic formulations"]
  },
  {
    id: "conventional-lotion",
    name: "Body Lotion (Conventional)",
    category: "skincare",
    emoji: "🧴",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 3, O: 3, B: 5, I: 4, N: 2, total: 17, notes: "Parabens hormone disruption, phthalates in fragrance, daily exposure" },
    commonChemicals: ["Parabens", "Mineral oil", "Petrolatum", "Fragrance", "BHT"],
    chemicalDetails: [
      { chemical: "Parabens", purpose: "Preservative", concern: "Estrogen mimicry" },
      { chemical: "Mineral oil", purpose: "Moisturizer", concern: "Petroleum-derived, may contain PAHs" },
      { chemical: "Fragrance", purpose: "Scent", concern: "Undisclosed phthalates and allergens" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Parabens", "Phthalates in fragrance"],
      estrogenicPotency: "moderate",
      healthConcerns: ["Daily application increases absorption", "Cumulative hormone effects"]
    },
    healthEffects: [
      "Skin absorbs up to 60% of applied products",
      "Daily use increases systemic exposure",
      "Hormone disruption from parabens",
      "Allergic sensitization"
    ],
    saferAlternatives: ["Shea butter", "Coconut oil", "Clean beauty lotions", "Unscented options"]
  },
  {
    id: "chemical-sunscreen",
    name: "Chemical Sunscreen",
    category: "sunscreen",
    emoji: "☀️",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 4, O: 3, B: 8, I: 3, N: 1, total: 19, notes: "Severe hormone disruption from oxybenzone, systemic absorption" },
    commonChemicals: ["Oxybenzone", "Octinoxate", "Homosalate", "Octocrylene", "Avobenzone"],
    chemicalDetails: [
      { chemical: "Oxybenzone", purpose: "UV filter", concern: "Hormone disruption, coral reef damage" },
      { chemical: "Octinoxate", purpose: "UV filter", concern: "Thyroid and reproductive effects" },
      { chemical: "Homosalate", purpose: "UV filter", concern: "Endocrine disruption, penetration enhancer" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Oxybenzone", "Octinoxate", "Homosalate"],
      estrogenicPotency: "high",
      healthConcerns: ["Systemic absorption", "Detected in bloodstream for weeks", "Hormone disruption"]
    },
    healthEffects: [
      "FDA found chemicals enter bloodstream after single use",
      "Detected in blood for up to 3 weeks",
      "Oxybenzone affects testosterone and estrogen",
      "Environmental damage to coral reefs"
    ],
    saferAlternatives: ["Mineral sunscreen (zinc oxide, titanium dioxide)", "Reef-safe formulations", "Protective clothing"],
    regulatoryNotes: "Hawaii banned oxybenzone and octinoxate to protect coral reefs"
  },
  {
    id: "conventional-deodorant",
    name: "Antiperspirant/Deodorant",
    category: "deodorant",
    emoji: "🧼",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 5, O: 2, B: 6, I: 4, N: 1, total: 18, notes: "Aluminum breast tissue accumulation, parabens found in tumors" },
    commonChemicals: ["Aluminum compounds", "Parabens", "Triclosan", "Propylene glycol", "Fragrance"],
    chemicalDetails: [
      { chemical: "Aluminum chlorohydrate", purpose: "Blocks sweat glands", concern: "Potential link to Alzheimer's and breast cancer" },
      { chemical: "Triclosan", purpose: "Antibacterial", concern: "Endocrine disruption, antibiotic resistance" },
      { chemical: "Parabens", purpose: "Preservative", concern: "Found in breast cancer tissue" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Parabens", "Triclosan"],
      estrogenicPotency: "moderate",
      healthConcerns: ["Applied near breast tissue daily", "Absorption through broken skin after shaving"]
    },
    healthEffects: [
      "Aluminum accumulation in breast tissue",
      "Parabens found in breast tumors",
      "Triclosan disrupts hormones",
      "Daily application near lymph nodes"
    ],
    saferAlternatives: ["Aluminum-free deodorants", "Crystal deodorants", "Natural baking soda formulas"],
    usageWarning: "Absorption increases significantly when applied to freshly shaved skin"
  },
  
  // LOW RISK (Category 3)
  {
    id: "mineral-makeup",
    name: "Mineral Makeup/Foundation",
    category: "makeup",
    emoji: "🎨",
    riskLevel: 3,
    riskName: "Low Risk",
    tobinScore: { T: 2, O: 1, B: 0, I: 2, N: 0, total: 5, notes: "Low risk, inhalation concern with loose powders only" },
    commonChemicals: ["Zinc oxide", "Titanium dioxide", "Iron oxides", "Mica"],
    chemicalDetails: [
      { chemical: "Zinc oxide", purpose: "Coverage and sun protection", concern: "Minimal; may cause dryness" },
      { chemical: "Titanium dioxide", purpose: "White pigment, UV protection", concern: "Inhalation risk with loose powders" },
      { chemical: "Mica", purpose: "Shimmer and texture", concern: "Inhalation risk; ethical sourcing concerns" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: ["Inhalation of fine particles"]
    },
    healthEffects: [
      "Generally well-tolerated",
      "May cause dryness in some skin types",
      "Loose powders pose inhalation risk",
      "Non-nano versions preferred for safety"
    ],
    saferAlternatives: ["Pressed mineral powders", "Non-nano formulations", "Certified organic options"]
  },
  {
    id: "natural-soap",
    name: "Natural/Castile Soap",
    category: "skincare",
    emoji: "🧼",
    riskLevel: 3,
    riskName: "Low Risk",
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "Gentle cleansing, minimal concerns, watch essential oils for children" },
    commonChemicals: ["Olive oil", "Coconut oil", "Lye (saponified)", "Essential oils"],
    chemicalDetails: [
      { chemical: "Saponified oils", purpose: "Cleansing", concern: "Minimal; can be drying for some" },
      { chemical: "Essential oils", purpose: "Natural fragrance", concern: "Potential allergen for sensitive individuals" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Trace amounts from plant oils"],
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: ["Rare allergic reactions to essential oils"]
    },
    healthEffects: [
      "Gentle cleansing without harsh chemicals",
      "May be drying without moisturizing oils",
      "Lavender and tea tree oils may have hormonal effects in children"
    ],
    saferAlternatives: ["Fragrance-free castile soap", "Oatmeal-based soaps"]
  },
  
  // SAFE/CLEAN (Category 4)
  {
    id: "mineral-sunscreen",
    name: "Mineral Sunscreen (Zinc/Titanium)",
    category: "sunscreen",
    emoji: "🌿",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Excellent choice - physical blockers, no systemic absorption" },
    commonChemicals: ["Zinc oxide", "Titanium dioxide"],
    chemicalDetails: [
      { chemical: "Zinc oxide", purpose: "Physical UV blocker", concern: "Non-nano versions sit on skin surface" },
      { chemical: "Titanium dioxide", purpose: "Physical UV blocker", concern: "Does not absorb into bloodstream" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Sits on skin surface without systemic absorption",
      "Immediate UV protection upon application",
      "Safe for sensitive skin and children",
      "Reef-safe"
    ],
    saferAlternatives: []
  },
  {
    id: "coconut-oil",
    name: "Pure Coconut Oil (Skincare)",
    category: "skincare",
    emoji: "🥥",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Pure natural moisturizer, antimicrobial, no concerns" },
    commonChemicals: ["Lauric acid", "Caprylic acid", "Vitamin E"],
    chemicalDetails: [
      { chemical: "Lauric acid", purpose: "Antimicrobial", concern: "None; beneficial fatty acid" },
      { chemical: "Medium-chain triglycerides", purpose: "Moisturizing", concern: "None" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Natural antimicrobial properties",
      "Effective moisturizer",
      "May be comedogenic for acne-prone skin"
    ],
    saferAlternatives: ["Jojoba oil for acne-prone skin", "Argan oil"]
  },
  {
    id: "shea-butter",
    name: "Pure Shea Butter",
    category: "skincare",
    emoji: "🧈",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Excellent natural moisturizer, anti-inflammatory, safe for babies" },
    commonChemicals: ["Stearic acid", "Oleic acid", "Vitamin A", "Vitamin E"],
    chemicalDetails: [
      { chemical: "Natural fatty acids", purpose: "Deep moisturizing", concern: "None" },
      { chemical: "Vitamin E", purpose: "Antioxidant", concern: "None; beneficial" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Anti-inflammatory properties",
      "Deep moisturizing without chemicals",
      "Safe for all ages including babies"
    ],
    saferAlternatives: []
  },
  {
    id: "natural-deodorant",
    name: "Natural Deodorant (Baking Soda-Free)",
    category: "deodorant",
    emoji: "🌱",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "No aluminum, no hormone disruptors, effective natural formula" },
    commonChemicals: ["Arrowroot powder", "Magnesium hydroxide", "Coconut oil", "Essential oils"],
    chemicalDetails: [
      { chemical: "Magnesium hydroxide", purpose: "Neutralizes odor", concern: "None; pH balanced" },
      { chemical: "Arrowroot powder", purpose: "Absorbs moisture", concern: "None" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "No aluminum or hormone disruptors",
      "Allows natural perspiration",
      "Effective odor control"
    ],
    saferAlternatives: []
  },
  
  // ORAL CARE PRODUCTS
  {
    id: "conventional-toothpaste",
    name: "Toothpaste (Conventional)",
    category: "oral_care",
    emoji: "🦷",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 3, O: 2, B: 4, I: 3, N: 1, total: 13, notes: "Triclosan thyroid disruption, SLS irritation, daily oral exposure" },
    commonChemicals: ["Triclosan", "Sodium lauryl sulfate", "Artificial sweeteners", "Titanium dioxide", "Propylene glycol"],
    chemicalDetails: [
      { chemical: "Triclosan", purpose: "Antibacterial", concern: "Endocrine disruption, antibiotic resistance" },
      { chemical: "Sodium lauryl sulfate", purpose: "Foaming agent", concern: "Mouth ulcers, irritation" },
      { chemical: "Artificial sweeteners", purpose: "Taste", concern: "Saccharin linked to cancer in studies" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Triclosan"],
      estrogenicPotency: "low",
      healthConcerns: ["Hormone disruption", "Gut microbiome effects"]
    },
    healthEffects: [
      "Triclosan can disrupt thyroid function",
      "SLS may cause canker sores",
      "Microplastics in some whitening formulas",
      "Fluoride concerns with overuse"
    ],
    saferAlternatives: ["Fluoride-free natural toothpaste", "Hydroxyapatite toothpaste", "Baking soda formulations"],
    regulatoryNotes: "FDA banned triclosan in hand soaps but still allowed in toothpaste"
  },
  {
    id: "mouthwash-alcohol",
    name: "Mouthwash (Alcohol-Based)",
    category: "oral_care",
    emoji: "💧",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 4, O: 3, B: 2, I: 4, N: 2, total: 15, notes: "Oral cancer link with long-term use, microbiome disruption" },
    commonChemicals: ["Alcohol (ethanol)", "Chlorhexidine", "Cetylpyridinium chloride", "Artificial dyes", "Saccharin"],
    chemicalDetails: [
      { chemical: "Alcohol", purpose: "Antiseptic carrier", concern: "Oral cancer risk with long-term use, dries mouth" },
      { chemical: "Chlorhexidine", purpose: "Antibacterial", concern: "Teeth staining, taste alteration, blood pressure effects" },
      { chemical: "Artificial dyes", purpose: "Color", concern: "Potential carcinogens, allergic reactions" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: ["Oral microbiome disruption", "Increased blood pressure"]
    },
    healthEffects: [
      "Alcohol-based mouthwash linked to increased oral cancer risk",
      "Disrupts oral microbiome",
      "Chlorhexidine can raise blood pressure",
      "Dry mouth increases cavity risk"
    ],
    saferAlternatives: ["Alcohol-free mouthwash", "Salt water rinse", "Oil pulling", "Xylitol-based rinses"],
    usageWarning: "Long-term daily use of alcohol mouthwash may increase oral cancer risk"
  },
  {
    id: "whitening-strips",
    name: "Teeth Whitening Strips",
    category: "oral_care",
    emoji: "✨",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 2, O: 3, B: 1, I: 4, N: 3, total: 13, notes: "Enamel damage, sensitivity, microplastic exposure" },
    commonChemicals: ["Hydrogen peroxide", "Carbamide peroxide", "Polyethylene", "PVP"],
    chemicalDetails: [
      { chemical: "Hydrogen peroxide", purpose: "Bleaching agent", concern: "Tooth sensitivity, enamel damage" },
      { chemical: "Carbamide peroxide", purpose: "Bleaching agent", concern: "Gum irritation, sensitivity" },
      { chemical: "Polyethylene", purpose: "Strip material", concern: "Microplastic exposure" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: ["Enamel erosion with overuse"]
    },
    healthEffects: [
      "Can cause tooth sensitivity",
      "May damage tooth enamel",
      "Gum irritation common",
      "Overuse leads to translucent teeth"
    ],
    saferAlternatives: ["Professional dental whitening", "Activated charcoal (occasional)", "Oil pulling"],
    usageWarning: "Do not exceed recommended usage frequency"
  },
  {
    id: "natural-toothpaste",
    name: "Natural/Hydroxyapatite Toothpaste",
    category: "oral_care",
    emoji: "🌿",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Excellent choice - remineralizes enamel, safe if swallowed" },
    commonChemicals: ["Hydroxyapatite", "Xylitol", "Calcium carbonate", "Essential oils"],
    chemicalDetails: [
      { chemical: "Hydroxyapatite", purpose: "Remineralization", concern: "None; naturally occurring in teeth" },
      { chemical: "Xylitol", purpose: "Sweetener, antimicrobial", concern: "None; reduces cavity-causing bacteria" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Remineralizes enamel naturally",
      "Safe if swallowed",
      "No fluoride concerns",
      "Xylitol fights cavity bacteria"
    ],
    saferAlternatives: []
  },
  
  // HAIR DYE PRODUCTS
  {
    id: "permanent-hair-dye",
    name: "Permanent Hair Dye",
    category: "haircare",
    emoji: "🎨",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 8, O: 5, B: 6, I: 5, N: 2, total: 26, notes: "PPD carcinogen, bladder cancer link, severe allergen" },
    commonChemicals: ["PPD (p-Phenylenediamine)", "Ammonia", "Resorcinol", "Hydrogen peroxide", "Coal tar"],
    chemicalDetails: [
      { chemical: "PPD", purpose: "Color developer", concern: "Severe allergic reactions, potential carcinogen, bladder cancer link" },
      { chemical: "Ammonia", purpose: "Opens hair cuticle", concern: "Respiratory irritation, asthma trigger" },
      { chemical: "Resorcinol", purpose: "Color coupling agent", concern: "Endocrine disruption, skin sensitizer" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Resorcinol", "4-ABP (byproduct)"],
      estrogenicPotency: "moderate",
      healthConcerns: ["Bladder cancer risk", "Hormone disruption", "Severe allergic reactions"]
    },
    healthEffects: [
      "PPD causes severe allergic reactions in some users",
      "Linked to increased bladder cancer risk",
      "Respiratory issues from ammonia fumes",
      "Scalp burns and hair damage",
      "Professional stylists have higher cancer rates"
    ],
    saferAlternatives: ["Henna (pure)", "Semi-permanent plant-based dyes", "Highlights/balayage (less scalp contact)"],
    usageWarning: "Always perform patch test 48 hours before use",
    regulatoryNotes: "PPD banned in some EU countries for skin contact but still used in hair dye"
  },
  {
    id: "semi-permanent-dye",
    name: "Semi-Permanent Hair Color",
    category: "haircare",
    emoji: "💜",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 4, O: 3, B: 3, I: 4, N: 1, total: 15, notes: "Lower risk than permanent, still sensitizing, paraben concern" },
    commonChemicals: ["HC dyes", "Basic dyes", "Ethanolamine", "Fragrance", "Preservatives"],
    chemicalDetails: [
      { chemical: "HC dyes", purpose: "Direct dyes", concern: "Less penetrating but still sensitizing" },
      { chemical: "Ethanolamine", purpose: "Ammonia alternative", concern: "Still irritating, may penetrate deeper" },
      { chemical: "Preservatives", purpose: "Shelf stability", concern: "May include parabens or formaldehyde releasers" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Parabens (if present)"],
      estrogenicPotency: "low",
      healthConcerns: ["Skin sensitization", "Cumulative exposure"]
    },
    healthEffects: [
      "Lower risk than permanent dyes",
      "Still can cause allergic reactions",
      "Less scalp penetration",
      "Hair damage from repeated use"
    ],
    saferAlternatives: ["Henna", "Vegetable-based dyes", "Color-depositing conditioners"]
  },
  {
    id: "hair-relaxer",
    name: "Hair Relaxer/Straightener",
    category: "haircare",
    emoji: "💇",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 9, O: 5, B: 9, I: 6, N: 2, total: 31, notes: "Uterine cancer link, formaldehyde carcinogen, severe hormone disruption" },
    commonChemicals: ["Formaldehyde", "Lye (sodium hydroxide)", "No-lye (calcium hydroxide)", "Cyclosiloxanes", "Parabens"],
    chemicalDetails: [
      { chemical: "Formaldehyde", purpose: "Hair straightening", concern: "Known carcinogen, respiratory issues" },
      { chemical: "Sodium hydroxide", purpose: "Breaks hair bonds", concern: "Severe scalp burns, hair loss" },
      { chemical: "Cyclosiloxanes", purpose: "Smoothing", concern: "Potential endocrine disruptor, bioaccumulative" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Cyclosiloxanes", "Parabens", "Phthalates"],
      estrogenicPotency: "high",
      healthConcerns: ["Uterine cancer link in studies", "Hormone disruption", "Reproductive effects"]
    },
    healthEffects: [
      "NIH study linked relaxers to increased uterine cancer risk",
      "Formaldehyde is a known carcinogen",
      "Chemical burns to scalp common",
      "Permanent hair damage and loss",
      "Respiratory issues for stylists"
    ],
    saferAlternatives: ["Heat styling", "Keratin treatments (formaldehyde-free)", "Embracing natural texture"],
    usageWarning: "Pregnant women should avoid; linked to reproductive health issues",
    regulatoryNotes: "FDA investigating hair relaxers; some formaldehyde products banned in EU/Canada"
  },
  {
    id: "henna-pure",
    name: "Pure Henna (Lawsonia)",
    category: "haircare",
    emoji: "🌱",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Excellent choice - natural dye used for millennia, conditions hair" },
    commonChemicals: ["Lawsone (natural dye)", "Plant matter"],
    chemicalDetails: [
      { chemical: "Lawsone", purpose: "Natural red-orange dye", concern: "None; used for millennia" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Trace plant compounds"],
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Conditions hair naturally",
      "No chemical burns or damage",
      "May strengthen hair",
      "Limited color range (red-orange to brown)"
    ],
    saferAlternatives: [],
    usageWarning: "Avoid 'black henna' which contains PPD"
  },
  
  // NAIL CARE PRODUCTS
  {
    id: "gel-nail-polish",
    name: "Gel Nail Polish/UV Gel",
    category: "nail_care",
    emoji: "💅",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 6, O: 4, B: 4, I: 5, N: 2, total: 21, notes: "HEMA permanent allergy risk, UV lamp cancer risk, nail damage" },
    commonChemicals: ["HEMA", "Methacrylates", "Photoinitiators", "Nitrocellulose", "Plasticizers"],
    chemicalDetails: [
      { chemical: "HEMA (2-Hydroxyethyl methacrylate)", purpose: "Adhesion", concern: "Severe allergic sensitization, permanent allergy" },
      { chemical: "Methacrylates", purpose: "Gel formation", concern: "Contact dermatitis, respiratory sensitization" },
      { chemical: "Photoinitiators", purpose: "UV curing", concern: "Skin sensitizer, potential carcinogen" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Plasticizers"],
      estrogenicPotency: "low",
      healthConcerns: ["Permanent allergy development", "Nail damage"]
    },
    healthEffects: [
      "HEMA can cause permanent nail product allergy",
      "UV lamps increase skin cancer risk on hands",
      "Weakens natural nails",
      "Removal process damages nail bed",
      "Respiratory issues for nail technicians"
    ],
    saferAlternatives: ["HEMA-free gel polish", "Regular nail polish", "Press-on nails"],
    usageWarning: "Apply sunscreen to hands before UV lamp exposure",
    regulatoryNotes: "HEMA restrictions proposed in EU cosmetics regulations"
  },
  {
    id: "acrylic-nails",
    name: "Acrylic Nail Products",
    category: "nail_care",
    emoji: "✨",
    riskLevel: 1,
    riskName: "High Risk",
    tobinScore: { T: 7, O: 4, B: 2, I: 6, N: 2, total: 21, notes: "MMA severe toxin, permanent nail damage, occupational asthma risk" },
    commonChemicals: ["MMA (Methyl methacrylate)", "EMA (Ethyl methacrylate)", "Benzoyl peroxide", "Acetone"],
    chemicalDetails: [
      { chemical: "MMA", purpose: "Hardening agent", concern: "Banned in many states; permanent nail damage, severe allergies" },
      { chemical: "EMA", purpose: "Safer alternative to MMA", concern: "Still sensitizing, respiratory irritant" },
      { chemical: "Benzoyl peroxide", purpose: "Initiator", concern: "Skin sensitizer, respiratory irritant" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: ["Respiratory damage for technicians", "Permanent allergies"]
    },
    healthEffects: [
      "MMA causes permanent nail bed damage",
      "Strong respiratory irritant",
      "Can cause permanent allergic sensitization",
      "Nail technicians at high risk for occupational asthma",
      "Fungal infections from trapped moisture"
    ],
    saferAlternatives: ["Dip powder (MMA-free)", "Press-on nails", "Natural nail strengtheners"],
    usageWarning: "Ensure salon uses EMA, not MMA",
    regulatoryNotes: "MMA banned in cosmetics in many US states but still illegally used in some salons"
  },
  {
    id: "nail-polish-remover",
    name: "Nail Polish Remover (Acetone)",
    category: "nail_care",
    emoji: "🧴",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 3, O: 2, B: 2, I: 3, N: 3, total: 13, notes: "Drying to nails, phthalates in fragrance, respiratory irritant" },
    commonChemicals: ["Acetone", "Ethyl acetate", "Isopropyl alcohol", "Fragrance"],
    chemicalDetails: [
      { chemical: "Acetone", purpose: "Dissolves polish", concern: "Dries nails/skin, respiratory irritant, flammable" },
      { chemical: "Ethyl acetate", purpose: "Gentler solvent", concern: "Less harsh but still drying" },
      { chemical: "Fragrance", purpose: "Mask chemical smell", concern: "May contain phthalates" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Phthalates in fragrance"],
      estrogenicPotency: "low",
      healthConcerns: ["Nail brittleness", "Skin dryness"]
    },
    healthEffects: [
      "Strips natural oils from nails and cuticles",
      "Can cause brittle, peeling nails",
      "Respiratory irritation in poorly ventilated areas",
      "Skin irritation with prolonged contact"
    ],
    saferAlternatives: ["Non-acetone removers", "Soy-based removers", "Peel-off base coats"]
  },
  {
    id: "cuticle-oil",
    name: "Natural Cuticle Oil",
    category: "nail_care",
    emoji: "💧",
    riskLevel: 4,
    riskName: "Safe/Clean",
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Excellent choice - natural oils strengthen nails, no concerns" },
    commonChemicals: ["Jojoba oil", "Vitamin E", "Sweet almond oil", "Essential oils"],
    chemicalDetails: [
      { chemical: "Jojoba oil", purpose: "Moisturizing", concern: "None; similar to skin's natural sebum" },
      { chemical: "Vitamin E", purpose: "Antioxidant, healing", concern: "None; beneficial for nails" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Trace amounts from plant oils"],
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Strengthens nails naturally",
      "Promotes healthy cuticles",
      "Prevents hangnails",
      "No harmful chemical exposure"
    ],
    saferAlternatives: []
  },
  {
    id: "clean-nail-polish",
    name: "10-Free Nail Polish",
    category: "nail_care",
    emoji: "🌈",
    riskLevel: 3,
    riskName: "Low Risk",
    tobinScore: { T: 2, O: 1, B: 0, I: 1, N: 0, total: 4, notes: "Free from 10 most toxic chemicals, safe for pregnant women" },
    commonChemicals: ["Nitrocellulose", "Butyl acetate", "Plant-derived plasticizers", "Natural pigments"],
    chemicalDetails: [
      { chemical: "Nitrocellulose", purpose: "Film former", concern: "Minimal; standard in all polish" },
      { chemical: "Butyl acetate", purpose: "Solvent", concern: "Low toxicity, some odor" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    healthEffects: [
      "Free from 10 most toxic chemicals",
      "No formaldehyde, toluene, or DBP",
      "Lower vapor toxicity",
      "Safer for pregnant women"
    ],
    saferAlternatives: ["Water-based nail polish", "Peel-off polish"]
  },
  
  // ADDITIONAL HAIRCARE
  {
    id: "dry-shampoo",
    name: "Dry Shampoo (Aerosol)",
    category: "haircare",
    emoji: "💨",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 6, O: 3, B: 3, I: 4, N: 2, total: 18, notes: "Benzene contamination recalls, talc asbestos risk, scalp damage" },
    commonChemicals: ["Benzene (contamination)", "Talc", "Propane/Butane", "Fragrance", "Aluminum starch"],
    chemicalDetails: [
      { chemical: "Benzene", purpose: "Unintentional contaminant", concern: "Known carcinogen; multiple recalls" },
      { chemical: "Talc", purpose: "Oil absorption", concern: "Potential asbestos contamination, respiratory issues" },
      { chemical: "Propellants", purpose: "Aerosol delivery", concern: "Inhalation risks, flammable" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Fragrance phthalates"],
      estrogenicPotency: "low",
      healthConcerns: ["Scalp buildup", "Hair follicle damage"]
    },
    healthEffects: [
      "Multiple brands recalled for benzene contamination",
      "Scalp irritation and buildup",
      "May contribute to hair loss with overuse",
      "Respiratory irritation from aerosol"
    ],
    saferAlternatives: ["Powder dry shampoo", "Arrowroot/cornstarch", "Less frequent washing"],
    usageWarning: "Check FDA recall list before purchasing",
    regulatoryNotes: "FDA issued multiple recalls for benzene-contaminated dry shampoos in 2022-2023"
  },
  {
    id: "hair-spray",
    name: "Hair Spray (Conventional)",
    category: "haircare",
    emoji: "💇‍♀️",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 3, O: 2, B: 3, I: 3, N: 1, total: 12, notes: "Phthalates in fragrance, respiratory irritation, scalp buildup" },
    commonChemicals: ["VA/Crotonates copolymer", "Alcohol denat", "Propellants", "Fragrance", "Silicones"],
    chemicalDetails: [
      { chemical: "Film-forming polymers", purpose: "Hold", concern: "Buildup, may contain formaldehyde" },
      { chemical: "Alcohol denat", purpose: "Quick drying", concern: "Drying to hair, flammable" },
      { chemical: "Fragrance", purpose: "Scent", concern: "Undisclosed chemicals, phthalates" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Phthalates in fragrance"],
      estrogenicPotency: "low",
      healthConcerns: ["Respiratory exposure", "Cumulative fragrance exposure"]
    },
    healthEffects: [
      "Respiratory irritation from inhalation",
      "Drying to hair",
      "Scalp buildup",
      "Eye irritation"
    ],
    saferAlternatives: ["Non-aerosol pump sprays", "Natural styling gels", "Sea salt sprays"]
  },
  
  // ADDITIONAL SKINCARE
  {
    id: "face-wash-acne",
    name: "Acne Face Wash (Conventional)",
    category: "skincare",
    emoji: "🧼",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 3, O: 3, B: 3, I: 5, N: 2, total: 16, notes: "Skin barrier disruption, triclosan concern, over-drying cycle" },
    commonChemicals: ["Benzoyl peroxide", "Salicylic acid", "Sodium lauryl sulfate", "Fragrance", "Triclosan"],
    chemicalDetails: [
      { chemical: "Benzoyl peroxide", purpose: "Kills acne bacteria", concern: "Skin dryness, irritation, bleaches fabrics" },
      { chemical: "Salicylic acid", purpose: "Exfoliation", concern: "Can be irritating, sun sensitivity" },
      { chemical: "SLS", purpose: "Foaming", concern: "Strips skin barrier, causes dryness" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Triclosan (if present)"],
      estrogenicPotency: "low",
      healthConcerns: ["Skin barrier disruption"]
    },
    healthEffects: [
      "Over-drying leads to increased oil production",
      "Can worsen acne if overused",
      "Irritation and peeling",
      "Sun sensitivity from acids"
    ],
    saferAlternatives: ["Gentle cleansers with niacinamide", "Oil cleansing method", "Sulfate-free formulations"]
  },
  {
    id: "body-wash-antibacterial",
    name: "Antibacterial Body Wash",
    category: "skincare",
    emoji: "🚿",
    riskLevel: 2,
    riskName: "Moderate Risk",
    tobinScore: { T: 4, O: 2, B: 5, I: 4, N: 1, total: 16, notes: "Triclosan hormone disruption, antibiotic resistance, microbiome damage" },
    commonChemicals: ["Triclosan/Triclocarban", "Benzalkonium chloride", "SLS/SLES", "Fragrance", "Parabens"],
    chemicalDetails: [
      { chemical: "Triclosan", purpose: "Antibacterial", concern: "Endocrine disruption, antibiotic resistance" },
      { chemical: "Triclocarban", purpose: "Antibacterial", concern: "Hormone disruption, environmental persistence" },
      { chemical: "SLES", purpose: "Foaming", concern: "May be contaminated with 1,4-dioxane" }
    ],
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["Triclosan", "Triclocarban", "Parabens"],
      estrogenicPotency: "moderate",
      healthConcerns: ["Hormone disruption", "Microbiome disruption"]
    },
    healthEffects: [
      "No more effective than regular soap",
      "Disrupts skin microbiome",
      "Contributes to antibiotic resistance",
      "Hormone disruption"
    ],
    saferAlternatives: ["Regular soap", "Castile soap", "Natural body wash"],
    regulatoryNotes: "FDA banned triclosan in consumer antiseptic washes in 2016"
  }
];

// Helper functions
export const getCosmeticById = (id: string): CosmeticItem | undefined => {
  return cosmeticsDatabase.find(c => c.id === id);
};

export const getCosmeticsByCategory = (category: CosmeticItem["category"]): CosmeticItem[] => {
  return cosmeticsDatabase.filter(c => c.category === category);
};

export const getCosmeticsByRisk = (riskLevel: CosmeticItem["riskLevel"]): CosmeticItem[] => {
  return cosmeticsDatabase.filter(c => c.riskLevel === riskLevel);
};

export const getRiskLevelColor = (riskLevel: number): string => {
  switch (riskLevel) {
    case 1: return "category-danger";
    case 2: return "category-warning";
    case 3: return "category-safe";
    case 4: return "category-therapeutic";
    default: return "muted";
  }
};

export const getRiskLevelName = (riskLevel: number): string => {
  switch (riskLevel) {
    case 1: return "High Risk";
    case 2: return "Moderate Risk";
    case 3: return "Low Risk";
    case 4: return "Safe/Clean";
    default: return "Unknown";
  }
};
