import { EstrogenicProfile } from "./chemicalDatabase";

// Dietary and certification classifications
export interface DietaryClassification {
  halal: "halal" | "haram" | "mashbooh" | "not_applicable"; // mashbooh = doubtful
  kosher: "kosher" | "not_kosher" | "pareve" | "dairy" | "meat" | "not_applicable";
  organic: "organic" | "non_organic" | "varies" | "not_applicable";
  certificationNotes?: string;
}

// TOBIN Score interface
export interface TobinScore {
  T: number; // Toxins (0-10)
  O: number; // Oxidative Stress (0-10)
  B: number; // Biological/Hormonal (0-10)
  I: number; // Inflammatory (0-10)
  N: number; // Nutrient Depleting (0-10)
  total: number; // Sum (0-50)
  notes?: string;
}

export interface FoodItem {
  id: string;
  name: string;
  category: 1 | 2 | 3 | 4;
  categoryName: string;
  emoji: string;
  
  // Raw nutritional data
  netCarbs: number; // grams
  fiber: number; // grams
  protein: number; // grams
  fat: number; // grams
  
  // Calculated factors
  absorptionCoefficient: number; // 0-1
  insulinSecretionFactor: number; // relative scale
  fatCarbCombinationFactor: number; // 0-2
  dnlConstant: number; // hepatic de novo lipogenesis
  ampkScore: number; // AMPK activation potential
  mitoEfficiency: number; // mitochondrial efficiency
  
  // Calculated indices
  glycogenicIndex: number; // GI*
  lipogenicIndex: number; // LI
  
  // TOBIN Score
  tobinScore?: TobinScore;
  
  // Dietary classifications
  dietaryClassification?: DietaryClassification;
  
  // Estrogenic and chemical profile
  estrogenicProfile?: EstrogenicProfile;
  chemicalContaminants?: string[]; // CDC chemicals commonly found
  contaminantWarning?: string;
  
  // Descriptive data
  mechanism: string;
  synergyWarning?: string;
  alternatives?: string[];
  benefits?: string[];
}

// Helper function to calculate TOBIN total
export const calculateTobinTotal = (score: Omit<TobinScore, 'total'>): number => {
  return score.T + score.O + score.B + score.I + score.N;
};

// Helper function to get TOBIN rating
export const getTobinRating = (total: number): { label: string; color: string; bgColor: string } => {
  if (total <= 10) return { label: 'Excellent', color: 'text-green-600', bgColor: 'bg-green-100' };
  if (total <= 20) return { label: 'Good', color: 'text-emerald-600', bgColor: 'bg-emerald-100' };
  if (total <= 30) return { label: 'Moderate', color: 'text-yellow-600', bgColor: 'bg-yellow-100' };
  if (total <= 40) return { label: 'Concern', color: 'text-orange-600', bgColor: 'bg-orange-100' };
  return { label: 'High Risk', color: 'text-red-600', bgColor: 'bg-red-100' };
};

// Helper functions for dietary classification colors
export const getHalalStatusColor = (status: DietaryClassification["halal"]): string => {
  switch (status) {
    case "halal": return "text-green-600 bg-green-100 dark:bg-green-900/30";
    case "haram": return "text-red-600 bg-red-100 dark:bg-red-900/30";
    case "mashbooh": return "text-yellow-600 bg-yellow-100 dark:bg-yellow-900/30";
    default: return "text-gray-500 bg-gray-100 dark:bg-gray-800";
  }
};

export const getKosherStatusColor = (status: DietaryClassification["kosher"]): string => {
  switch (status) {
    case "kosher": 
    case "pareve": return "text-blue-600 bg-blue-100 dark:bg-blue-900/30";
    case "dairy": return "text-cyan-600 bg-cyan-100 dark:bg-cyan-900/30";
    case "meat": return "text-amber-600 bg-amber-100 dark:bg-amber-900/30";
    case "not_kosher": return "text-red-600 bg-red-100 dark:bg-red-900/30";
    default: return "text-gray-500 bg-gray-100 dark:bg-gray-800";
  }
};

export const getOrganicStatusColor = (status: DietaryClassification["organic"]): string => {
  switch (status) {
    case "organic": return "text-emerald-600 bg-emerald-100 dark:bg-emerald-900/30";
    case "non_organic": return "text-orange-600 bg-orange-100 dark:bg-orange-900/30";
    case "varies": return "text-purple-600 bg-purple-100 dark:bg-purple-900/30";
    default: return "text-gray-500 bg-gray-100 dark:bg-gray-800";
  }
};

// Calculate GI* = (Cnet × kabs × Isec) / (Ffiber + Pbuffer)
function calculateGI(food: Omit<FoodItem, 'glycogenicIndex' | 'lipogenicIndex' | 'id' | 'categoryName'>): number {
  const numerator = food.netCarbs * food.absorptionCoefficient * food.insulinSecretionFactor;
  const denominator = food.fiber + (food.protein * 0.5) + 0.1; // +0.1 to avoid division by zero
  return Math.round((numerator / denominator) * 100) / 100;
}

// Calculate LI = ((Isec × Cnet) + (Fcomb × kDNL)) / (AMPKscore + Meff)
function calculateLI(food: Omit<FoodItem, 'glycogenicIndex' | 'lipogenicIndex' | 'id' | 'categoryName'>): number {
  const numerator = (food.insulinSecretionFactor * food.netCarbs) + (food.fatCarbCombinationFactor * food.dnlConstant);
  const denominator = food.ampkScore + food.mitoEfficiency + 0.1;
  return Math.round((numerator / denominator) * 100) / 100;
}

const rawFoods: Omit<FoodItem, 'glycogenicIndex' | 'lipogenicIndex' | 'id' | 'categoryName'>[] = [
  // CATEGORY 1 - Rapid Weight Gain Foods (High GI*, High LI)
  {
    name: "White Sugar",
    category: 1,
    emoji: "🍬",
    netCarbs: 100,
    fiber: 0,
    protein: 0,
    fat: 0,
    absorptionCoefficient: 1.0,
    insulinSecretionFactor: 1.8,
    fatCarbCombinationFactor: 0.5,
    dnlConstant: 2.0,
    ampkScore: 0.1,
    mitoEfficiency: 0.1,
    mechanism: "Maximum insulin spike, direct hepatic lipogenesis, zero nutritional buffering",
    synergyWarning: "Combined with fat creates maximum lipogenic response",
    alternatives: ["Stevia", "Monk fruit", "Small amounts of honey with protein"],
    tobinScore: { T: 2, O: 7, B: 3, I: 5, N: 8, total: 25, notes: "High oxidative stress, depletes B vitamins and minerals" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none"
    },
    dietaryClassification: { halal: "halal", kosher: "kosher", organic: "varies", certificationNotes: "Refined sugar is halal/kosher; organic cane sugar available" }
  },
  {
    name: "High Fructose Corn Syrup",
    category: 1,
    emoji: "🥤",
    netCarbs: 76,
    fiber: 0,
    protein: 0,
    fat: 0,
    absorptionCoefficient: 1.0,
    insulinSecretionFactor: 1.9,
    fatCarbCombinationFactor: 0.8,
    dnlConstant: 2.5,
    ampkScore: 0.05,
    mitoEfficiency: 0.1,
    mechanism: "Bypasses glycogen storage, directly converted to liver fat via de novo lipogenesis",
    synergyWarning: "Found in sodas, processed foods - compounds with other carbs",
    alternatives: ["Water", "Unsweetened tea", "Sparkling water with lemon"],
    chemicalContaminants: ["Mercury (from caustic soda processing)"],
    contaminantWarning: "HFCS production can involve mercury-containing caustic soda",
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none"
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "kosher",
      organic: "non_organic",
      certificationNotes: "Typically non-organic; highly processed ingredient"
    },
    tobinScore: { T: 4, O: 8, B: 2, I: 6, N: 9, total: 29, notes: "Mercury traces, extreme oxidative load, depletes nutrients" }
  },
  {
    name: "White Bread",
    category: 1,
    emoji: "🍞",
    netCarbs: 49,
    fiber: 2.7,
    protein: 9,
    fat: 3.2,
    absorptionCoefficient: 0.95,
    insulinSecretionFactor: 1.5,
    fatCarbCombinationFactor: 0.6,
    dnlConstant: 1.5,
    ampkScore: 0.2,
    mitoEfficiency: 0.3,
    mechanism: "Rapid glucose absorption, high insulin AUC, glycogen overflow to fat",
    synergyWarning: "With butter/spreads increases lipogenic potential significantly",
    alternatives: ["Almond flour bread", "Lettuce wraps", "Cloud bread"],
    chemicalContaminants: ["Glyphosate residues", "Potassium bromate"],
    contaminantWarning: "Non-organic wheat heavily sprayed with glyphosate as desiccant",
    tobinScore: { T: 5, O: 4, B: 3, I: 5, N: 4, total: 21, notes: "Glyphosate toxin load, inflammatory gluten" },
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Lignans (trace)"],
      hasXenoestrogens: true,
      xenoestrogens: ["Glyphosate"],
      estrogenicPotency: "low",
      healthConcerns: ["Glyphosate disrupts gut microbiome and hormone pathways"]
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "Check for organic wheat; some contain non-kosher additives"
    }
  },
  {
    name: "French Fries",
    category: 1,
    emoji: "🍟",
    netCarbs: 41,
    fiber: 3.8,
    protein: 3.4,
    fat: 15,
    absorptionCoefficient: 0.9,
    insulinSecretionFactor: 1.4,
    fatCarbCombinationFactor: 1.8,
    dnlConstant: 2.0,
    ampkScore: 0.1,
    mitoEfficiency: 0.2,
    mechanism: "High carb + seed oil combo maximizes fat storage, inflammatory omega-6",
    synergyWarning: "Acrylamide formation + oxidized oils compound metabolic damage",
    alternatives: ["Baked sweet potato wedges", "Roasted vegetables", "Cauliflower fries"],
    chemicalContaminants: ["Acrylamide", "PFAS (from packaging)"],
    contaminantWarning: "Acrylamide forms during high-heat frying; PFAS from grease-resistant packaging",
    tobinScore: { T: 7, O: 6, B: 4, I: 7, N: 3, total: 27, notes: "Acrylamide carcinogen, oxidized seed oils, PFAS exposure" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["PFAS from packaging"],
      estrogenicPotency: "low",
      healthConcerns: ["PFAS are endocrine disruptors that persist in body"]
    },
    dietaryClassification: {
      halal: "mashbooh",
      kosher: "pareve",
      organic: "non_organic",
      certificationNotes: "May be fried in shared oil with non-halal items; check frying oil source"
    }
  },
  {
    name: "Soda (Regular)",
    category: 1,
    emoji: "🥤",
    netCarbs: 39,
    fiber: 0,
    protein: 0,
    fat: 0,
    absorptionCoefficient: 1.0,
    insulinSecretionFactor: 1.7,
    fatCarbCombinationFactor: 0.3,
    dnlConstant: 2.2,
    ampkScore: 0.05,
    mitoEfficiency: 0.1,
    mechanism: "Liquid calories bypass satiety signals, rapid glucose-fructose delivery to liver",
    alternatives: ["Sparkling water", "Green tea", "Black coffee"],
    chemicalContaminants: ["BPA from can lining", "4-MEI (caramel color)"],
    contaminantWarning: "Canned sodas may contain BPA; caramel coloring contains potential carcinogen 4-MEI",
    tobinScore: { T: 5, O: 7, B: 5, I: 6, N: 8, total: 31, notes: "BPA hormone disruption, 4-MEI carcinogen, severe nutrient depletion" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["BPA from aluminum can linings"],
      estrogenicPotency: "moderate",
      healthConcerns: ["BPA mimics estrogen; linked to obesity, diabetes, reproductive issues"]
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "kosher",
      organic: "non_organic",
      certificationNotes: "Most major brands are certified; check for kosher symbol"
    }
  },
  {
    name: "Donut",
    category: 1,
    emoji: "🍩",
    netCarbs: 51,
    fiber: 1.7,
    protein: 4.9,
    fat: 25,
    absorptionCoefficient: 0.95,
    insulinSecretionFactor: 1.6,
    fatCarbCombinationFactor: 2.0,
    dnlConstant: 2.3,
    ampkScore: 0.1,
    mitoEfficiency: 0.15,
    mechanism: "Sugar + refined flour + trans fats = maximum lipogenic triple threat",
    synergyWarning: "Morning consumption on empty stomach amplifies insulin response",
    alternatives: ["Greek yogurt with berries", "Eggs", "Nut butter on celery"],
    tobinScore: { T: 4, O: 6, B: 4, I: 6, N: 6, total: 26, notes: "Trans fats, refined flour inflammation, oxidative stress" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none"
    },
    dietaryClassification: {
      halal: "mashbooh",
      kosher: "dairy",
      organic: "non_organic",
      certificationNotes: "Often contains lard or non-halal shortening; check ingredients"
    }
  },
  {
    name: "White Rice",
    category: 1,
    emoji: "🍚",
    netCarbs: 28,
    fiber: 0.4,
    protein: 2.7,
    fat: 0.3,
    absorptionCoefficient: 0.92,
    insulinSecretionFactor: 1.4,
    fatCarbCombinationFactor: 0.2,
    dnlConstant: 1.3,
    ampkScore: 0.2,
    mitoEfficiency: 0.3,
    mechanism: "High glycemic load, minimal fiber buffering, rapid glycogen filling",
    alternatives: ["Cauliflower rice", "Quinoa (smaller portions)", "Riced broccoli"],
    chemicalContaminants: ["Arsenic (inorganic)"],
    contaminantWarning: "Rice accumulates arsenic from soil and water; rinse and cook in excess water",
    tobinScore: { T: 5, O: 3, B: 2, I: 3, N: 4, total: 17, notes: "Arsenic toxin accumulation, nutrient stripped during processing" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: ["Arsenic is a carcinogen that accumulates in rice"]
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "Naturally halal/kosher; choose organic to reduce arsenic"
    }
  },
  {
    name: "Potato Chips",
    category: 1,
    emoji: "🥔",
    netCarbs: 50,
    fiber: 4.4,
    protein: 7,
    fat: 35,
    absorptionCoefficient: 0.88,
    insulinSecretionFactor: 1.3,
    fatCarbCombinationFactor: 1.9,
    dnlConstant: 2.1,
    ampkScore: 0.1,
    mitoEfficiency: 0.15,
    mechanism: "Hyperpalatable carb-fat combination with addictive flavor engineering",
    synergyWarning: "Designed to override satiety - 'bet you can't eat just one'",
    alternatives: ["Kale chips", "Roasted seaweed", "Cucumber slices with guacamole"],
    chemicalContaminants: ["Acrylamide", "PFAS from packaging"],
    contaminantWarning: "Acrylamide forms during frying; packaging may contain PFAS",
    tobinScore: { T: 6, O: 5, B: 3, I: 5, N: 4, total: 23, notes: "Acrylamide carcinogen, PFAS endocrine disruptors, oxidized oils" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: true,
      xenoestrogens: ["PFAS"],
      estrogenicPotency: "low"
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "Most plain chips halal/kosher; check flavored varieties for animal ingredients"
    }
  },
  
  // CATEGORY 2 - Conditional Foods (Moderate GI*, Moderate LI)
  {
    name: "Black Beans",
    category: 2,
    emoji: "🫘",
    netCarbs: 24,
    fiber: 15,
    protein: 21,
    fat: 1.4,
    absorptionCoefficient: 0.6,
    insulinSecretionFactor: 0.8,
    fatCarbCombinationFactor: 0.2,
    dnlConstant: 0.5,
    ampkScore: 0.7,
    mitoEfficiency: 0.6,
    mechanism: "High fiber slows absorption, resistant starch feeds gut bacteria",
    synergyWarning: "Can spike insulin in those with existing insulin resistance",
    alternatives: ["Smaller portions with high protein", "Lupini beans", "Edamame"],
    benefits: ["Prebiotic fiber", "Plant protein", "Magnesium source"],
    tobinScore: { T: 1, O: 2, B: 2, I: 2, N: 1, total: 8, notes: "Low risk, excellent fiber, supports gut health" },
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Isoflavones (low)", "Lignans"],
      hasXenoestrogens: false,
      estrogenicPotency: "low",
      healthConcerns: []
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "Naturally halal/kosher; organic reduces pesticide exposure"
    }
  },
  {
    name: "Oatmeal",
    category: 2,
    emoji: "🥣",
    netCarbs: 27,
    fiber: 4,
    protein: 5,
    fat: 3,
    absorptionCoefficient: 0.7,
    insulinSecretionFactor: 0.9,
    fatCarbCombinationFactor: 0.3,
    dnlConstant: 0.6,
    ampkScore: 0.5,
    mitoEfficiency: 0.5,
    mechanism: "Beta-glucan slows digestion, but still significant carb load",
    synergyWarning: "Instant varieties lose fiber benefit - choose steel cut",
    alternatives: ["Chia pudding", "Egg scramble", "Greek yogurt parfait"],
    chemicalContaminants: ["Glyphosate residues"],
    contaminantWarning: "Non-organic oats frequently contain glyphosate from pre-harvest spraying",
    tobinScore: { T: 3, O: 2, B: 2, I: 2, N: 2, total: 11, notes: "Glyphosate in non-organic, otherwise good fiber source" },
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Lignans", "Avenanthramides"],
      hasXenoestrogens: true,
      xenoestrogens: ["Glyphosate (if non-organic)"],
      estrogenicPotency: "low"
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "Choose organic to avoid glyphosate; some brands certified kosher"
    }
  },
  {
    name: "Whole Milk",
    category: 2,
    emoji: "🥛",
    netCarbs: 12,
    fiber: 0,
    protein: 8,
    fat: 8,
    absorptionCoefficient: 0.8,
    insulinSecretionFactor: 1.1,
    fatCarbCombinationFactor: 0.5,
    dnlConstant: 0.4,
    ampkScore: 0.4,
    mitoEfficiency: 0.5,
    mechanism: "Insulinogenic despite moderate GI due to whey protein response",
    alternatives: ["Unsweetened almond milk", "Coconut milk", "Heavy cream (small amounts)"],
    tobinScore: { T: 2, O: 1, B: 4, I: 2, N: 1, total: 10, notes: "Contains natural bovine estrogens, otherwise nutritious" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "low",
      healthConcerns: ["Contains natural mammalian estrogens from pregnant cows"]
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "dairy",
      organic: "varies",
      certificationNotes: "Must be from halal-slaughtered animals for strict interpretation; kosher requires supervision"
    }
  },
  {
    name: "Banana",
    category: 2,
    emoji: "🍌",
    netCarbs: 23,
    fiber: 2.6,
    protein: 1.1,
    fat: 0.3,
    absorptionCoefficient: 0.85,
    insulinSecretionFactor: 1.0,
    fatCarbCombinationFactor: 0.1,
    dnlConstant: 0.7,
    ampkScore: 0.4,
    mitoEfficiency: 0.5,
    mechanism: "High in potassium but significant sugar load, especially when ripe",
    synergyWarning: "Best consumed post-exercise when glycogen stores are depleted",
    alternatives: ["Berries", "Green apple slices", "Avocado"],
    chemicalContaminants: ["Chlorpyrifos (if non-organic)"],
    tobinScore: { T: 2, O: 2, B: 1, I: 1, N: 1, total: 7, notes: "Low risk, good potassium, watch pesticides if non-organic" },
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Lignans (trace)"],
      hasXenoestrogens: false,
      estrogenicPotency: "none"
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "All bananas halal/kosher; organic reduces pesticide exposure"
    }
  },
  {
    name: "Sweet Potato",
    category: 2,
    emoji: "🍠",
    netCarbs: 17,
    fiber: 3,
    protein: 1.6,
    fat: 0.1,
    absorptionCoefficient: 0.75,
    insulinSecretionFactor: 0.85,
    fatCarbCombinationFactor: 0.1,
    dnlConstant: 0.5,
    ampkScore: 0.5,
    mitoEfficiency: 0.55,
    mechanism: "Lower GI than white potato, antioxidants present, but still starchy",
    alternatives: ["Roasted butternut squash", "Mashed cauliflower", "Turnips"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 1, total: 5, notes: "Excellent antioxidants, beta-carotene, low toxin load" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none"
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "pareve",
      organic: "varies",
      certificationNotes: "Naturally halal/kosher; organic reduces pesticide exposure"
    }
  },
  {
    name: "Greek Yogurt (Plain)",
    category: 2,
    emoji: "🥄",
    netCarbs: 4,
    fiber: 0,
    protein: 17,
    fat: 5,
    absorptionCoefficient: 0.6,
    insulinSecretionFactor: 0.7,
    fatCarbCombinationFactor: 0.3,
    dnlConstant: 0.3,
    ampkScore: 0.6,
    mitoEfficiency: 0.6,
    mechanism: "High protein buffers insulin response, probiotics support gut health",
    benefits: ["Probiotics", "High protein", "Calcium source"],
    tobinScore: { T: 1, O: 1, B: 2, I: 1, N: 1, total: 6, notes: "Probiotic benefits, high protein, some natural hormones" },
    estrogenicProfile: {
      hasPhytoestrogens: false,
      hasXenoestrogens: false,
      estrogenicPotency: "none"
    },
    dietaryClassification: {
      halal: "halal",
      kosher: "dairy",
      organic: "varies",
      certificationNotes: "Check for gelatin (not halal/kosher unless specified); dairy products need kosher certification"
    }
  },
  
  // CATEGORY 3 - Metabolically Safe Foods (Low GI*, Low LI)
  {
    name: "Broccoli",
    category: 3,
    emoji: "🥦",
    netCarbs: 4,
    fiber: 2.6,
    protein: 2.8,
    fat: 0.4,
    absorptionCoefficient: 0.4,
    insulinSecretionFactor: 0.3,
    fatCarbCombinationFactor: 0.1,
    dnlConstant: 0.1,
    ampkScore: 0.8,
    mitoEfficiency: 0.8,
    mechanism: "High fiber, sulforaphane activates Nrf2, supports detoxification",
    benefits: ["Sulforaphane", "Vitamin C", "Folate", "Fiber"],
    chemicalContaminants: ["Chlorpyrifos (if non-organic)"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "Excellent detox support, anti-inflammatory, nutrient dense" },
    estrogenicProfile: {
      hasPhytoestrogens: true,
      phytoestrogens: ["Indole-3-carbinol (anti-estrogenic)"],
      hasXenoestrogens: false,
      estrogenicPotency: "none",
      healthConcerns: []
    },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Cruciferous vegetables are naturally halal/kosher; organic recommended" }
  },
  {
    name: "Spinach",
    category: 3,
    emoji: "🥬",
    netCarbs: 1.4,
    fiber: 2.2,
    protein: 2.9,
    fat: 0.4,
    absorptionCoefficient: 0.3,
    insulinSecretionFactor: 0.2,
    fatCarbCombinationFactor: 0.05,
    dnlConstant: 0.05,
    ampkScore: 0.85,
    mitoEfficiency: 0.85,
    mechanism: "Mineral-rich, virtually zero insulin impact, supports methylation",
    benefits: ["Iron", "Magnesium", "Nitrates for blood flow", "Folate"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "Nutrient powerhouse, supports detox and methylation" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies" }
  },
  {
    name: "Eggs",
    category: 3,
    emoji: "🥚",
    netCarbs: 1.1,
    fiber: 0,
    protein: 13,
    fat: 11,
    absorptionCoefficient: 0.3,
    insulinSecretionFactor: 0.4,
    fatCarbCombinationFactor: 0.2,
    dnlConstant: 0.1,
    ampkScore: 0.7,
    mitoEfficiency: 0.75,
    mechanism: "Complete protein, choline for brain health, minimal insulin response",
    benefits: ["Complete amino acids", "Choline", "B vitamins", "Vitamin D"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "Complete nutrition, choline for brain, minimal concerns" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Pasture-raised organic preferred" }
  },
  {
    name: "Salmon",
    category: 3,
    emoji: "🐟",
    netCarbs: 0,
    fiber: 0,
    protein: 20,
    fat: 13,
    absorptionCoefficient: 0.1,
    insulinSecretionFactor: 0.3,
    fatCarbCombinationFactor: 0.1,
    dnlConstant: 0.05,
    ampkScore: 0.9,
    mitoEfficiency: 0.9,
    mechanism: "Omega-3 improves insulin sensitivity, reduces inflammation",
    benefits: ["EPA/DHA omega-3", "Astaxanthin", "Vitamin D", "Selenium"],
    tobinScore: { T: 2, O: 1, B: 1, I: 0, N: 0, total: 4, notes: "Anti-inflammatory omega-3, watch mercury in large fish" },
    dietaryClassification: { halal: "halal", kosher: "kosher", organic: "varies", certificationNotes: "Fish with fins and scales are kosher; wild-caught preferred over farmed" }
  },
  {
    name: "Chicken Breast",
    category: 3,
    emoji: "🍗",
    netCarbs: 0,
    fiber: 0,
    protein: 31,
    fat: 3.6,
    absorptionCoefficient: 0.2,
    insulinSecretionFactor: 0.35,
    fatCarbCombinationFactor: 0.1,
    dnlConstant: 0.05,
    ampkScore: 0.75,
    mitoEfficiency: 0.7,
    mechanism: "Lean protein source, minimal metabolic disruption",
    benefits: ["High protein", "B vitamins", "Low calorie density"],
    tobinScore: { T: 2, O: 1, B: 1, I: 1, N: 0, total: 5, notes: "Clean protein, watch for antibiotics in non-organic" },
    dietaryClassification: { halal: "halal", kosher: "meat", organic: "varies", certificationNotes: "Must be halal/kosher slaughtered; organic reduces antibiotic exposure" }
  },
  {
    name: "Almonds",
    category: 3,
    emoji: "🌰",
    netCarbs: 10,
    fiber: 12.5,
    protein: 21,
    fat: 49,
    absorptionCoefficient: 0.35,
    insulinSecretionFactor: 0.3,
    fatCarbCombinationFactor: 0.4,
    dnlConstant: 0.15,
    ampkScore: 0.7,
    mitoEfficiency: 0.75,
    mechanism: "High fiber-to-carb ratio, healthy fats, protein buffers response",
    benefits: ["Vitamin E", "Magnesium", "Healthy monounsaturated fats"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "Excellent vitamin E, heart-healthy fats, nutrient dense" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies" }
  },
  {
    name: "Cauliflower",
    category: 3,
    emoji: "🥬",
    netCarbs: 3,
    fiber: 2,
    protein: 1.9,
    fat: 0.3,
    absorptionCoefficient: 0.35,
    insulinSecretionFactor: 0.25,
    fatCarbCombinationFactor: 0.05,
    dnlConstant: 0.05,
    ampkScore: 0.8,
    mitoEfficiency: 0.8,
    mechanism: "Versatile low-carb substitute, glucosinolates support detox",
    benefits: ["Vitamin C", "Sulforaphane precursors", "Fiber"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "Detox support, low-carb, anti-inflammatory" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Naturally halal/kosher; organic reduces pesticide residue" }
  },
  {
    name: "Grass-Fed Beef",
    category: 3,
    emoji: "🥩",
    netCarbs: 0,
    fiber: 0,
    protein: 26,
    fat: 15,
    absorptionCoefficient: 0.2,
    insulinSecretionFactor: 0.4,
    fatCarbCombinationFactor: 0.15,
    dnlConstant: 0.05,
    ampkScore: 0.7,
    mitoEfficiency: 0.75,
    mechanism: "CLA and omega-3 from grass-fed, iron and B12 for energy",
    benefits: ["CLA", "Iron", "B12", "Zinc", "Complete protein"],
    tobinScore: { T: 1, O: 1, B: 1, I: 1, N: 0, total: 4, notes: "CLA benefits, essential nutrients, grass-fed is optimal" },
    dietaryClassification: { halal: "halal", kosher: "meat", organic: "organic", certificationNotes: "Must be halal/kosher slaughtered and processed; grass-fed implies better practices" }
  },
  
  // CATEGORY 4 - Therapeutic / Anti-Lipogenic Foods
  {
    name: "Extra Virgin Olive Oil",
    category: 4,
    emoji: "🫒",
    netCarbs: 0,
    fiber: 0,
    protein: 0,
    fat: 100,
    absorptionCoefficient: 0.1,
    insulinSecretionFactor: 0.1,
    fatCarbCombinationFactor: 0.1,
    dnlConstant: 0.02,
    ampkScore: 1.2,
    mitoEfficiency: 1.1,
    mechanism: "Oleocanthal activates AMPK, polyphenols reduce inflammation",
    benefits: ["AMPK activation", "Anti-inflammatory oleocanthal", "Cardiovascular protection"],
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Therapeutic - anti-inflammatory, AMPK activation, heart protective" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Naturally halal/kosher; choose organic cold-pressed for purity" }
  },
  {
    name: "Avocado",
    category: 4,
    emoji: "🥑",
    netCarbs: 2,
    fiber: 7,
    protein: 2,
    fat: 15,
    absorptionCoefficient: 0.25,
    insulinSecretionFactor: 0.2,
    fatCarbCombinationFactor: 0.15,
    dnlConstant: 0.05,
    ampkScore: 1.0,
    mitoEfficiency: 0.95,
    mechanism: "Monounsaturated fat with fiber, potassium supports insulin sensitivity",
    benefits: ["Potassium", "Fiber", "Healthy fats", "Glutathione precursors"],
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Therapeutic - glutathione support, healthy fats, excellent potassium" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "All avocados halal/kosher; organic reduces pesticide exposure" }
  },
  {
    name: "Green Tea",
    category: 4,
    emoji: "🍵",
    netCarbs: 0,
    fiber: 0,
    protein: 0,
    fat: 0,
    absorptionCoefficient: 0.05,
    insulinSecretionFactor: 0.05,
    fatCarbCombinationFactor: 0,
    dnlConstant: 0.01,
    ampkScore: 1.3,
    mitoEfficiency: 1.2,
    mechanism: "EGCG activates AMPK and fat oxidation, catechins boost thermogenesis",
    benefits: ["EGCG", "L-theanine", "AMPK activation", "Fat oxidation boost"],
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Therapeutic - powerful antioxidants, fat burning, mental clarity" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Pure green tea is halal/kosher; organic Japanese varieties preferred for lower pesticides" }
  },
  {
    name: "Apple Cider Vinegar",
    category: 4,
    emoji: "🍎",
    netCarbs: 0.1,
    fiber: 0,
    protein: 0,
    fat: 0,
    absorptionCoefficient: 0.02,
    insulinSecretionFactor: 0.05,
    fatCarbCombinationFactor: 0,
    dnlConstant: 0.01,
    ampkScore: 1.1,
    mitoEfficiency: 1.0,
    mechanism: "Acetic acid reduces post-meal glucose and insulin spikes by 20-30%",
    benefits: ["Reduces insulin response", "Improves glucose uptake", "Supports digestion"],
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Therapeutic - blood sugar regulation, digestive support" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Naturally halal/kosher; choose raw organic with 'mother' for best benefits" }
  },
  {
    name: "Dark Chocolate (85%+)",
    category: 4,
    emoji: "🍫",
    netCarbs: 15,
    fiber: 11,
    protein: 8,
    fat: 43,
    absorptionCoefficient: 0.4,
    insulinSecretionFactor: 0.35,
    fatCarbCombinationFactor: 0.3,
    dnlConstant: 0.2,
    ampkScore: 1.1,
    mitoEfficiency: 1.0,
    mechanism: "Flavanols stimulate mitochondrial biogenesis, theobromine improves blood flow",
    benefits: ["Flavanols", "Mitochondrial biogenesis", "Mood enhancement"],
    tobinScore: { T: 1, O: 0, B: 0, I: 0, N: 0, total: 1, notes: "Therapeutic - watch for heavy metals in cacao, otherwise excellent" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Check for halal/kosher certification; some contain non-kosher emulsifiers; fair-trade organic preferred" }
  },
  {
    name: "Sauerkraut",
    category: 4,
    emoji: "🥬",
    netCarbs: 1.8,
    fiber: 2.9,
    protein: 0.9,
    fat: 0.1,
    absorptionCoefficient: 0.2,
    insulinSecretionFactor: 0.15,
    fatCarbCombinationFactor: 0.02,
    dnlConstant: 0.02,
    ampkScore: 1.0,
    mitoEfficiency: 0.95,
    mechanism: "Probiotics improve insulin sensitivity, organic acids support gut barrier",
    benefits: ["Probiotics", "Vitamin K2", "Improved insulin sensitivity"],
    tobinScore: { T: 0, O: 0, B: 0, I: 0, N: 0, total: 0, notes: "Therapeutic - probiotic powerhouse, gut healing, vitamin K2" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Naturally halal/kosher when made with just cabbage and salt; choose raw unpasteurized" }
  },
  {
    name: "Turmeric",
    category: 4,
    emoji: "🟡",
    netCarbs: 4.4,
    fiber: 2.1,
    protein: 0.8,
    fat: 0.3,
    absorptionCoefficient: 0.3,
    insulinSecretionFactor: 0.2,
    fatCarbCombinationFactor: 0.05,
    dnlConstant: 0.03,
    ampkScore: 1.2,
    mitoEfficiency: 1.1,
    mechanism: "Curcumin activates AMPK, reduces inflammation, improves insulin signaling",
    benefits: ["Curcumin", "Anti-inflammatory", "AMPK activation", "Liver support"],
    chemicalContaminants: ["Lead (some imported sources)"],
    contaminantWarning: "Some imported turmeric may contain lead; choose reputable organic sources",
    tobinScore: { T: 2, O: 0, B: 0, I: 0, N: 0, total: 2, notes: "Therapeutic - watch lead in imports, otherwise powerful anti-inflammatory" },
    dietaryClassification: { halal: "halal", kosher: "pareve", organic: "varies", certificationNotes: "Naturally halal/kosher; organic preferred to avoid lead contamination" }
  },
  {
    name: "Wild-Caught Sardines",
    category: 4,
    emoji: "🐟",
    netCarbs: 0,
    fiber: 0,
    protein: 25,
    fat: 11,
    absorptionCoefficient: 0.1,
    insulinSecretionFactor: 0.25,
    fatCarbCombinationFactor: 0.08,
    dnlConstant: 0.03,
    ampkScore: 1.1,
    mitoEfficiency: 1.05,
    mechanism: "High omega-3, CoQ10 supports mitochondria, low mercury small fish",
    benefits: ["Omega-3", "CoQ10", "Calcium (with bones)", "Vitamin D"],
    tobinScore: { T: 1, O: 0, B: 0, I: 0, N: 0, total: 1, notes: "Therapeutic - low mercury, omega-3 rich, excellent calcium source" },
    dietaryClassification: { halal: "halal", kosher: "kosher", organic: "not_applicable", certificationNotes: "Fish with fins and scales are kosher; wild-caught is the 'organic' equivalent for fish" }
  }
];

// Process all foods with calculated indices
export const foodDatabase: FoodItem[] = rawFoods.map((food, index) => {
  const gi = calculateGI(food);
  const li = calculateLI(food);
  
  const categoryNames: Record<number, string> = {
    1: "Rapid Weight Gain",
    2: "Conditional",
    3: "Metabolically Safe",
    4: "Therapeutic"
  };
  
  return {
    ...food,
    id: `food-${index}`,
    categoryName: categoryNames[food.category],
    glycogenicIndex: gi,
    lipogenicIndex: li
  };
});

export const getCategoryColor = (category: number): string => {
  switch (category) {
    case 1: return "category-danger";
    case 2: return "category-warning";
    case 3: return "category-safe";
    case 4: return "category-therapeutic";
    default: return "muted";
  }
};

export const getCategoryGradient = (category: number): string => {
  switch (category) {
    case 1: return "gradient-danger";
    case 2: return "from-warning to-orange-400";
    case 3: return "gradient-safe";
    case 4: return "gradient-therapeutic";
    default: return "from-muted to-muted";
  }
};

export const getIndexRating = (value: number, type: 'gi' | 'li'): { label: string; color: string } => {
  if (type === 'gi') {
    if (value < 0.6) return { label: "Low", color: "text-category-safe" };
    if (value < 1.2) return { label: "Moderate", color: "text-category-warning" };
    return { label: "High", color: "text-category-danger" };
  } else {
    if (value < 0.5) return { label: "Low", color: "text-category-safe" };
    if (value < 1.0) return { label: "Moderate", color: "text-category-warning" };
    return { label: "High", color: "text-category-danger" };
  }
};
