import { useState, useEffect, useCallback } from 'react';
import api from '@/lib/api';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';
import { useShoppingList } from './useShoppingList';
import { foodDatabase } from '@/data/foodDatabase';
import { cosmeticsDatabase } from '@/data/cosmeticsDatabase';

export interface Achievement {
  id: string;
  name: string;
  description: string;
  icon: string;
  category: 'beginner' | 'explorer' | 'health' | 'master' | 'special';
  rarity: 'common' | 'uncommon' | 'rare' | 'epic' | 'legendary';
  requirement: number;
  checkProgress: (context: AchievementContext) => number;
}

export interface UserAchievement {
  id: string;
  achievement_id: string;
  earned_at: string;
  progress: number;
}

interface AchievementContext {
  shoppingListItems: any[];
  totalExposure: number;
  itemsWithLowTobin: number;
  categoriesExplored: number;
  daysActive: number;
}

// Define all achievements
export const ACHIEVEMENTS: Achievement[] = [
  // Beginner achievements
  {
    id: 'first_item',
    name: 'First Steps',
    description: 'Add your first item to the shopping list',
    icon: '🎯',
    category: 'beginner',
    rarity: 'common',
    requirement: 1,
    checkProgress: (ctx) => Math.min(ctx.shoppingListItems.length, 1),
  },
  {
    id: 'five_items',
    name: 'Getting Started',
    description: 'Add 5 items to your shopping list',
    icon: '📝',
    category: 'beginner',
    rarity: 'common',
    requirement: 5,
    checkProgress: (ctx) => Math.min(ctx.shoppingListItems.length, 5),
  },
  {
    id: 'ten_items',
    name: 'Shopping Pro',
    description: 'Add 10 items to your shopping list',
    icon: '🛒',
    category: 'beginner',
    rarity: 'uncommon',
    requirement: 10,
    checkProgress: (ctx) => Math.min(ctx.shoppingListItems.length, 10),
  },
  
  // Health achievements
  {
    id: 'clean_start',
    name: 'Clean Start',
    description: 'Add an item with TOBIN score under 10',
    icon: '✨',
    category: 'health',
    rarity: 'common',
    requirement: 1,
    checkProgress: (ctx) => Math.min(ctx.itemsWithLowTobin, 1),
  },
  {
    id: 'health_conscious',
    name: 'Health Conscious',
    description: 'Add 5 items with TOBIN score under 15',
    icon: '💚',
    category: 'health',
    rarity: 'uncommon',
    requirement: 5,
    checkProgress: (ctx) => Math.min(ctx.itemsWithLowTobin, 5),
  },
  {
    id: 'wellness_warrior',
    name: 'Wellness Warrior',
    description: 'Add 10 items with TOBIN score under 15',
    icon: '🏆',
    category: 'health',
    rarity: 'rare',
    requirement: 10,
    checkProgress: (ctx) => Math.min(ctx.itemsWithLowTobin, 10),
  },
  {
    id: 'low_exposure',
    name: 'Low Exposure Champion',
    description: 'Keep total exposure under 100 with 5+ items',
    icon: '🛡️',
    category: 'health',
    rarity: 'rare',
    requirement: 1,
    checkProgress: (ctx) => ctx.shoppingListItems.length >= 5 && ctx.totalExposure < 100 ? 1 : 0,
  },
  {
    id: 'detox_master',
    name: 'Detox Master',
    description: 'Keep total exposure under 50 with 3+ items',
    icon: '🌿',
    category: 'health',
    rarity: 'epic',
    requirement: 1,
    checkProgress: (ctx) => ctx.shoppingListItems.length >= 3 && ctx.totalExposure < 50 ? 1 : 0,
  },
  
  // Explorer achievements
  {
    id: 'category_explorer',
    name: 'Category Explorer',
    description: 'Add items from 3 different categories',
    icon: '🗺️',
    category: 'explorer',
    rarity: 'uncommon',
    requirement: 3,
    checkProgress: (ctx) => Math.min(ctx.categoriesExplored, 3),
  },
  {
    id: 'variety_seeker',
    name: 'Variety Seeker',
    description: 'Add items from 5 different categories',
    icon: '🌈',
    category: 'explorer',
    rarity: 'rare',
    requirement: 5,
    checkProgress: (ctx) => Math.min(ctx.categoriesExplored, 5),
  },
  
  // Master achievements
  {
    id: 'tobin_expert',
    name: 'TOBIN Expert',
    description: 'Add 20 items to your shopping list',
    icon: '🎓',
    category: 'master',
    rarity: 'rare',
    requirement: 20,
    checkProgress: (ctx) => Math.min(ctx.shoppingListItems.length, 20),
  },
  {
    id: 'health_guru',
    name: 'Health Guru',
    description: 'Add 25 items with TOBIN score under 20',
    icon: '🧘',
    category: 'master',
    rarity: 'epic',
    requirement: 25,
    checkProgress: (ctx) => Math.min(ctx.itemsWithLowTobin, 25),
  },
  {
    id: 'legendary_shopper',
    name: 'Legendary Shopper',
    description: 'Add 50 items to your shopping list',
    icon: '👑',
    category: 'master',
    rarity: 'legendary',
    requirement: 50,
    checkProgress: (ctx) => Math.min(ctx.shoppingListItems.length, 50),
  },
  
  // Special achievements
  {
    id: 'perfect_score',
    name: 'Perfect Score',
    description: 'Add an item with TOBIN score of 5 or less',
    icon: '⭐',
    category: 'special',
    rarity: 'epic',
    requirement: 1,
    checkProgress: (ctx) => {
      const perfectItems = ctx.shoppingListItems.filter(item => {
        const score = item.tobinScore?.total || 50;
        return score <= 5;
      });
      return Math.min(perfectItems.length, 1);
    },
  },
  {
    id: 'zen_master',
    name: 'Zen Master',
    description: 'Have all items in list with TOBIN under 15',
    icon: '☯️',
    category: 'special',
    rarity: 'legendary',
    requirement: 1,
    checkProgress: (ctx) => {
      if (ctx.shoppingListItems.length < 5) return 0;
      const allLow = ctx.shoppingListItems.every(item => {
        const score = item.tobinScore?.total || 50;
        return score < 15;
      });
      return allLow ? 1 : 0;
    },
  },
];

export const useAchievements = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const { items: shoppingListItems } = useShoppingList();
  const [userAchievements, setUserAchievements] = useState<UserAchievement[]>([]);
  const [loading, setLoading] = useState(false);

  const fetchAchievements = useCallback(async () => {
    if (!user) return;
    
    setLoading(true);
    try {
      // Note: In self-hosted version, you'd need to add an achievements endpoint
      // For now, we'll store in localStorage as a fallback
      const stored = localStorage.getItem(`achievements_${user.id}`);
      if (stored) {
        setUserAchievements(JSON.parse(stored));
      }
    } catch (error) {
      console.error('Error fetching achievements:', error);
    }
    setLoading(false);
  }, [user]);

  useEffect(() => {
    if (user) {
      fetchAchievements();
    } else {
      setUserAchievements([]);
    }
  }, [user, fetchAchievements]);

  // Get items with TOBIN scores
  const itemsWithScores = shoppingListItems.map(item => {
    if (item.item_type === 'food') {
      const food = foodDatabase.find(f => f.id === item.item_id);
      return { ...item, tobinScore: food?.tobinScore, category: food?.categoryName };
    } else {
      const cosmetic = cosmeticsDatabase.find(c => c.id === item.item_id);
      return { ...item, tobinScore: cosmetic?.tobinScore, category: cosmetic?.category };
    }
  });

  // Build context for achievement checking
  const buildContext = useCallback((): AchievementContext => {
    const totalExposure = itemsWithScores.reduce((sum, item) => {
      return sum + (item.tobinScore?.total || 0) * (item.quantity || 1);
    }, 0);

    const itemsWithLowTobin = itemsWithScores.filter(item => {
      const score = item.tobinScore?.total || 50;
      return score < 15;
    }).length;

    const categories = new Set(itemsWithScores.map(item => item.category).filter(Boolean));

    return {
      shoppingListItems: itemsWithScores,
      totalExposure,
      itemsWithLowTobin,
      categoriesExplored: categories.size,
      daysActive: 1,
    };
  }, [itemsWithScores]);

  // Check and unlock achievements
  const checkAchievements = useCallback(async () => {
    if (!user) return;

    const context = buildContext();
    const newlyUnlocked: Achievement[] = [];

    for (const achievement of ACHIEVEMENTS) {
      const alreadyEarned = userAchievements.some(ua => ua.achievement_id === achievement.id);
      if (alreadyEarned) continue;

      const progress = achievement.checkProgress(context);
      if (progress >= achievement.requirement) {
        const newAchievement: UserAchievement = {
          id: `${user.id}_${achievement.id}`,
          achievement_id: achievement.id,
          earned_at: new Date().toISOString(),
          progress: progress,
        };
        
        newlyUnlocked.push(achievement);
        setUserAchievements(prev => {
          const updated = [...prev, newAchievement];
          localStorage.setItem(`achievements_${user.id}`, JSON.stringify(updated));
          return updated;
        });
      }
    }

    if (newlyUnlocked.length > 0) {
      // Show toast for each new achievement
      newlyUnlocked.forEach(achievement => {
        toast({
          title: `🎉 Achievement Unlocked!`,
          description: `${achievement.icon} ${achievement.name}: ${achievement.description}`,
        });
      });
    }
  }, [user, userAchievements, buildContext, toast]);

  // Check achievements when shopping list changes
  useEffect(() => {
    if (user && shoppingListItems.length > 0) {
      checkAchievements();
    }
  }, [shoppingListItems.length, user]);

  const getAchievementProgress = (achievementId: string): number => {
    const achievement = ACHIEVEMENTS.find(a => a.id === achievementId);
    if (!achievement) return 0;
    
    const context = buildContext();
    return achievement.checkProgress(context);
  };

  const isAchievementUnlocked = (achievementId: string): boolean => {
    return userAchievements.some(ua => ua.achievement_id === achievementId);
  };

  const getUnlockedCount = (): number => {
    return userAchievements.length;
  };

  const getTotalPoints = (): number => {
    const points = { common: 10, uncommon: 25, rare: 50, epic: 100, legendary: 250 };
    return userAchievements.reduce((sum, ua) => {
      const achievement = ACHIEVEMENTS.find(a => a.id === ua.achievement_id);
      return sum + (achievement ? points[achievement.rarity] : 0);
    }, 0);
  };

  return {
    achievements: ACHIEVEMENTS,
    userAchievements,
    loading,
    checkAchievements,
    getAchievementProgress,
    isAchievementUnlocked,
    getUnlockedCount,
    getTotalPoints,
    refetch: fetchAchievements,
  };
};
