import { useState } from 'react';
import { aiApi } from '@/lib/apiServices';

// Classify by name/categories so product is added to the correct section (food vs cosmetic)
// Matches Arabic etc. (e.g. لوشن = lotion) when OFF returns name/keywords in local language
function looksLikeCosmetic(name: string | null | undefined, categories: string | string[] | null | undefined): boolean {
  const catStr = categories == null ? '' : Array.isArray(categories) ? categories.join(', ') : categories;
  const combined = ((name || '') + ' ' + catStr).toLowerCase().replace(/-/g, ' ');
  return (
    /lotion|lotions|cream|shampoo|sunscreen|makeup|mascara|lipstick|perfume|deodorant|soap|skincare|haircare|cosmetic|beauty|nail polish|serum|moisturizer|moisturizers|cleanser|toner|exfoliant|face mask|body wash|hand cream|lip balm|hair dye|conditioner|styling|foundation|concealer|blush|eyeliner|eyeshadow|brow|spf|anti-aging|wrinkle|retinol|hyaluronic|paraben|fragrance|essential oil|body lotion|hair care|skin care|personal care|emulsion|dermatolog|topical|body care/.test(combined) ||
    /لوشن|كريم|شامبو|غسول|مرطب|مستحضر|عناية بالبشرة|عناية بالشعر|تجميل/.test(combined)
  );
}

export interface OpenFoodFactsProduct {
  name: string;
  brand: string | null;
  categories: string | null;
  imageUrl: string | null;
  ingredients: string | null;
  nutrition: {
    calories: number | null;
    fat: number | null;
    carbs: number | null;
    protein: number | null;
    fiber: number | null;
    sugar: number | null;
    salt: number | null;
  } | null;
  nutriscoreGrade: string | null;
  novaGroup: number | null;
}

export interface AIAnalysis {
  summary: string;
  safetyRating: 'Safe' | 'Moderate Concern' | 'High Concern';
  keyIngredients: string[];
  concerns: string[];
  benefits: string[];
  suitableFor: string[];
  avoidIf: string[];
  alternatives: string[];
  sources?: string[];
  /** TOBIN total (0-50) when provided by AI */
  tobinScore?: number | null;
  /** Glycogenic Index (GI*) when provided by AI or computed */
  glycogenicIndex?: number | null;
  /** Lipogenic Index (LI) when provided by AI or computed */
  lipogenicIndex?: number | null;
  /** Nutri-Score (a-e) when provided by AI */
  nutriscoreGrade?: string | null;
  /** NOVA group (1-4) when provided by AI */
  novaGroup?: number | null;
  /** Halal status: halal | haram | mashbooh | not_applicable */
  halal?: string | null;
  /** Kosher status: kosher | not_kosher | pareve | dairy | meat | not_applicable */
  kosher?: string | null;
  /** Organic: organic | non_organic | varies | not_applicable */
  organic?: string | null;
  /** Certification or cross-contact notes */
  certificationNotes?: string | null;
}

export interface BarcodeLookupResult {
  found: boolean;
  product?: OpenFoodFactsProduct;
  source: 'openfoodfacts' | 'openbeautyfacts' | 'local' | 'community' | 'ai';
  productType?: 'food' | 'cosmetic';
  aiGenerated?: boolean;
  aiAnalysis?: AIAnalysis | null;
  error?: string;
}

export const useBarcodeLookup = () => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const lookupBarcode = async (barcode: string): Promise<BarcodeLookupResult> => {
    setLoading(true);
    setError(null);

    try {
      const data = await aiApi.lookupBarcode(barcode);

      if (data.found && data.product) {
        const source = (data.aiGenerated ? 'ai' : data.source === 'openbeautyfacts' || data.source?.startsWith('openbeautyfacts') ? 'openbeautyfacts' : 'openfoodfacts') as BarcodeLookupResult['source'];
        const apiType = data.productType === 'cosmetic' ? 'cosmetic' : data.productType === 'food' ? 'food' : null;
        const productType: 'food' | 'cosmetic' = apiType ?? (looksLikeCosmetic(data.product.name, data.product.categories) ? 'cosmetic' : 'food');
        const raw = data.aiAnalysis || (data as any).ai_analysis;
        const aiAnalysis = raw ? {
          summary: raw.summary ?? '',
          safetyRating: (raw.safetyRating ?? raw.safety_rating) === 'Safe' ? 'Safe' : (raw.safetyRating ?? raw.safety_rating) === 'High Concern' ? 'High Concern' : 'Moderate Concern',
          keyIngredients: Array.isArray(raw.keyIngredients) ? raw.keyIngredients : Array.isArray(raw.key_ingredients) ? raw.key_ingredients : [],
          concerns: Array.isArray(raw.concerns) ? raw.concerns : [],
          benefits: Array.isArray(raw.benefits) ? raw.benefits : [],
          suitableFor: Array.isArray(raw.suitableFor) ? raw.suitableFor : Array.isArray(raw.suitable_for) ? raw.suitable_for : [],
          avoidIf: Array.isArray(raw.avoidIf) ? raw.avoidIf : Array.isArray(raw.avoid_if) ? raw.avoid_if : [],
          alternatives: Array.isArray(raw.alternatives) ? raw.alternatives : [],
          sources: Array.isArray(raw.sources) ? raw.sources : [],
          tobinScore: typeof (raw.tobinScore ?? raw.tobin_score) === 'number' ? raw.tobinScore ?? raw.tobin_score : null,
          glycogenicIndex: typeof (raw.glycogenicIndex ?? raw.glycogenic_index) === 'number' ? raw.glycogenicIndex ?? raw.glycogenic_index : null,
          lipogenicIndex: typeof (raw.lipogenicIndex ?? raw.lipogenic_index) === 'number' ? raw.lipogenicIndex ?? raw.lipogenic_index : null,
          nutriscoreGrade: raw.nutriscoreGrade ?? raw.nutriscore_grade ?? null,
          novaGroup: typeof (raw.novaGroup ?? raw.nova_group) === 'number' ? raw.novaGroup ?? raw.nova_group : null,
          halal: raw.halal ?? null,
          kosher: raw.kosher ?? null,
          organic: raw.organic ?? null,
          certificationNotes: raw.certificationNotes ?? raw.certification_notes ?? null,
        } : null;
        return {
          found: true,
          product: {
            name: data.product.name,
            brand: data.product.brand || null,
            categories: data.product.categories || null,
            imageUrl: data.product.image_url || null,
            ingredients: data.product.ingredients || null,
            nutrition: data.product.nutrition || null,
            nutriscoreGrade: data.product.nutriscore_grade ?? data.product.nutriscoreGrade ?? null,
            novaGroup: data.product.nova_group ?? data.product.novaGroup ?? null,
          },
          source,
          productType,
          aiGenerated: Boolean(data.aiGenerated),
          aiAnalysis,
        };
      }

      return { found: false, source: 'openfoodfacts', error: data.error };
    } catch (err) {
      console.error('Barcode lookup error:', err);
      const errorMessage = err instanceof Error ? err.message : 'Unknown error';
      setError(errorMessage);
      return { found: false, source: 'openfoodfacts', error: errorMessage };
    } finally {
      setLoading(false);
    }
  };

  return {
    lookupBarcode,
    loading,
    error,
  };
};
