import { useState, useCallback } from 'react';
import { Capacitor } from '@capacitor/core';
import { BarcodeScanner } from '@capacitor-community/barcode-scanner';
import { useToast } from './use-toast';

export const useBarcodeScanner = () => {
  const { toast } = useToast();
  const [isScanning, setIsScanning] = useState(false);
  const [lastScannedCode, setLastScannedCode] = useState<string | null>(null);

  const checkPermission = async (): Promise<boolean> => {
    if (!Capacitor.isNativePlatform()) {
      toast({
        title: "Native app required",
        description: "Barcode scanning requires the native mobile app",
        variant: "destructive",
      });
      return false;
    }

    const status = await BarcodeScanner.checkPermission({ force: false });

    if (status.granted) {
      return true;
    }

    if (status.denied) {
      toast({
        title: "Permission denied",
        description: "Please enable camera access in your device settings",
        variant: "destructive",
      });
      return false;
    }

    if (status.asked) {
      // User has been asked before but didn't grant
      const newStatus = await BarcodeScanner.checkPermission({ force: true });
      return newStatus.granted ?? false;
    }

    if (status.neverAsked) {
      const newStatus = await BarcodeScanner.checkPermission({ force: true });
      return newStatus.granted ?? false;
    }

    return false;
  };

  const startScan = useCallback(async (): Promise<string | null> => {
    const hasPermission = await checkPermission();
    if (!hasPermission) {
      return null;
    }

    setIsScanning(true);

    try {
      // Make background transparent for camera preview
      document.querySelector('body')?.classList.add('scanner-active');
      
      await BarcodeScanner.hideBackground();
      
      const result = await BarcodeScanner.startScan();

      if (result.hasContent && result.content) {
        setLastScannedCode(result.content);
        toast({
          title: "Barcode scanned",
          description: `Code: ${result.content}`,
        });
        return result.content;
      }

      return null;
    } catch (error) {
      console.error('Scan error:', error);
      toast({
        title: "Scan failed",
        description: "Could not read the barcode. Please try again.",
        variant: "destructive",
      });
      return null;
    } finally {
      setIsScanning(false);
      document.querySelector('body')?.classList.remove('scanner-active');
      await BarcodeScanner.showBackground();
    }
  }, [toast]);

  const stopScan = useCallback(async () => {
    await BarcodeScanner.stopScan();
    document.querySelector('body')?.classList.remove('scanner-active');
    await BarcodeScanner.showBackground();
    setIsScanning(false);
  }, []);

  const isNativePlatform = Capacitor.isNativePlatform();

  return {
    isScanning,
    lastScannedCode,
    startScan,
    stopScan,
    isNativePlatform,
  };
};
