import { useState, useEffect, useCallback } from 'react';
import { communityApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';
import { formatDistanceToNow } from 'date-fns';

export interface CommunityPost {
  id: string;
  user_id: string;
  author_name: string;
  content: string;
  category: 'tip' | 'question' | 'achievement' | 'discovery';
  likes_count: number;
  replies_count: number;
  created_at: string;
  isLiked?: boolean;
}

export interface PostReply {
  id: string;
  post_id: string;
  user_id: string;
  author_name: string;
  content: string;
  created_at: string;
}

export const useCommunity = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [posts, setPosts] = useState<CommunityPost[]>([]);
  const [userLikes, setUserLikes] = useState<Set<string>>(new Set());
  const [loading, setLoading] = useState(true);
  const [stats, setStats] = useState({ members: 0, posts: 0, tips: 0 });

  // Fetch posts
  const fetchPosts = useCallback(async () => {
    setLoading(true);
    
    try {
      const postsData = await communityApi.getPosts(undefined, 50, 0);

      // Note: In self-hosted version, you'd need to track likes differently
      // For now, we'll fetch fresh data each time
      setPosts(postsData as CommunityPost[]);
      
      // Update stats
      const tipsCount = postsData?.filter((p: any) => p.category === 'tip').length || 0;
      setStats({
        members: Math.floor(Math.random() * 5000) + 10000, // Simulated for now
        posts: postsData?.length || 0,
        tips: tipsCount,
      });
    } catch (error) {
      console.error('Error fetching posts:', error);
    }
    
    setLoading(false);
  }, []);

  // Set up initial load
  useEffect(() => {
    fetchPosts();
  }, [fetchPosts]);

  // Create a post
  const createPost = async (content: string, category: CommunityPost['category']) => {
    if (!user) {
      toast({
        title: 'Sign in required',
        description: 'Please sign in to post',
        variant: 'destructive',
      });
      return false;
    }

    try {
      const authorName = user.email?.split('@')[0] || 'Anonymous';
      await communityApi.createPost({
        content,
        category,
        author_name: authorName,
      });

      await fetchPosts();
      toast({
        title: 'Posted!',
        description: 'Your post has been shared with the community',
      });
      return true;
    } catch (error) {
      console.error('Error creating post:', error);
      toast({
        title: 'Error',
        description: 'Failed to create post',
        variant: 'destructive',
      });
      return false;
    }
  };

  // Toggle like on a post
  const toggleLike = async (postId: string) => {
    if (!user) {
      toast({
        title: 'Sign in required',
        description: 'Please sign in to like posts',
        variant: 'destructive',
      });
      return;
    }

    try {
      const { liked } = await communityApi.likePost(postId);
      
      // Update local state
      if (liked) {
        setUserLikes(prev => new Set(prev).add(postId));
        setPosts(prev => prev.map(p => 
          p.id === postId ? { ...p, likes_count: (p.likes_count || 0) + 1 } : p
        ));
      } else {
        setUserLikes(prev => {
          const next = new Set(prev);
          next.delete(postId);
          return next;
        });
        setPosts(prev => prev.map(p => 
          p.id === postId ? { ...p, likes_count: Math.max((p.likes_count || 1) - 1, 0) } : p
        ));
      }
    } catch (error) {
      console.error('Error toggling like:', error);
    }
  };

  // Delete a post (simplified - API doesn't have delete endpoint in community routes)
  const deletePost = async (postId: string) => {
    if (!user) return false;

    toast({
      title: 'Not available',
      description: 'Post deletion not implemented in self-hosted version',
      variant: 'destructive',
    });
    return false;
  };

  // Format time ago
  const formatTimeAgo = (dateString: string) => {
    try {
      return formatDistanceToNow(new Date(dateString), { addSuffix: true });
    } catch {
      return 'recently';
    }
  };

  // Get posts with isLiked flag
  const postsWithLikes = posts.map(post => ({
    ...post,
    isLiked: userLikes.has(post.id),
  }));

  // Fetch replies for a post
  const fetchReplies = async (postId: string): Promise<PostReply[]> => {
    try {
      const data = await communityApi.getReplies(postId);
      return data as PostReply[];
    } catch (error) {
      console.error('Error fetching replies:', error);
      return [];
    }
  };

  // Create a reply
  const createReply = async (postId: string, content: string) => {
    if (!user) {
      toast({
        title: 'Sign in required',
        description: 'Please sign in to reply',
        variant: 'destructive',
      });
      return false;
    }

    try {
      const authorName = user.email?.split('@')[0] || 'Anonymous';
      await communityApi.addReply(postId, content, authorName);

      // Update local state
      setPosts(prev => prev.map(p => 
        p.id === postId ? { ...p, replies_count: (p.replies_count || 0) + 1 } : p
      ));

      toast({
        title: 'Reply posted!',
        description: 'Your reply has been added',
      });
      return true;
    } catch (error) {
      console.error('Error creating reply:', error);
      toast({
        title: 'Error',
        description: 'Failed to post reply',
        variant: 'destructive',
      });
      return false;
    }
  };

  // Delete a reply (simplified)
  const deleteReply = async (replyId: string, postId: string) => {
    if (!user) return false;

    toast({
      title: 'Not available',
      description: 'Reply deletion not implemented in self-hosted version',
      variant: 'destructive',
    });
    return false;
  };

  // Subscribe to reply updates (no-op in self-hosted version without websockets)
  const subscribeToReplies = (postId: string, callback: (replies: PostReply[], eventType: string) => void) => {
    // No real-time in self-hosted version without websockets
    return () => {};
  };

  return {
    posts: postsWithLikes,
    loading,
    stats,
    createPost,
    toggleLike,
    deletePost,
    formatTimeAgo,
    fetchReplies,
    createReply,
    deleteReply,
    subscribeToReplies,
    refetch: fetchPosts,
  };
};
