import { useState } from 'react';
import { useAuth } from '@/hooks/useAuth';
import { toast } from 'sonner';
import api from '@/lib/api';

interface CommunityProduct {
  barcode: string;
  product_name: string;
  brand?: string;
  product_type: 'food' | 'cosmetic';
  categories?: string;
  ingredients?: string;
}

export const useCommunityProducts = () => {
  const { user } = useAuth();
  const [loading, setLoading] = useState(false);

  const addProduct = async (product: CommunityProduct): Promise<boolean> => {
    if (!user) {
      toast.error('Please sign in to contribute products');
      return false;
    }

    setLoading(true);
    try {
      await api.post('/community/products', {
        barcode: product.barcode,
        product_name: product.product_name,
        brand: product.brand || null,
        product_type: product.product_type,
        categories: product.categories || null,
        ingredients: product.ingredients || null,
      });

      toast.success('Thank you for contributing! This product is now in our database.');
      return true;
    } catch (err: any) {
      if (err.message?.includes('duplicate') || err.message?.includes('23505')) {
        toast.error('This product has already been added by someone');
      } else {
        console.error('Error adding product:', err);
        toast.error('Failed to add product');
      }
      return false;
    } finally {
      setLoading(false);
    }
  };

  return {
    addProduct,
    loading,
  };
};
