import { useState, useEffect } from 'react';
import { userApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

export interface DietaryPreferences {
  halal_preference: string;
  kosher_preference: string;
  organic_preference: string;
  avoid_xenoestrogens: boolean;
  max_risk_level: number;
}

const defaultPreferences: DietaryPreferences = {
  halal_preference: 'all',
  kosher_preference: 'all',
  organic_preference: 'all',
  avoid_xenoestrogens: false,
  max_risk_level: 5,
};

export const useDietaryPreferences = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [preferences, setPreferences] = useState<DietaryPreferences>(defaultPreferences);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (user) {
      fetchPreferences();
    } else {
      setPreferences(defaultPreferences);
    }
  }, [user]);

  const fetchPreferences = async () => {
    if (!user) return;
    
    setLoading(true);
    try {
      const data = await userApi.getDietaryPreferences();
      
      if (data) {
        setPreferences({
          halal_preference: data.halal_preference || 'all',
          kosher_preference: data.kosher_preference || 'all',
          organic_preference: data.organic_preference || 'all',
          avoid_xenoestrogens: data.avoid_xenoestrogens || false,
          max_risk_level: data.max_risk_level || 5,
        });
      }
    } catch (error: any) {
      // Ignore "not found" errors
      if (!error.message?.includes('not found')) {
        console.error('Error fetching preferences:', error);
      }
    }
    setLoading(false);
  };

  const updatePreferences = async (newPreferences: Partial<DietaryPreferences>) => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to save preferences",
        variant: "destructive",
      });
      return false;
    }

    const updatedPrefs = { ...preferences, ...newPreferences };
    
    try {
      await userApi.updateDietaryPreferences(updatedPrefs);
      setPreferences(updatedPrefs);
      toast({
        title: "Preferences saved",
        description: "Your dietary preferences have been updated",
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to save preferences",
        variant: "destructive",
      });
      return false;
    }
  };

  return {
    preferences,
    loading,
    updatePreferences,
    refetch: fetchPreferences,
  };
};
