import { useState, useEffect, useRef, useCallback } from 'react';
import { favoritesApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

interface Favorite {
  id: string;
  item_id: string;
  item_type: 'food' | 'cosmetic';
}

export const useFavorites = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [favorites, setFavorites] = useState<Favorite[]>([]);
  const [loading, setLoading] = useState(false);
  const userIdRef = useRef<string | null>(null);

  const fetchFavorites = useCallback(async () => {
    const userId = userIdRef.current;
    if (!userId) return;
    
    setLoading(true);
    try {
      const data = await favoritesApi.getAll();
      setFavorites(data as Favorite[]);
    } catch (error) {
      console.error('Error fetching favorites:', error);
    }
    setLoading(false);
  }, []);

  useEffect(() => {
    userIdRef.current = user?.id ?? null;
    
    if (user) {
      fetchFavorites();
    } else {
      setFavorites([]);
    }
  }, [user, fetchFavorites]);

  const addFavorite = async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to save favorites",
        variant: "destructive",
      });
      return false;
    }

    try {
      const data = await favoritesApi.add(itemId, itemType);
      
      // Optimistic update
      if (data && (data as any).id) {
        setFavorites(prev => [...prev, data as Favorite]);
      }

      toast({
        title: "Added to favorites",
        description: `${itemType === 'food' ? 'Food' : 'Cosmetic'} added to your favorites`,
      });
      return true;
    } catch (error: any) {
      if (error.message?.includes('already')) {
        toast({
          title: "Already favorited",
          description: "This item is already in your favorites",
        });
      } else {
        toast({
          title: "Error",
          description: "Failed to add favorite",
          variant: "destructive",
        });
      }
      return false;
    }
  };

  const removeFavorite = async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (!user) return false;

    // Optimistic update
    setFavorites(prev => prev.filter(f => !(f.item_id === itemId && f.item_type === itemType)));

    try {
      await favoritesApi.remove(itemType, itemId);
      toast({
        title: "Removed from favorites",
        description: "Item removed from your favorites",
      });
      return true;
    } catch (error) {
      // Revert on error
      fetchFavorites();
      toast({
        title: "Error",
        description: "Failed to remove favorite",
        variant: "destructive",
      });
      return false;
    }
  };

  const isFavorite = (itemId: string, itemType: 'food' | 'cosmetic') => {
    return favorites.some(f => f.item_id === itemId && f.item_type === itemType);
  };

  const toggleFavorite = async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (isFavorite(itemId, itemType)) {
      return removeFavorite(itemId, itemType);
    } else {
      return addFavorite(itemId, itemType);
    }
  };

  return {
    favorites,
    loading,
    addFavorite,
    removeFavorite,
    isFavorite,
    toggleFavorite,
    refetch: fetchFavorites,
  };
};
