import { useState, useEffect, useCallback } from 'react';
import { communityApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

export interface FollowStats {
  followersCount: number;
  followingCount: number;
}

export const useFollows = (targetUserId?: string) => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [isFollowing, setIsFollowing] = useState(false);
  const [stats, setStats] = useState<FollowStats>({ followersCount: 0, followingCount: 0 });
  const [followingList, setFollowingList] = useState<string[]>([]);
  const [loading, setLoading] = useState(true);

  const fetchFollowStatus = useCallback(async () => {
    if (!user) {
      setLoading(false);
      return;
    }

    try {
      const follows = await communityApi.getFollows();
      const followingIds = follows.map((f: any) => f.following_id);
      setFollowingList(followingIds);

      if (targetUserId) {
        setIsFollowing(followingIds.includes(targetUserId));
      }

      // Simplified stats - in full implementation, you'd have a backend endpoint
      setStats({
        followersCount: 0,
        followingCount: followingIds.length,
      });
    } catch (error) {
      console.error('Error fetching follow status:', error);
    }

    setLoading(false);
  }, [user, targetUserId]);

  const fetchFollowingList = useCallback(async () => {
    if (!user) {
      setFollowingList([]);
      return;
    }

    try {
      const follows = await communityApi.getFollows();
      setFollowingList(follows.map((f: any) => f.following_id));
    } catch (error) {
      console.error('Error fetching following list:', error);
    }
  }, [user]);

  useEffect(() => {
    fetchFollowStatus();
  }, [fetchFollowStatus]);

  const toggleFollow = async (userId: string) => {
    if (!user) {
      toast({
        title: 'Sign in required',
        description: 'Please sign in to follow users',
        variant: 'destructive',
      });
      return;
    }

    if (userId === user.id) {
      toast({
        title: 'Cannot follow yourself',
        description: 'You cannot follow your own profile',
        variant: 'destructive',
      });
      return;
    }

    const currentlyFollowing = followingList.includes(userId);

    try {
      if (currentlyFollowing) {
        await communityApi.unfollow(userId);
        setFollowingList(prev => prev.filter(id => id !== userId));
        if (userId === targetUserId) {
          setIsFollowing(false);
          setStats(prev => ({ ...prev, followersCount: Math.max(0, prev.followersCount - 1) }));
        }
        toast({
          title: 'Unfollowed',
          description: 'You are no longer following this user',
        });
      } else {
        await communityApi.follow(userId);
        setFollowingList(prev => [...prev, userId]);
        if (userId === targetUserId) {
          setIsFollowing(true);
          setStats(prev => ({ ...prev, followersCount: prev.followersCount + 1 }));
        }
        toast({
          title: 'Following!',
          description: 'You will see their activity in your feed',
        });
      }
    } catch (error) {
      console.error('Error toggling follow:', error);
      toast({
        title: 'Error',
        description: 'Failed to update follow status',
        variant: 'destructive',
      });
    }
  };

  const isUserFollowed = (userId: string) => followingList.includes(userId);

  return {
    isFollowing,
    stats,
    followingList,
    loading,
    toggleFollow,
    isUserFollowed,
    refetch: fetchFollowStatus,
  };
};
