import { useState, useEffect } from 'react';
import { userApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

export interface HealthProfile {
  // Health conditions
  has_autoimmune: boolean;
  has_allergies: boolean;
  has_digestive_issues: boolean;
  has_hormonal_imbalance: boolean;
  has_cardiovascular_issues: boolean;
  has_skin_sensitivity: boolean;
  // TOBIN thresholds
  toxin_threshold: number;
  oxidative_threshold: number;
  biological_threshold: number;
  inflammatory_threshold: number;
  nutrient_threshold: number;
  total_threshold: number;
  // Preferences
  show_alerts: boolean;
}

export interface TobinScore {
  total: number;
  toxins: number;
  oxidative: number;
  biological: number;
  inflammatory: number;
  nutrient: number;
}

const defaultProfile: HealthProfile = {
  has_autoimmune: false,
  has_allergies: false,
  has_digestive_issues: false,
  has_hormonal_imbalance: false,
  has_cardiovascular_issues: false,
  has_skin_sensitivity: false,
  toxin_threshold: 5,
  oxidative_threshold: 5,
  biological_threshold: 5,
  inflammatory_threshold: 5,
  nutrient_threshold: 5,
  total_threshold: 25,
  show_alerts: true,
};

export const useHealthProfile = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [profile, setProfile] = useState<HealthProfile>(defaultProfile);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (user) {
      fetchProfile();
    } else {
      setProfile(defaultProfile);
    }
  }, [user]);

  const fetchProfile = async () => {
    if (!user) return;
    
    setLoading(true);
    try {
      const data = await userApi.getHealthProfile();
      
      if (data) {
        setProfile({
          has_autoimmune: data.has_autoimmune || false,
          has_allergies: data.has_allergies || false,
          has_digestive_issues: data.has_digestive_issues || false,
          has_hormonal_imbalance: data.has_hormonal_imbalance || false,
          has_cardiovascular_issues: data.has_cardiovascular_issues || false,
          has_skin_sensitivity: data.has_skin_sensitivity || false,
          toxin_threshold: data.toxin_threshold || 5,
          oxidative_threshold: data.oxidative_threshold || 5,
          biological_threshold: data.biological_threshold || 5,
          inflammatory_threshold: data.inflammatory_threshold || 5,
          nutrient_threshold: data.nutrient_threshold || 5,
          total_threshold: data.total_threshold || 25,
          show_alerts: data.show_alerts ?? true,
        });
      }
    } catch (error: any) {
      // Ignore "not found" errors - just use defaults
      if (!error.message?.includes('not found')) {
        console.error('Error fetching health profile:', error);
      }
    }
    setLoading(false);
  };

  const updateProfile = async (newProfile: Partial<HealthProfile>) => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to save your health profile",
        variant: "destructive",
      });
      return false;
    }

    const updatedProfile = { ...profile, ...newProfile };
    
    try {
      await userApi.updateHealthProfile(updatedProfile);
      setProfile(updatedProfile);
      toast({
        title: "Profile saved",
        description: "Your health profile has been updated",
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to save health profile",
        variant: "destructive",
      });
      return false;
    }
  };

  const checkTobinAlerts = (tobinScore: TobinScore): string[] => {
    if (!profile.show_alerts) return [];
    
    const alerts: string[] = [];
    
    if (tobinScore.total > profile.total_threshold) {
      alerts.push(`Total TOBIN score (${tobinScore.total}) exceeds your threshold (${profile.total_threshold})`);
    }
    if (tobinScore.toxins > profile.toxin_threshold) {
      alerts.push(`Toxin level (${tobinScore.toxins}) exceeds your threshold (${profile.toxin_threshold})`);
    }
    if (tobinScore.oxidative > profile.oxidative_threshold) {
      alerts.push(`Oxidative stress (${tobinScore.oxidative}) exceeds your threshold (${profile.oxidative_threshold})`);
    }
    if (tobinScore.biological > profile.biological_threshold) {
      alerts.push(`Biological risk (${tobinScore.biological}) exceeds your threshold (${profile.biological_threshold})`);
    }
    if (tobinScore.inflammatory > profile.inflammatory_threshold) {
      alerts.push(`Inflammatory impact (${tobinScore.inflammatory}) exceeds your threshold (${profile.inflammatory_threshold})`);
    }
    if (tobinScore.nutrient > profile.nutrient_threshold) {
      alerts.push(`Nutrient deficit (${tobinScore.nutrient}) exceeds your threshold (${profile.nutrient_threshold})`);
    }
    
    return alerts;
  };

  const getConditionRecommendations = (): string[] => {
    const recommendations: string[] = [];
    
    if (profile.has_autoimmune) {
      recommendations.push("Consider lowering inflammatory and biological thresholds due to autoimmune condition");
    }
    if (profile.has_allergies) {
      recommendations.push("Watch for biological triggers that may worsen allergies");
    }
    if (profile.has_digestive_issues) {
      recommendations.push("Pay attention to inflammatory and biological scores for digestive health");
    }
    if (profile.has_hormonal_imbalance) {
      recommendations.push("Monitor toxin levels as they may affect hormonal balance");
    }
    if (profile.has_cardiovascular_issues) {
      recommendations.push("Focus on oxidative and inflammatory scores for heart health");
    }
    if (profile.has_skin_sensitivity) {
      recommendations.push("Watch toxin and inflammatory levels for skin health");
    }
    
    return recommendations;
  };

  return {
    profile,
    loading,
    updateProfile,
    checkTobinAlerts,
    getConditionRecommendations,
    refetch: fetchProfile,
  };
};
