import { useState, useEffect } from 'react';
import { mealPlansApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';
import { format, startOfWeek, endOfWeek, addDays } from 'date-fns';

interface MealPlan {
  id: string;
  recipe_id: string;
  planned_date: string;
  meal_slot: string;
  notes: string | null;
  reminder_time: string | null;
  created_at: string;
  recipe_title?: string;
  recipe_content?: string;
  recipe?: {
    id: string;
    title: string;
    meal_type: string;
    servings: number;
  };
}

export type MealSlot = 'breakfast' | 'lunch' | 'dinner' | 'snack';
export type ReminderTime = 'morning_before' | 'evening_before' | 'morning_of' | null;

export const MEAL_SLOTS: { value: MealSlot; label: string; emoji: string }[] = [
  { value: 'breakfast', label: 'Breakfast', emoji: '🌅' },
  { value: 'lunch', label: 'Lunch', emoji: '☀️' },
  { value: 'dinner', label: 'Dinner', emoji: '🌙' },
  { value: 'snack', label: 'Snack', emoji: '🍎' },
];

export const REMINDER_OPTIONS: { value: ReminderTime; label: string }[] = [
  { value: null, label: 'No reminder' },
  { value: 'morning_before', label: 'Morning before (9 AM)' },
  { value: 'evening_before', label: 'Evening before (6 PM)' },
  { value: 'morning_of', label: 'Morning of (7 AM)' },
];

export const useMealPlanner = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [mealPlans, setMealPlans] = useState<MealPlan[]>([]);
  const [loading, setLoading] = useState(false);
  const [currentWeekStart, setCurrentWeekStart] = useState(() => 
    startOfWeek(new Date(), { weekStartsOn: 1 })
  );

  useEffect(() => {
    if (user) {
      fetchMealPlans();
    } else {
      setMealPlans([]);
    }
  }, [user, currentWeekStart]);

  const fetchMealPlans = async () => {
    if (!user) return;
    
    setLoading(true);
    const weekEnd = endOfWeek(currentWeekStart, { weekStartsOn: 1 });
    
    try {
      const data = await mealPlansApi.getAll(
        format(currentWeekStart, 'yyyy-MM-dd'),
        format(weekEnd, 'yyyy-MM-dd')
      );
      
      // Map the response to include recipe info
      const plans = (data || []).map(plan => ({
        ...plan,
        recipe: plan.recipe_title ? {
          id: plan.recipe_id,
          title: plan.recipe_title,
          meal_type: 'any',
          servings: 4,
        } : undefined,
      }));
      
      setMealPlans(plans as MealPlan[]);
    } catch (error) {
      console.error('Error fetching meal plans:', error);
    }
    setLoading(false);
  };

  const addMealPlan = async (
    recipeId: string, 
    plannedDate: Date, 
    mealSlot: MealSlot,
    notes?: string,
    reminderTime?: ReminderTime
  ) => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to plan meals",
        variant: "destructive",
      });
      return false;
    }

    try {
      await mealPlansApi.create({
        recipe_id: recipeId,
        planned_date: format(plannedDate, 'yyyy-MM-dd'),
        meal_slot: mealSlot,
        notes: notes || null,
        reminder_time: reminderTime || null,
      });

      await fetchMealPlans();
      toast({
        title: "Meal scheduled!",
        description: `Added to ${format(plannedDate, 'EEEE, MMM d')}`,
      });
      return true;
    } catch (error) {
      console.error('Error adding meal plan:', error);
      toast({
        title: "Error",
        description: "Failed to add meal to plan",
        variant: "destructive",
      });
      return false;
    }
  };

  const moveMealPlan = async (
    id: string,
    newDate: Date,
    newSlot: MealSlot
  ) => {
    if (!user) return false;

    try {
      await mealPlansApi.update(id, {
        planned_date: format(newDate, 'yyyy-MM-dd'),
        meal_slot: newSlot,
      });

      await fetchMealPlans();
      toast({
        title: "Meal moved!",
        description: `Moved to ${format(newDate, 'EEEE, MMM d')}`,
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to move meal",
        variant: "destructive",
      });
      return false;
    }
  };

  const removeMealPlan = async (id: string) => {
    if (!user) return false;

    try {
      await mealPlansApi.delete(id);
      await fetchMealPlans();
      toast({
        title: "Removed",
        description: "Meal removed from plan",
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to remove meal from plan",
        variant: "destructive",
      });
      return false;
    }
  };

  const updateMealPlan = async (
    id: string, 
    updates: { planned_date?: Date; meal_slot?: MealSlot; notes?: string; reminder_time?: ReminderTime }
  ) => {
    if (!user) return false;

    const updateData: Record<string, string | null> = {};
    if (updates.planned_date) {
      updateData.planned_date = format(updates.planned_date, 'yyyy-MM-dd');
    }
    if (updates.meal_slot) {
      updateData.meal_slot = updates.meal_slot;
    }
    if (updates.notes !== undefined) {
      updateData.notes = updates.notes || null;
    }
    if (updates.reminder_time !== undefined) {
      updateData.reminder_time = updates.reminder_time;
    }

    try {
      await mealPlansApi.update(id, updateData);
      await fetchMealPlans();
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update meal plan",
        variant: "destructive",
      });
      return false;
    }
  };

  const setReminder = async (id: string, reminderTime: ReminderTime) => {
    return updateMealPlan(id, { reminder_time: reminderTime });
  };

  const getWeekDays = () => {
    return Array.from({ length: 7 }, (_, i) => addDays(currentWeekStart, i));
  };

  const getMealsForDay = (date: Date) => {
    const dateStr = format(date, 'yyyy-MM-dd');
    return mealPlans.filter(plan => plan.planned_date === dateStr);
  };

  const getMealsForDayAndSlot = (date: Date, slot: MealSlot) => {
    const dateStr = format(date, 'yyyy-MM-dd');
    return mealPlans.filter(
      plan => plan.planned_date === dateStr && plan.meal_slot === slot
    );
  };

  const goToNextWeek = () => {
    setCurrentWeekStart(addDays(currentWeekStart, 7));
  };

  const goToPreviousWeek = () => {
    setCurrentWeekStart(addDays(currentWeekStart, -7));
  };

  const goToCurrentWeek = () => {
    setCurrentWeekStart(startOfWeek(new Date(), { weekStartsOn: 1 }));
  };

  return {
    mealPlans,
    loading,
    currentWeekStart,
    addMealPlan,
    removeMealPlan,
    updateMealPlan,
    moveMealPlan,
    setReminder,
    getWeekDays,
    getMealsForDay,
    getMealsForDayAndSlot,
    goToNextWeek,
    goToPreviousWeek,
    goToCurrentWeek,
    refetch: fetchMealPlans,
  };
};
