import { useState, useCallback } from 'react';
import { useAuth } from './useAuth';
import api from '@/lib/api';

export interface MessageReaction {
  id: string;
  message_id: string;
  user_id: string;
  emoji: string;
  created_at: string;
}

export interface GroupedReaction {
  emoji: string;
  count: number;
  userReacted: boolean;
}

// API endpoints for message reactions
const reactionsApi = {
  getForMessages: (messageIds: string[]) => 
    api.post<MessageReaction[]>('/messaging/reactions', { message_ids: messageIds }),
  add: (messageId: string, emoji: string) => 
    api.post(`/messaging/reactions/${messageId}`, { emoji }),
  remove: (messageId: string, emoji: string) => 
    api.delete(`/messaging/reactions/${messageId}/${encodeURIComponent(emoji)}`),
};

export const useMessageReactions = (conversationId: string) => {
  const { user } = useAuth();
  const [reactions, setReactions] = useState<Map<string, MessageReaction[]>>(new Map());
  const [loading, setLoading] = useState(true);

  const fetchReactions = useCallback(async (messageIds: string[]) => {
    if (!messageIds.length) {
      setLoading(false);
      return;
    }

    try {
      const data = await reactionsApi.getForMessages(messageIds);

      const reactionMap = new Map<string, MessageReaction[]>();
      data?.forEach(reaction => {
        const existing = reactionMap.get(reaction.message_id) || [];
        reactionMap.set(reaction.message_id, [...existing, reaction]);
      });

      setReactions(reactionMap);
    } catch (error) {
      console.error('Error fetching reactions:', error);
    }
    setLoading(false);
  }, []);

  const addReaction = async (messageId: string, emoji: string): Promise<boolean> => {
    if (!user) return false;

    try {
      const newReaction = await reactionsApi.add(messageId, emoji);
      
      // Optimistic update
      setReactions(prev => {
        const updated = new Map(prev);
        const existing = updated.get(messageId) || [];
        updated.set(messageId, [...existing, newReaction as MessageReaction]);
        return updated;
      });
      
      return true;
    } catch (error: any) {
      // If duplicate, it's fine
      if (error.message?.includes('duplicate')) return true;
      console.error('Error adding reaction:', error);
      return false;
    }
  };

  const removeReaction = async (messageId: string, emoji: string): Promise<boolean> => {
    if (!user) return false;

    try {
      await reactionsApi.remove(messageId, emoji);
      
      // Optimistic update
      setReactions(prev => {
        const updated = new Map(prev);
        const existing = updated.get(messageId) || [];
        updated.set(messageId, existing.filter(r => !(r.user_id === user.id && r.emoji === emoji)));
        return updated;
      });
      
      return true;
    } catch (error) {
      console.error('Error removing reaction:', error);
      return false;
    }
  };

  const toggleReaction = async (messageId: string, emoji: string): Promise<boolean> => {
    if (!user) return false;

    const messageReactions = reactions.get(messageId) || [];
    const existingReaction = messageReactions.find(
      r => r.user_id === user.id && r.emoji === emoji
    );

    if (existingReaction) {
      return removeReaction(messageId, emoji);
    } else {
      return addReaction(messageId, emoji);
    }
  };

  const getGroupedReactions = (messageId: string): GroupedReaction[] => {
    const messageReactions = reactions.get(messageId) || [];
    const grouped = new Map<string, { count: number; userReacted: boolean }>();

    messageReactions.forEach(r => {
      const existing = grouped.get(r.emoji) || { count: 0, userReacted: false };
      grouped.set(r.emoji, {
        count: existing.count + 1,
        userReacted: existing.userReacted || r.user_id === user?.id,
      });
    });

    return Array.from(grouped.entries()).map(([emoji, data]) => ({
      emoji,
      count: data.count,
      userReacted: data.userReacted,
    }));
  };

  return {
    reactions,
    loading,
    fetchReactions,
    addReaction,
    removeReaction,
    toggleReaction,
    getGroupedReactions,
  };
};

// Common emoji reactions
export const REACTION_EMOJIS = ['❤️', '👍', '😂', '😮', '😢', '🔥'];
