import { useState, useEffect, useCallback } from 'react';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';
import { useNotifications } from './useNotifications';
import api from '@/lib/api';

export interface GroupParticipant {
  user_id: string;
  display_name: string | null;
  avatar_url: string | null;
  joined_at?: string;
}

export interface Conversation {
  id: string;
  participant_1: string;
  participant_2: string;
  last_message_at: string;
  created_at: string;
  archived_by_1?: boolean;
  archived_by_2?: boolean;
  is_group?: boolean;
  name?: string | null;
  created_by?: string | null;
  other_user?: {
    user_id: string;
    display_name: string | null;
    avatar_url: string | null;
  };
  participants?: GroupParticipant[];
  last_message?: string;
  unread_count?: number;
  isArchived?: boolean;
}

export interface Message {
  id: string;
  conversation_id: string;
  sender_id: string;
  content: string;
  is_read: boolean;
  created_at: string;
  attachment_url?: string | null;
  attachment_type?: string | null;
  attachment_name?: string | null;
  voice_url?: string | null;
  voice_duration?: number | null;
  sender_profile?: {
    display_name: string | null;
    avatar_url: string | null;
  };
}

export interface AttachmentUpload {
  file: File;
  url?: string;
}

// Messaging API endpoints
const messagingApi = {
  getConversations: () => api.get<Conversation[]>('/messaging/conversations'),
  getConversation: (id: string) => api.get<Conversation>(`/messaging/conversations/${id}`),
  createConversation: (otherUserId: string) => 
    api.post<Conversation>('/messaging/conversations', { other_user_id: otherUserId }),
  createGroup: (name: string, participantIds: string[]) => 
    api.post<Conversation>('/messaging/groups', { name, participant_ids: participantIds }),
  addParticipant: (conversationId: string, userId: string) => 
    api.post(`/messaging/groups/${conversationId}/participants`, { user_id: userId }),
  leaveGroup: (conversationId: string) => 
    api.delete(`/messaging/groups/${conversationId}/leave`),
  getMessages: (conversationId: string) => 
    api.get<Message[]>(`/messaging/conversations/${conversationId}/messages`),
  sendMessage: (conversationId: string, content: string, attachment?: any, voice?: any) => 
    api.post<Message>(`/messaging/conversations/${conversationId}/messages`, { content, attachment, voice }),
  markAsRead: (conversationId: string) => 
    api.post(`/messaging/conversations/${conversationId}/read`),
  archiveConversation: (conversationId: string) => 
    api.post(`/messaging/conversations/${conversationId}/archive`),
  unarchiveConversation: (conversationId: string) => 
    api.post(`/messaging/conversations/${conversationId}/unarchive`),
  searchMessages: (query: string) => 
    api.get<{ message: Message; conversation: Conversation }[]>(`/messaging/search?q=${encodeURIComponent(query)}`),
  uploadAttachment: (file: File, conversationId: string) => {
    const formData = new FormData();
    formData.append('file', file);
    formData.append('conversation_id', conversationId);
    return api.request<{ url: string }>('/messaging/upload', {
      method: 'POST',
      body: formData,
      headers: {}, // Let browser set Content-Type for FormData
    });
  },
  uploadVoice: (blob: Blob, conversationId: string, duration: number) => {
    const formData = new FormData();
    formData.append('file', blob, 'voice.webm');
    formData.append('conversation_id', conversationId);
    formData.append('duration', String(duration));
    return api.request<{ url: string; duration: number }>('/messaging/upload-voice', {
      method: 'POST',
      body: formData,
      headers: {},
    });
  },
};

export const useMessaging = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const { createNotification } = useNotifications();
  const [conversations, setConversations] = useState<Conversation[]>([]);
  const [loading, setLoading] = useState(true);
  const [totalUnread, setTotalUnread] = useState(0);

  const fetchConversations = useCallback(async () => {
    if (!user) {
      setConversations([]);
      setLoading(false);
      return;
    }

    try {
      const data = await messagingApi.getConversations();
      
      // Process conversations to add computed fields
      const enrichedConversations = data.map(conv => ({
        ...conv,
        isArchived: conv.participant_1 === user.id 
          ? conv.archived_by_1 
          : conv.archived_by_2,
      }));

      setConversations(enrichedConversations);
      setTotalUnread(enrichedConversations.reduce((sum, c) => sum + (c.unread_count || 0), 0));
    } catch (error) {
      console.error('Error fetching conversations:', error);
    }
    setLoading(false);
  }, [user]);

  useEffect(() => {
    if (!user) return;
    fetchConversations();
    
    // In a full implementation, you'd set up WebSocket for real-time updates
    // For now, poll every 30 seconds
    const interval = setInterval(fetchConversations, 30000);
    return () => clearInterval(interval);
  }, [user, fetchConversations]);

  const startConversation = async (otherUserId: string): Promise<string | null> => {
    if (!user) {
      toast({
        title: 'Sign in required',
        description: 'Please sign in to send messages',
        variant: 'destructive',
      });
      return null;
    }

    if (otherUserId === user.id) {
      toast({
        title: 'Cannot message yourself',
        description: 'You cannot start a conversation with yourself',
        variant: 'destructive',
      });
      return null;
    }

    try {
      const conv = await messagingApi.createConversation(otherUserId);
      await fetchConversations();
      return conv.id;
    } catch (error) {
      console.error('Error creating conversation:', error);
      toast({
        title: 'Error',
        description: 'Failed to start conversation',
        variant: 'destructive',
      });
      return null;
    }
  };

  const createGroupChat = async (name: string, participantIds: string[]): Promise<string | null> => {
    if (!user) {
      toast({
        title: 'Sign in required',
        description: 'Please sign in to create a group',
        variant: 'destructive',
      });
      return null;
    }

    if (participantIds.length < 2) {
      toast({
        title: 'Not enough participants',
        description: 'Add at least 2 other people to create a group',
        variant: 'destructive',
      });
      return null;
    }

    try {
      const conv = await messagingApi.createGroup(name.trim() || 'Group Chat', participantIds);
      await fetchConversations();
      toast({
        title: 'Group created',
        description: `${name || 'Group Chat'} has been created`,
      });
      return conv.id;
    } catch (error) {
      console.error('Error creating group:', error);
      toast({
        title: 'Error',
        description: 'Failed to create group',
        variant: 'destructive',
      });
      return null;
    }
  };

  const addGroupParticipant = async (conversationId: string, userId: string): Promise<boolean> => {
    if (!user) return false;

    try {
      await messagingApi.addParticipant(conversationId, userId);
      await fetchConversations();
      return true;
    } catch (error) {
      console.error('Error adding participant:', error);
      toast({
        title: 'Error',
        description: 'Failed to add participant',
        variant: 'destructive',
      });
      return false;
    }
  };

  const leaveGroup = async (conversationId: string): Promise<boolean> => {
    if (!user) return false;

    try {
      await messagingApi.leaveGroup(conversationId);
      await fetchConversations();
      toast({
        title: 'Left group',
      });
      return true;
    } catch (error) {
      console.error('Error leaving group:', error);
      toast({
        title: 'Error',
        description: 'Failed to leave group',
        variant: 'destructive',
      });
      return false;
    }
  };

  const uploadAttachment = async (file: File, conversationId: string): Promise<string | null> => {
    if (!user) return null;

    try {
      const result = await messagingApi.uploadAttachment(file, conversationId);
      return result.url;
    } catch (error) {
      console.error('Error uploading attachment:', error);
      toast({
        title: 'Upload failed',
        description: 'Failed to upload attachment',
        variant: 'destructive',
      });
      return null;
    }
  };

  const uploadVoiceMessage = async (blob: Blob, conversationId: string, duration: number): Promise<{ url: string; duration: number } | null> => {
    if (!user) return null;

    try {
      return await messagingApi.uploadVoice(blob, conversationId, duration);
    } catch (error) {
      console.error('Error uploading voice message:', error);
      toast({
        title: 'Upload failed',
        description: 'Failed to upload voice message',
        variant: 'destructive',
      });
      return null;
    }
  };

  const sendMessage = async (
    conversationId: string, 
    content: string,
    attachment?: { url: string; type: string; name: string },
    voice?: { url: string; duration: number }
  ): Promise<boolean> => {
    if (!user) return false;

    try {
      await messagingApi.sendMessage(conversationId, content, attachment, voice);
      return true;
    } catch (error) {
      console.error('Error sending message:', error);
      toast({
        title: 'Error',
        description: 'Failed to send message',
        variant: 'destructive',
      });
      return false;
    }
  };

  const getMessages = async (conversationId: string): Promise<Message[]> => {
    try {
      return await messagingApi.getMessages(conversationId);
    } catch (error) {
      console.error('Error fetching messages:', error);
      return [];
    }
  };

  const markAsRead = async (conversationId: string) => {
    if (!user) return;

    try {
      await messagingApi.markAsRead(conversationId);
      
      // Update local state
      setConversations(prev => 
        prev.map(c => c.id === conversationId ? { ...c, unread_count: 0 } : c)
      );
      setTotalUnread(prev => {
        const conv = conversations.find(c => c.id === conversationId);
        return Math.max(0, prev - (conv?.unread_count || 0));
      });
    } catch (error) {
      console.error('Error marking as read:', error);
    }
  };

  const searchMessages = async (query: string): Promise<{
    message: Message;
    conversation: Conversation;
  }[]> => {
    if (!user || !query.trim()) return [];

    try {
      return await messagingApi.searchMessages(query);
    } catch (error) {
      console.error('Error searching messages:', error);
      return [];
    }
  };

  const archiveConversation = async (conversationId: string): Promise<boolean> => {
    if (!user) return false;

    try {
      await messagingApi.archiveConversation(conversationId);
      
      // Update local state
      setConversations(prev => 
        prev.map(c => c.id === conversationId ? { ...c, isArchived: true } : c)
      );

      toast({
        title: 'Conversation archived',
        description: 'You can find it in your archived chats',
      });

      return true;
    } catch (error) {
      console.error('Error archiving conversation:', error);
      toast({
        title: 'Error',
        description: 'Failed to archive conversation',
        variant: 'destructive',
      });
      return false;
    }
  };

  const unarchiveConversation = async (conversationId: string): Promise<boolean> => {
    if (!user) return false;

    try {
      await messagingApi.unarchiveConversation(conversationId);
      
      setConversations(prev => 
        prev.map(c => c.id === conversationId ? { ...c, isArchived: false } : c)
      );

      toast({
        title: 'Conversation unarchived',
      });

      return true;
    } catch (error) {
      console.error('Error unarchiving conversation:', error);
      return false;
    }
  };

  // Computed lists
  const activeConversations = conversations.filter(c => !c.isArchived);
  const archivedConversations = conversations.filter(c => c.isArchived);
  const groupConversations = conversations.filter(c => c.is_group);

  return {
    conversations,
    activeConversations,
    archivedConversations,
    groupConversations,
    loading,
    totalUnread,
    startConversation,
    createGroupChat,
    addGroupParticipant,
    leaveGroup,
    sendMessage,
    uploadAttachment,
    uploadVoiceMessage,
    getMessages,
    markAsRead,
    searchMessages,
    archiveConversation,
    unarchiveConversation,
    refetch: fetchConversations,
  };
};
