import { useState } from 'react';
import { aiApi } from '@/lib/apiServices';

export interface PhytoCompound {
  name: string;
  source: string;
  potency: 'weak' | 'moderate' | 'strong';
}

export interface PhytoAnalysisResult {
  detected: boolean;
  overallRisk: 'none' | 'low' | 'moderate' | 'high';
  compounds: PhytoCompound[];
  summary: string;
  recommendation: string;
}

export const usePhytoestrogenAnalysis = () => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [result, setResult] = useState<PhytoAnalysisResult | null>(null);

  const analyzeIngredients = async (
    ingredients: string,
    productName?: string,
    useAI = false
  ): Promise<PhytoAnalysisResult | null> => {
    if (!ingredients) {
      setError('No ingredients to analyze');
      return null;
    }

    setLoading(true);
    setError(null);

    try {
      const { analysis } = await aiApi.analyzePhytoestrogens(ingredients);
      
      // Parse the AI response
      let parsed: any;
      try {
        parsed = typeof analysis === 'string' ? JSON.parse(analysis) : analysis;
      } catch {
        // If parsing fails, create a basic result
        parsed = {
          detected: false,
          overallRisk: 'none',
          compounds: [],
          summary: analysis,
          recommendation: 'Unable to parse detailed analysis.',
        };
      }

      const result: PhytoAnalysisResult = {
        detected: parsed.detected || parsed.compounds?.length > 0 || false,
        overallRisk: parsed.overallRisk || parsed.overall_risk || 'none',
        compounds: (parsed.compounds || []).map((c: any) => ({
          name: c.name,
          source: c.source || c.food_source || 'Unknown',
          potency: c.potency || c.level || 'moderate',
        })),
        summary: parsed.summary || parsed.health_impact || '',
        recommendation: parsed.recommendation || parsed.recommendations?.join('. ') || '',
      };

      setResult(result);
      return result;
    } catch (err) {
      console.error('Phytoestrogen analysis error:', err);
      setError(err instanceof Error ? err.message : 'Unknown error');
      return null;
    } finally {
      setLoading(false);
    }
  };

  const reset = () => {
    setResult(null);
    setError(null);
  };

  return {
    analyzeIngredients,
    loading,
    error,
    result,
    reset,
  };
};
