import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { phytoestrogenApi } from "@/lib/apiServices";
import { useAuth } from "./useAuth";
import { toast } from "sonner";

export interface PhytoestrogenGoals {
  id: string;
  user_id: string;
  daily_limit: number;
  weekly_limit: number;
  alert_at_percentage: number;
  show_alerts: boolean;
  created_at: string;
  updated_at: string;
}

const DEFAULT_GOALS: Omit<PhytoestrogenGoals, "id" | "user_id" | "created_at" | "updated_at"> = {
  daily_limit: 10,
  weekly_limit: 50,
  alert_at_percentage: 80,
  show_alerts: true,
};

export function usePhytoestrogenGoals() {
  const { user } = useAuth();
  const queryClient = useQueryClient();

  // Fetch goals
  const { data: goals, isLoading } = useQuery({
    queryKey: ["phytoestrogen-goals", user?.id],
    queryFn: async () => {
      if (!user?.id) return null;
      return (await phytoestrogenApi.getGoals()) as PhytoestrogenGoals | null;
    },
    enabled: !!user?.id,
  });

  // Create or update goals
  const saveGoalsMutation = useMutation({
    mutationFn: async (newGoals: Partial<Omit<PhytoestrogenGoals, "id" | "user_id" | "created_at" | "updated_at">>) => {
      if (!user?.id) throw new Error("Not authenticated");
      return phytoestrogenApi.saveGoals({
        daily_limit: newGoals.daily_limit,
        weekly_limit: newGoals.weekly_limit,
        show_alerts: newGoals.show_alerts,
        alert_at_percentage: newGoals.alert_at_percentage,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["phytoestrogen-goals"] });
      toast.success("Goals saved successfully");
    },
    onError: (error) => {
      console.error("Error saving goals:", error);
      toast.error("Failed to save goals");
    },
  });

  // Get effective goals (with defaults)
  const effectiveGoals = goals || {
    ...DEFAULT_GOALS,
    id: "",
    user_id: user?.id || "",
    created_at: "",
    updated_at: "",
  };

  // Check progress against goals
  const checkProgress = (dailyScore: number, weeklyScore: number) => {
    const dailyPercentage = (dailyScore / effectiveGoals.daily_limit) * 100;
    const weeklyPercentage = (weeklyScore / effectiveGoals.weekly_limit) * 100;

    return {
      daily: {
        current: dailyScore,
        limit: effectiveGoals.daily_limit,
        percentage: Math.min(dailyPercentage, 100),
        isWarning: dailyPercentage >= effectiveGoals.alert_at_percentage,
        isExceeded: dailyPercentage >= 100,
      },
      weekly: {
        current: weeklyScore,
        limit: effectiveGoals.weekly_limit,
        percentage: Math.min(weeklyPercentage, 100),
        isWarning: weeklyPercentage >= effectiveGoals.alert_at_percentage,
        isExceeded: weeklyPercentage >= 100,
      },
      shouldAlert: effectiveGoals.show_alerts && (
        dailyPercentage >= effectiveGoals.alert_at_percentage ||
        weeklyPercentage >= effectiveGoals.alert_at_percentage
      ),
    };
  };

  return {
    goals: effectiveGoals,
    isLoading,
    hasCustomGoals: !!goals,
    saveGoals: saveGoalsMutation.mutate,
    isSaving: saveGoalsMutation.isPending,
    checkProgress,
  };
}
