import { useState, useEffect, useCallback } from "react";
import { useQuery, useMutation, useQueryClient } from "@tanstack/react-query";
import { phytoestrogenApi } from "@/lib/apiServices";
import { useAuth } from "./useAuth";
import { toast } from "sonner";
import { PhytoestrogensData } from "@/data/chemicalDatabase";

export interface PhytoestrogenExposureEntry {
  id: string;
  user_id: string;
  phytoestrogen_id: string;
  phytoestrogen_name: string;
  potency: "weak" | "moderate" | "strong";
  food_source: string;
  servings: number;
  logged_at: string;
  notes: string | null;
  created_at: string;
}

export interface DailyExposure {
  date: string;
  total: number;
  strong: number;
  moderate: number;
  weak: number;
  entries: PhytoestrogenExposureEntry[];
}

// Weight multipliers for calculating exposure score
const POTENCY_WEIGHTS = {
  strong: 3,
  moderate: 2,
  weak: 1,
};

export function usePhytoestrogenTracking() {
  const { user } = useAuth();
  const queryClient = useQueryClient();

  // Fetch exposure logs
  const { data: exposureLogs, isLoading } = useQuery({
    queryKey: ["phytoestrogen-exposure", user?.id],
    queryFn: async () => {
      if (!user?.id) return [];
      return (await phytoestrogenApi.getExposure()) as PhytoestrogenExposureEntry[];
    },
    enabled: !!user?.id,
  });

  // Log new exposure
  const logExposureMutation = useMutation({
    mutationFn: async ({
      phytoestrogen,
      foodSource,
      servings = 1,
      notes,
      loggedAt,
    }: {
      phytoestrogen: PhytoestrogensData;
      foodSource: string;
      servings?: number;
      notes?: string;
      loggedAt?: Date;
    }) => {
      if (!user?.id) throw new Error("Not authenticated");
      return phytoestrogenApi.logExposure({
        phytoestrogen_id: phytoestrogen.id,
        phytoestrogen_name: phytoestrogen.name,
        potency: phytoestrogen.potency,
        food_source: foodSource,
        servings,
        notes: notes || undefined,
        logged_at: loggedAt?.toISOString() || new Date().toISOString(),
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["phytoestrogen-exposure"] });
      toast.success("Exposure logged successfully");
    },
    onError: (error) => {
      console.error("Error logging exposure:", error);
      toast.error("Failed to log exposure");
    },
  });

  // Delete exposure entry
  const deleteExposureMutation = useMutation({
    mutationFn: async (entryId: string) => {
      if (!user?.id) throw new Error("Not authenticated");
      await phytoestrogenApi.deleteExposure(entryId);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["phytoestrogen-exposure"] });
      toast.success("Entry deleted");
    },
    onError: (error) => {
      console.error("Error deleting entry:", error);
      toast.error("Failed to delete entry");
    },
  });

  // Calculate daily exposure summaries
  const getDailyExposures = useCallback((days: number = 30): DailyExposure[] => {
    if (!exposureLogs) return [];

    const dailyMap = new Map<string, DailyExposure>();
    const now = new Date();

    // Initialize days
    for (let i = 0; i < days; i++) {
      const date = new Date(now);
      date.setDate(date.getDate() - i);
      const dateStr = date.toISOString().split("T")[0];
      dailyMap.set(dateStr, {
        date: dateStr,
        total: 0,
        strong: 0,
        moderate: 0,
        weak: 0,
        entries: [],
      });
    }

    // Populate with actual data
    exposureLogs.forEach((entry) => {
      const dateStr = new Date(entry.logged_at).toISOString().split("T")[0];
      const daily = dailyMap.get(dateStr);
      if (daily) {
        const weight = POTENCY_WEIGHTS[entry.potency] * entry.servings;
        daily.total += weight;
        daily[entry.potency] += entry.servings;
        daily.entries.push(entry);
      }
    });

    return Array.from(dailyMap.values()).sort((a, b) => 
      new Date(a.date).getTime() - new Date(b.date).getTime()
    );
  }, [exposureLogs]);

  // Get weekly summary
  const getWeeklySummary = useCallback(() => {
    const dailyExposures = getDailyExposures(7);
    return {
      totalScore: dailyExposures.reduce((sum, d) => sum + d.total, 0),
      strongCount: dailyExposures.reduce((sum, d) => sum + d.strong, 0),
      moderateCount: dailyExposures.reduce((sum, d) => sum + d.moderate, 0),
      weakCount: dailyExposures.reduce((sum, d) => sum + d.weak, 0),
      totalEntries: dailyExposures.reduce((sum, d) => sum + d.entries.length, 0),
      averageDaily: dailyExposures.reduce((sum, d) => sum + d.total, 0) / 7,
    };
  }, [getDailyExposures]);

  // Get today's exposure
  const getTodaysExposure = useCallback((): DailyExposure => {
    const today = new Date().toISOString().split("T")[0];
    const dailyExposures = getDailyExposures(1);
    return dailyExposures.find((d) => d.date === today) || {
      date: today,
      total: 0,
      strong: 0,
      moderate: 0,
      weak: 0,
      entries: [],
    };
  }, [getDailyExposures]);

  // Get top exposure sources
  const getTopExposureSources = useCallback((limit: number = 5) => {
    if (!exposureLogs) return [];

    const sourceMap = new Map<string, { name: string; count: number; score: number }>();
    
    exposureLogs.forEach((entry) => {
      const key = entry.phytoestrogen_id;
      const existing = sourceMap.get(key);
      const weight = POTENCY_WEIGHTS[entry.potency] * entry.servings;
      
      if (existing) {
        existing.count += entry.servings;
        existing.score += weight;
      } else {
        sourceMap.set(key, {
          name: entry.phytoestrogen_name,
          count: entry.servings,
          score: weight,
        });
      }
    });

    return Array.from(sourceMap.values())
      .sort((a, b) => b.score - a.score)
      .slice(0, limit);
  }, [exposureLogs]);

  return {
    exposureLogs,
    isLoading,
    logExposure: logExposureMutation.mutate,
    isLogging: logExposureMutation.isPending,
    deleteExposure: deleteExposureMutation.mutate,
    isDeleting: deleteExposureMutation.isPending,
    getDailyExposures,
    getWeeklySummary,
    getTodaysExposure,
    getTopExposureSources,
  };
}
