import { useState, useCallback } from "react";
import { API_URL } from "@/lib/api";

interface RecipeRequest {
  mealType?: string;
  servings?: number;
  dietaryRestrictions?: string;
  preferences?: string;
}

export function useRecipeGenerator() {
  const [content, setContent] = useState("");
  const [isGenerating, setIsGenerating] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const generateRecipes = useCallback(async (request: RecipeRequest) => {
    setIsGenerating(true);
    setContent("");
    setError(null);

    try {
      const token = localStorage.getItem('accessToken');
      
      const response = await fetch(`${API_URL}/ai/generate-recipes`, {
        method: "POST",
        headers: {
          "Content-Type": "application/json",
          ...(token ? { Authorization: `Bearer ${token}` } : {}),
        },
        body: JSON.stringify(request),
      });

      if (!response.ok) {
        const errorData = await response.json();
        throw new Error(errorData.error || "Failed to generate recipes");
      }

      const data = await response.json();
      
      // Handle both streaming and non-streaming responses
      if (data.recipe) {
        const recipeContent = typeof data.recipe === 'string' ? data.recipe : JSON.stringify(data.recipe, null, 2);
        setContent(recipeContent);
      } else if (response.body) {
        // Handle streaming response
        const reader = response.body.getReader();
        const decoder = new TextDecoder();
        let buffer = "";
        let fullContent = "";

        while (true) {
          const { done, value } = await reader.read();
          if (done) break;

          buffer += decoder.decode(value, { stream: true });

          let newlineIndex: number;
          while ((newlineIndex = buffer.indexOf("\n")) !== -1) {
            let line = buffer.slice(0, newlineIndex);
            buffer = buffer.slice(newlineIndex + 1);

            if (line.endsWith("\r")) line = line.slice(0, -1);
            if (line.startsWith(":") || line.trim() === "") continue;
            if (!line.startsWith("data: ")) continue;

            const jsonStr = line.slice(6).trim();
            if (jsonStr === "[DONE]") break;

            try {
              const parsed = JSON.parse(jsonStr);
              const delta = parsed.choices?.[0]?.delta?.content;
              if (delta) {
                fullContent += delta;
                setContent(fullContent);
              }
            } catch {
              buffer = line + "\n" + buffer;
              break;
            }
          }
        }
      }
    } catch (err) {
      const message = err instanceof Error ? err.message : "An error occurred";
      setError(message);
    } finally {
      setIsGenerating(false);
    }
  }, []);

  const reset = useCallback(() => {
    setContent("");
    setError(null);
  }, []);

  return {
    content,
    isGenerating,
    error,
    generateRecipes,
    reset,
  };
}
