import { useState, useEffect } from 'react';
import { recipesApi, aiApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

interface NutritionalInfo {
  calories: number;
  protein: number;
  carbs: number;
  fat: number;
  fiber: number;
  sugar: number;
  sodium: number;
  highlights: string[];
  warnings: string[];
}

interface SavedRecipe {
  id: string;
  title: string;
  content: string;
  meal_type: string;
  servings: number;
  created_at: string;
  share_token: string | null;
  is_shared: boolean;
  nutritional_info: NutritionalInfo | null;
}

export type { SavedRecipe, NutritionalInfo };

export const useSavedRecipes = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [recipes, setRecipes] = useState<SavedRecipe[]>([]);
  const [loading, setLoading] = useState(false);

  useEffect(() => {
    if (user) {
      fetchRecipes();
    } else {
      setRecipes([]);
    }
  }, [user]);

  const fetchRecipes = async () => {
    if (!user) return;
    
    setLoading(true);
    try {
      const data = await recipesApi.getAll();
      setRecipes((data || []).map(r => ({
        ...r,
        nutritional_info: r.nutritional_info as NutritionalInfo | null,
      })));
    } catch (error) {
      console.error('Error fetching saved recipes:', error);
    }
    setLoading(false);
  };

  const saveRecipe = async (title: string, content: string, mealType: string, servings: number) => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to save recipes to your cookbook",
        variant: "destructive",
      });
      return false;
    }

    try {
      await recipesApi.create({
        title,
        content,
        meal_type: mealType,
        servings,
      });

      await fetchRecipes();
      toast({
        title: "Recipe saved!",
        description: "Added to your personal cookbook",
      });
      return true;
    } catch (error) {
      console.error('Error saving recipe:', error);
      toast({
        title: "Error",
        description: "Failed to save recipe",
        variant: "destructive",
      });
      return false;
    }
  };

  const deleteRecipe = async (id: string) => {
    if (!user) return false;

    try {
      await recipesApi.delete(id);
      await fetchRecipes();
      toast({
        title: "Recipe deleted",
        description: "Removed from your cookbook",
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to delete recipe",
        variant: "destructive",
      });
      return false;
    }
  };

  const generateShareLink = async (id: string): Promise<string | null> => {
    if (!user) return null;

    try {
      await recipesApi.update(id, { is_shared: true });
      await fetchRecipes();
      
      const recipe = recipes.find(r => r.id === id);
      if (recipe?.share_token) {
        return `${window.location.origin}/shared-recipe/${recipe.share_token}`;
      }
      
      // Refetch to get the new share token
      const updated = await recipesApi.get(id);
      if (updated?.share_token) {
        return `${window.location.origin}/shared-recipe/${updated.share_token}`;
      }
      
      return null;
    } catch (error) {
      console.error('Error generating share link:', error);
      toast({
        title: "Error",
        description: "Failed to generate share link",
        variant: "destructive",
      });
      return null;
    }
  };

  const disableSharing = async (id: string) => {
    if (!user) return false;

    try {
      await recipesApi.update(id, { is_shared: false });
      await fetchRecipes();
      toast({
        title: "Sharing disabled",
        description: "This recipe is now private",
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to disable sharing",
        variant: "destructive",
      });
      return false;
    }
  };

  const analyzeNutrition = async (id: string): Promise<NutritionalInfo | null> => {
    if (!user) return null;

    const recipe = recipes.find(r => r.id === id);
    if (!recipe) return null;

    try {
      const { nutrition } = await aiApi.analyzeNutrition(recipe.content, `${recipe.servings} servings`);
      const parsed = typeof nutrition === 'string' ? JSON.parse(nutrition) : nutrition;

      // Update recipe with nutritional info
      await recipesApi.update(id, { nutritional_info: parsed });
      await fetchRecipes();
      
      return parsed as NutritionalInfo;
    } catch (error) {
      console.error('Error analyzing nutrition:', error);
      toast({
        title: "Error",
        description: "Failed to analyze nutrition",
        variant: "destructive",
      });
      return null;
    }
  };

  return {
    recipes,
    loading,
    saveRecipe,
    deleteRecipe,
    generateShareLink,
    disableSharing,
    analyzeNutrition,
    refetch: fetchRecipes,
  };
};
