import { useState, useEffect, useRef, useCallback } from 'react';
import { scanHistoryApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

// Product details that can be stored in scan history
export interface ScanProductDetails {
  // Open Food Facts / API data
  brand?: string;
  categories?: string;
  ingredients?: string;
  imageUrl?: string | null;
  nutrition?: {
    calories: number | null;
    fat: number | null;
    carbs: number | null;
    protein: number | null;
    fiber?: number | null;
    sugar?: number | null;
    sodium?: number | null;
  };
  nutriscoreGrade?: string;
  novaGroup?: number;
  
  // AI Analysis data
  aiAnalysis?: {
    safetyLevel?: 'safe' | 'low_concern' | 'moderate_concern' | 'high_concern';
    summary?: string;
    concerns?: string[];
    ingredients?: string[];
    saferAlternatives?: string[];
    sources?: string[];
    tobinScore?: number | null;
    glycogenicIndex?: number | null;
    lipogenicIndex?: number | null;
    nutriscoreGrade?: string | null;
    novaGroup?: number | null;
    halal?: string | null;
    kosher?: string | null;
    organic?: string | null;
    certificationNotes?: string | null;
  };
  // Dietary from AI (also at top level for filtering)
  dietaryClassification?: {
    halal?: string;
    kosher?: string;
    organic?: string;
    certificationNotes?: string;
  };
  
  // Product source info
  source?: 'local' | 'openfoodfacts' | 'ai' | 'community';
  apiSourceString?: string;
  isAiGenerated?: boolean;
}

export interface ScanHistoryItem {
  id: string;
  barcode: string;
  product_name: string;
  product_type: 'food' | 'cosmetic';
  product_id: string;
  scanned_at: string;
  image_url: string | null;
  product_details: ScanProductDetails | null;
}

export const useScanHistory = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [history, setHistory] = useState<ScanHistoryItem[]>([]);
  const [loading, setLoading] = useState(true);
  const userIdRef = useRef<string | null>(null);

  const fetchHistory = useCallback(async () => {
    const userId = userIdRef.current;
    if (!userId) {
      setHistory([]);
      setLoading(false);
      return;
    }

    try {
      const data = await scanHistoryApi.getAll(50);
      
      // Parse product_details from JSON
      const parsedData = (data || []).map(item => ({
        ...item,
        product_type: item.product_type as 'food' | 'cosmetic',
        product_details: item.product_details as ScanProductDetails | null
      }));
      
      setHistory(parsedData);
    } catch (error) {
      console.error('Error fetching scan history:', error);
    } finally {
      setLoading(false);
    }
  }, []);

  useEffect(() => {
    userIdRef.current = user?.id ?? null;
    
    if (user) {
      fetchHistory();
    } else {
      setHistory([]);
      setLoading(false);
    }
  }, [user, fetchHistory]);

  const addToHistory = useCallback(async (
    barcode: string,
    productName: string,
    productType: 'food' | 'cosmetic',
    productId: string,
    imageUrl?: string | null,
    productDetails?: ScanProductDetails
  ) => {
    if (!user) return;

    try {
      await scanHistoryApi.add({
        barcode,
        product_name: productName,
        product_type: productType,
        product_id: productId,
        image_url: imageUrl || null,
        product_details: productDetails || null,
      });
      await fetchHistory();
    } catch (error) {
      console.error('Error adding to scan history:', error);
    }
  }, [user, fetchHistory]);

  const deleteFromHistory = useCallback(async (id: string): Promise<boolean> => {
    if (!user) return false;

    try {
      await scanHistoryApi.remove(id);
      setHistory(prev => prev.filter(item => item.id !== id));
      toast({
        title: "Removed from history",
        description: "The scan has been removed from your history.",
      });
      return true;
    } catch (error) {
      console.error('Error deleting from history:', error);
      toast({
        title: "Error",
        description: "Could not remove scan from history.",
        variant: "destructive",
      });
      return false;
    }
  }, [user, toast]);

  const clearHistory = useCallback(async () => {
    if (!user) return;

    try {
      await scanHistoryApi.clearAll();
      setHistory([]);
      toast({
        title: "History cleared",
        description: "Your scan history has been cleared.",
      });
    } catch (error) {
      console.error('Error clearing history:', error);
      toast({
        title: "Error",
        description: "Could not clear scan history.",
        variant: "destructive",
      });
    }
  }, [user, toast]);

  return {
    history,
    loading,
    addToHistory,
    deleteFromHistory,
    clearHistory,
    refetch: fetchHistory,
  };
};
