import { useState, useEffect, useRef, useCallback } from 'react';
import { shoppingListApi } from '@/lib/apiServices';
import { useAuth } from './useAuth';
import { useToast } from './use-toast';

interface ShoppingListItem {
  id: string;
  item_id: string;
  item_type: 'food' | 'cosmetic';
  quantity: number;
  notes: string | null;
  is_purchased: boolean;
  created_at: string;
}

export const useShoppingList = () => {
  const { user } = useAuth();
  const { toast } = useToast();
  const [items, setItems] = useState<ShoppingListItem[]>([]);
  const [loading, setLoading] = useState(false);
  const userIdRef = useRef<string | null>(null);

  const fetchItems = useCallback(async () => {
    const userId = userIdRef.current;
    if (!userId) return;
    
    setLoading(true);
    try {
      const data = await shoppingListApi.getAll();
      setItems(data as ShoppingListItem[]);
    } catch (error) {
      console.error('Error fetching shopping list:', error);
    }
    setLoading(false);
  }, []);

  useEffect(() => {
    userIdRef.current = user?.id ?? null;
    
    if (user) {
      fetchItems();
    } else {
      setItems([]);
    }
  }, [user, fetchItems]);

  const addItem = async (itemId: string, itemType: 'food' | 'cosmetic', notes?: string) => {
    if (!user) {
      toast({
        title: "Sign in required",
        description: "Please sign in to add items to your shopping list",
        variant: "destructive",
      });
      return false;
    }

    try {
      const data = await shoppingListApi.add(itemId, itemType, 1, notes);
      
      // Optimistic update
      if (data && (data as any).id) {
        setItems(prev => [data as ShoppingListItem, ...prev]);
      }

      toast({
        title: "Added to shopping list",
        description: "Item added to your shopping list",
      });
      return true;
    } catch (error: any) {
      if (error.message?.includes('already')) {
        toast({
          title: "Already in list",
          description: "This item is already in your shopping list",
        });
      } else {
        toast({
          title: "Error",
          description: "Failed to add item to shopping list",
          variant: "destructive",
        });
      }
      return false;
    }
  };

  const removeItem = async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (!user) return false;

    // Optimistic update
    setItems(prev => prev.filter(i => !(i.item_id === itemId && i.item_type === itemType)));

    try {
      // Find the item to get its ID
      const item = items.find(i => i.item_id === itemId && i.item_type === itemType);
      if (item) {
        await shoppingListApi.remove(item.id);
      }
      
      toast({
        title: "Removed",
        description: "Item removed from shopping list",
      });
      return true;
    } catch (error) {
      // Revert on error
      fetchItems();
      toast({
        title: "Error",
        description: "Failed to remove item",
        variant: "destructive",
      });
      return false;
    }
  };

  const togglePurchased = async (id: string, isPurchased: boolean) => {
    if (!user) return false;

    try {
      await shoppingListApi.update(id, { is_purchased: isPurchased });
      setItems(prev => prev.map(i => i.id === id ? { ...i, is_purchased: isPurchased } : i));
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update item",
        variant: "destructive",
      });
      return false;
    }
  };

  const updateQuantity = async (id: string, quantity: number) => {
    if (!user || quantity < 1) return false;

    try {
      await shoppingListApi.update(id, { quantity });
      setItems(prev => prev.map(i => i.id === id ? { ...i, quantity } : i));
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to update quantity",
        variant: "destructive",
      });
      return false;
    }
  };

  const clearPurchased = async () => {
    if (!user) return false;

    try {
      await shoppingListApi.clearPurchased();
      setItems(prev => prev.filter(i => !i.is_purchased));
      toast({
        title: "Cleared",
        description: "Purchased items removed from list",
      });
      return true;
    } catch (error) {
      toast({
        title: "Error",
        description: "Failed to clear purchased items",
        variant: "destructive",
      });
      return false;
    }
  };

  const isInList = (itemId: string, itemType: 'food' | 'cosmetic') => {
    return items.some(i => i.item_id === itemId && i.item_type === itemType);
  };

  const toggleItem = async (itemId: string, itemType: 'food' | 'cosmetic') => {
    if (isInList(itemId, itemType)) {
      return removeItem(itemId, itemType);
    } else {
      return addItem(itemId, itemType);
    }
  };

  return {
    items,
    loading,
    addItem,
    removeItem,
    togglePurchased,
    updateQuantity,
    clearPurchased,
    isInList,
    toggleItem,
    refetch: fetchItems,
  };
};
