import { useState, useCallback, useRef, useEffect } from "react";

interface UseSpeechSynthesisOptions {
  onEnd?: () => void;
  onError?: (error: string) => void;
  rate?: number;
  pitch?: number;
  voiceLang?: string;
}

export function useSpeechSynthesis(options: UseSpeechSynthesisOptions = {}) {
  const { onEnd, onError, rate = 1, pitch = 1, voiceLang = "en-US" } = options;
  
  const [isSpeaking, setIsSpeaking] = useState(false);
  const [isPaused, setIsPaused] = useState(false);
  const [isSupported, setIsSupported] = useState(false);
  const [voices, setVoices] = useState<SpeechSynthesisVoice[]>([]);
  const utteranceRef = useRef<SpeechSynthesisUtterance | null>(null);

  // Check browser support and load voices
  useEffect(() => {
    if (typeof window !== "undefined" && "speechSynthesis" in window) {
      setIsSupported(true);

      const loadVoices = () => {
        const availableVoices = window.speechSynthesis.getVoices();
        setVoices(availableVoices);
      };

      loadVoices();
      
      // Voices might load asynchronously
      if (window.speechSynthesis.onvoiceschanged !== undefined) {
        window.speechSynthesis.onvoiceschanged = loadVoices;
      }
    }

    return () => {
      if (typeof window !== "undefined" && "speechSynthesis" in window) {
        window.speechSynthesis.cancel();
      }
    };
  }, []);

  // Clean text for better speech (remove markdown, etc.)
  const cleanTextForSpeech = useCallback((text: string): string => {
    return text
      // Remove markdown bold
      .replace(/\*\*([^*]+)\*\*/g, "$1")
      // Remove markdown italic
      .replace(/\*([^*]+)\*/g, "$1")
      // Remove bullet points and convert to pauses
      .replace(/^[•\-]\s*/gm, "")
      // Remove emojis (basic range)
      .replace(/[\u{1F600}-\u{1F6FF}]/gu, "")
      .replace(/[\u{1F300}-\u{1F5FF}]/gu, "")
      .replace(/[\u{1F680}-\u{1F6FF}]/gu, "")
      .replace(/[\u{2600}-\u{26FF}]/gu, "")
      .replace(/[\u{2700}-\u{27BF}]/gu, "")
      // Clean up multiple spaces
      .replace(/\s+/g, " ")
      .trim();
  }, []);

  const speak = useCallback((text: string) => {
    if (!isSupported) {
      onError?.("Text-to-speech is not supported in this browser");
      return;
    }

    // Cancel any ongoing speech
    window.speechSynthesis.cancel();

    const cleanedText = cleanTextForSpeech(text);
    if (!cleanedText) return;

    const utterance = new SpeechSynthesisUtterance(cleanedText);
    utteranceRef.current = utterance;

    // Find a suitable voice (prefer natural-sounding English voices)
    const preferredVoice = voices.find(
      (v) =>
        v.lang.startsWith(voiceLang.split("-")[0]) &&
        (v.name.includes("Natural") ||
          v.name.includes("Enhanced") ||
          v.name.includes("Premium") ||
          v.name.includes("Google") ||
          v.name.includes("Samantha") ||
          v.name.includes("Alex"))
    ) || voices.find((v) => v.lang.startsWith(voiceLang.split("-")[0]));

    if (preferredVoice) {
      utterance.voice = preferredVoice;
    }

    utterance.rate = rate;
    utterance.pitch = pitch;
    utterance.lang = voiceLang;

    utterance.onstart = () => {
      setIsSpeaking(true);
      setIsPaused(false);
    };

    utterance.onend = () => {
      setIsSpeaking(false);
      setIsPaused(false);
      onEnd?.();
    };

    utterance.onerror = (event) => {
      setIsSpeaking(false);
      setIsPaused(false);
      if (event.error !== "canceled") {
        onError?.(`Speech error: ${event.error}`);
      }
    };

    window.speechSynthesis.speak(utterance);
  }, [isSupported, voices, voiceLang, rate, pitch, cleanTextForSpeech, onEnd, onError]);

  const stop = useCallback(() => {
    if (isSupported) {
      window.speechSynthesis.cancel();
      setIsSpeaking(false);
      setIsPaused(false);
    }
  }, [isSupported]);

  const pause = useCallback(() => {
    if (isSupported && isSpeaking) {
      window.speechSynthesis.pause();
      setIsPaused(true);
    }
  }, [isSupported, isSpeaking]);

  const resume = useCallback(() => {
    if (isSupported && isPaused) {
      window.speechSynthesis.resume();
      setIsPaused(false);
    }
  }, [isSupported, isPaused]);

  const toggle = useCallback(() => {
    if (isPaused) {
      resume();
    } else if (isSpeaking) {
      pause();
    }
  }, [isPaused, isSpeaking, pause, resume]);

  return {
    speak,
    stop,
    pause,
    resume,
    toggle,
    isSpeaking,
    isPaused,
    isSupported,
    voices,
  };
}
