import { useState, useEffect, useCallback } from 'react';
import api from '@/lib/api';
import { useAuth } from './useAuth';

// Subscription tier configuration
export const SUBSCRIPTION_TIERS = {
  premium: {
    price_id: "price_YOUR_STRIPE_PRICE_ID", // Replace with your Stripe price ID
    product_id: "prod_YOUR_STRIPE_PRODUCT_ID", // Replace with your Stripe product ID
    name: "Premium",
    price: 9.99,
    interval: "month",
    trial_days: 7,
    features: [
      "Unlimited barcode scans",
      "AI-powered ingredient analysis",
      "Personalized health insights",
      "Priority support",
      "Ad-free experience",
      "Export health reports",
    ],
  },
} as const;

interface SubscriptionStatus {
  subscribed: boolean;
  is_trial: boolean;
  product_id: string | null;
  subscription_end: string | null;
  trial_end: string | null;
}

export const useSubscription = () => {
  const { user } = useAuth();
  const [subscriptionStatus, setSubscriptionStatus] = useState<SubscriptionStatus>({
    subscribed: false,
    is_trial: false,
    product_id: null,
    subscription_end: null,
    trial_end: null,
  });
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);

  const checkSubscription = useCallback(async () => {
    if (!user) {
      setSubscriptionStatus({
        subscribed: false,
        is_trial: false,
        product_id: null,
        subscription_end: null,
        trial_end: null,
      });
      return;
    }

    setLoading(true);
    setError(null);

    try {
      const data = await api.get<SubscriptionStatus>('/subscription/status');
      
      setSubscriptionStatus({
        subscribed: data.subscribed || false,
        is_trial: data.is_trial || false,
        product_id: data.product_id || null,
        subscription_end: data.subscription_end || null,
        trial_end: data.trial_end || null,
      });
    } catch (err) {
      console.error('Error checking subscription:', err);
      setError(err instanceof Error ? err.message : 'Failed to check subscription');
      // Don't show premium when status check fails (e.g. no subscription, API error)
      setSubscriptionStatus({
        subscribed: false,
        is_trial: false,
        product_id: null,
        subscription_end: null,
        trial_end: null,
      });
    } finally {
      setLoading(false);
    }
  }, [user]);

  const createCheckout = async () => {
    if (!user) {
      throw new Error('Please sign in to subscribe');
    }

    setLoading(true);
    setError(null);

    try {
      // Backend uses STRIPE_PRICE_ID from .env when priceId not sent (Premium with 7-day trial)
      const priceId = SUBSCRIPTION_TIERS.premium.price_id;
      const body = priceId && !priceId.startsWith('price_YOUR_') ? { priceId } : {};
      const data = await api.post<{ url: string }>('/subscription/create-checkout', body);

      if (data?.url) {
        window.open(data.url, '_blank');
      }
    } catch (err) {
      console.error('Error creating checkout:', err);
      setError(err instanceof Error ? err.message : 'Failed to create checkout');
      throw err;
    } finally {
      setLoading(false);
    }
  };

  const openCustomerPortal = async () => {
    if (!user) {
      throw new Error('Please sign in to manage subscription');
    }

    setLoading(true);
    setError(null);

    try {
      const data = await api.post<{ url: string }>('/subscription/customer-portal');
      
      if (data?.url) {
        window.open(data.url, '_blank');
      }
    } catch (err) {
      console.error('Error opening customer portal:', err);
      setError(err instanceof Error ? err.message : 'Failed to open customer portal');
      throw err;
    } finally {
      setLoading(false);
    }
  };

  // Check subscription on mount and when user changes
  useEffect(() => {
    checkSubscription();
  }, [checkSubscription]);

  // Refresh when user returns to the tab (e.g. after canceling in Stripe portal)
  useEffect(() => {
    if (!user) return;
    const onVisibilityChange = () => {
      if (document.visibilityState === 'visible') {
        checkSubscription();
      }
    };
    document.addEventListener('visibilitychange', onVisibilityChange);
    return () => document.removeEventListener('visibilitychange', onVisibilityChange);
  }, [user, checkSubscription]);

  // Refresh subscription status periodically
  useEffect(() => {
    if (!user) return;

    const interval = setInterval(() => {
      checkSubscription();
    }, 60000); // Check every minute

    return () => clearInterval(interval);
  }, [user, checkSubscription]);

  return {
    ...subscriptionStatus,
    loading,
    error,
    checkSubscription,
    createCheckout,
    openCustomerPortal,
    tier: subscriptionStatus.product_id 
      ? Object.values(SUBSCRIPTION_TIERS).find(t => t.product_id === subscriptionStatus.product_id) 
      : null,
  };
};
