import { useState, useCallback } from 'react';
import { useNavigate } from 'react-router-dom';
import api from '@/lib/api';
import { toast } from 'sonner';

export const SUBSCRIPTION_TIERS = {
  premium: {
    id: 'premium',
    name: 'Premium',
    price: 9.99,
    interval: 'month',
    priceId: 'price_YOUR_STRIPE_PRICE_ID', // Replace with your Stripe price ID
    features: [
      'Unlimited barcode scans',
      'AI-powered ingredient analysis',
      'Personalized recipe generation',
      'Advanced phytoestrogen tracking',
      'Priority support',
      'Ad-free experience',
    ],
  },
};

interface SubscriptionStatus {
  subscribed: boolean;
  is_trial: boolean;
  subscription_end: string | null;
  trial_end: string | null;
}

export const useSubscription = () => {
  const [loading, setLoading] = useState(false);
  const [status, setStatus] = useState<SubscriptionStatus>({
    subscribed: false,
    is_trial: false,
    subscription_end: null,
    trial_end: null,
  });
  const navigate = useNavigate();

  const checkSubscription = useCallback(async () => {
    setLoading(true);
    try {
      const data = await api.get<SubscriptionStatus>('/subscription/status');
      setStatus(data);
    } catch (error) {
      console.error('Error checking subscription:', error);
    } finally {
      setLoading(false);
    }
  }, []);

  const createCheckout = useCallback(async () => {
    setLoading(true);
    try {
      const { url } = await api.post<{ url: string }>('/subscription/create-checkout', {
        priceId: SUBSCRIPTION_TIERS.premium.priceId,
      });
      if (url) {
        window.open(url, '_blank');
      }
    } catch (error) {
      console.error('Error creating checkout:', error);
      toast.error('Failed to start checkout');
    } finally {
      setLoading(false);
    }
  }, []);

  const openCustomerPortal = useCallback(async () => {
    setLoading(true);
    try {
      const { url } = await api.post<{ url: string }>('/subscription/customer-portal');
      if (url) {
        window.open(url, '_blank');
      }
    } catch (error) {
      console.error('Error opening portal:', error);
      toast.error('Failed to open subscription management');
    } finally {
      setLoading(false);
    }
  }, []);

  return {
    ...status,
    loading,
    checkSubscription,
    createCheckout,
    openCustomerPortal,
  };
};
