import { useState, useCallback, useRef } from 'react';
import { useAuth } from './useAuth';

interface TypingUser {
  user_id: string;
  display_name: string;
  typing_at: string;
}

// For self-hosted version, typing indicators require WebSocket implementation
// This is a simplified version that works locally without real-time
export const useTypingIndicator = (conversationId: string) => {
  const { user } = useAuth();
  const [typingUsers, setTypingUsers] = useState<TypingUser[]>([]);
  const typingTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  const startTyping = useCallback(async (displayName: string) => {
    if (!user) return;

    // Clear existing timeout
    if (typingTimeoutRef.current) {
      clearTimeout(typingTimeoutRef.current);
    }

    // In a full implementation, this would send to WebSocket
    // For now, we just track locally
    console.log(`User ${displayName} started typing in ${conversationId}`);

    // Auto-stop typing after 3 seconds of no input
    typingTimeoutRef.current = setTimeout(() => {
      stopTyping();
    }, 3000);
  }, [user, conversationId]);

  const stopTyping = useCallback(async () => {
    if (!user) return;

    if (typingTimeoutRef.current) {
      clearTimeout(typingTimeoutRef.current);
      typingTimeoutRef.current = null;
    }

    // In a full implementation, this would send to WebSocket
    console.log(`User stopped typing in ${conversationId}`);
  }, [user, conversationId]);

  // Get display text for typing indicator
  const typingText = typingUsers.length > 0
    ? typingUsers.length === 1
      ? `${typingUsers[0].display_name} is typing...`
      : `${typingUsers.length} people are typing...`
    : null;

  return {
    typingUsers,
    typingText,
    isTyping: typingUsers.length > 0,
    startTyping,
    stopTyping,
  };
};
