import { useState, useCallback, useRef, useEffect } from 'react';
import { useToast } from './use-toast';

// Type for native BarcodeDetector API
declare global {
  interface Window {
    BarcodeDetector?: new (options?: { formats: string[] }) => BarcodeDetector;
  }
  interface BarcodeDetector {
    detect(image: ImageBitmapSource): Promise<DetectedBarcode[]>;
  }
  interface DetectedBarcode {
    boundingBox: DOMRectReadOnly;
    rawValue: string;
    format: string;
    cornerPoints: { x: number; y: number }[];
  }
}

// Check if BarcodeDetector API is available
const isBarcodeDetectorSupported = () => {
  return 'BarcodeDetector' in window;
};

export const useWebBarcodeScanner = () => {
  const { toast } = useToast();
  const [isScanning, setIsScanning] = useState(false);
  const [lastScannedCode, setLastScannedCode] = useState<string | null>(null);
  const [apiSupported, setApiSupported] = useState<boolean | null>(null);
  
  const scannerElementId = 'web-barcode-scanner';
  const videoRef = useRef<HTMLVideoElement | null>(null);
  const canvasRef = useRef<HTMLCanvasElement | null>(null);
  const detectorRef = useRef<BarcodeDetector | null>(null);
  const streamRef = useRef<MediaStream | null>(null);
  const animationFrameRef = useRef<number | null>(null);
  const isScanningRef = useRef(false);
  const lastDetectedRef = useRef<string | null>(null);
  const detectionCountRef = useRef(0);
  const onSuccessRef = useRef<((code: string) => void) | null>(null);

  // Check API support on mount
  useEffect(() => {
    setApiSupported(isBarcodeDetectorSupported());
  }, []);

  const stopScan = useCallback(async () => {
    isScanningRef.current = false;
    
    if (animationFrameRef.current) {
      cancelAnimationFrame(animationFrameRef.current);
      animationFrameRef.current = null;
    }
    
    if (streamRef.current) {
      streamRef.current.getTracks().forEach(track => track.stop());
      streamRef.current = null;
    }
    
    if (videoRef.current) {
      videoRef.current.srcObject = null;
    }
    
    setIsScanning(false);
    lastDetectedRef.current = null;
    detectionCountRef.current = 0;
  }, []);

  const scanFrame = useCallback(async () => {
    if (!isScanningRef.current || !videoRef.current || !canvasRef.current || !detectorRef.current) {
      return;
    }

    const video = videoRef.current;
    const canvas = canvasRef.current;
    const ctx = canvas.getContext('2d');

    if (!ctx || video.readyState !== video.HAVE_ENOUGH_DATA) {
      animationFrameRef.current = requestAnimationFrame(scanFrame);
      return;
    }

    // Set canvas size to match video
    canvas.width = video.videoWidth;
    canvas.height = video.videoHeight;

    // Draw video frame to canvas
    ctx.drawImage(video, 0, 0);

    try {
      const barcodes = await detectorRef.current.detect(canvas);
      
      if (barcodes.length > 0) {
        const barcode = barcodes[0];
        const code = barcode.rawValue;

        console.log('Barcode detected:', code, 'Format:', barcode.format);

        // Require multiple consistent detections
        if (lastDetectedRef.current === code) {
          detectionCountRef.current++;
        } else {
          lastDetectedRef.current = code;
          detectionCountRef.current = 1;
        }

        // Draw detection box
        ctx.strokeStyle = '#00ff00';
        ctx.lineWidth = 4;
        ctx.strokeRect(
          barcode.boundingBox.x,
          barcode.boundingBox.y,
          barcode.boundingBox.width,
          barcode.boundingBox.height
        );

        // Confirm after 2 consistent detections
        if (detectionCountRef.current >= 2) {
          setLastScannedCode(code);
          toast({
            title: 'Barcode scanned',
            description: `Code: ${code} (${barcode.format})`,
          });
          
          if (onSuccessRef.current) {
            onSuccessRef.current(code);
          }
          
          stopScan();
          return;
        }
      }
    } catch (error) {
      console.error('Detection error:', error);
    }

    // Continue scanning
    if (isScanningRef.current) {
      animationFrameRef.current = requestAnimationFrame(scanFrame);
    }
  }, [toast, stopScan]);

  const startScan = useCallback(
    async (onSuccess: (code: string) => void): Promise<void> => {
      if (isScanningRef.current) {
        console.log('Scanner already running');
        return;
      }

      // Check API support
      if (!isBarcodeDetectorSupported()) {
        toast({
          title: 'Not supported',
          description: 'Barcode scanning is not supported in this browser. Try Chrome or Edge.',
          variant: 'destructive',
        });
        return;
      }

      const element = document.getElementById(scannerElementId);
      if (!element) {
        console.error('Scanner element not found');
        return;
      }

      onSuccessRef.current = onSuccess;

      try {
        // Create BarcodeDetector with all common formats
        detectorRef.current = new window.BarcodeDetector!({
          formats: [
            'ean_13',
            'ean_8',
            'upc_a',
            'upc_e',
            'code_128',
            'code_39',
            'code_93',
            'codabar',
            'itf',
            'qr_code',
            'data_matrix',
          ],
        });

        // Request camera access
        const stream = await navigator.mediaDevices.getUserMedia({
          video: {
            facingMode: 'environment',
            width: { ideal: 1280 },
            height: { ideal: 720 },
          },
        });

        streamRef.current = stream;

        // Create video element
        let video = element.querySelector('video') as HTMLVideoElement;
        if (!video) {
          video = document.createElement('video');
          video.setAttribute('playsinline', 'true');
          video.setAttribute('autoplay', 'true');
          video.setAttribute('muted', 'true');
          video.style.width = '100%';
          video.style.height = '100%';
          video.style.objectFit = 'cover';
          element.appendChild(video);
        }
        videoRef.current = video;

        // Create canvas for detection (hidden)
        let canvas = element.querySelector('canvas') as HTMLCanvasElement;
        if (!canvas) {
          canvas = document.createElement('canvas');
          canvas.style.display = 'none';
          element.appendChild(canvas);
        }
        canvasRef.current = canvas;

        // Connect stream to video
        video.srcObject = stream;
        await video.play();

        isScanningRef.current = true;
        setIsScanning(true);

        console.log('Native BarcodeDetector started');

        // Start detection loop
        animationFrameRef.current = requestAnimationFrame(scanFrame);
      } catch (error) {
        console.error('Error starting scanner:', error);
        isScanningRef.current = false;
        setIsScanning(false);
        
        if (error instanceof Error && error.name === 'NotAllowedError') {
          toast({
            title: 'Camera access denied',
            description: 'Please allow camera access to scan barcodes.',
            variant: 'destructive',
          });
        } else {
          toast({
            title: 'Camera error',
            description: 'Could not access camera. Please check permissions.',
            variant: 'destructive',
          });
        }
      }
    },
    [toast, scanFrame, stopScan]
  );

  // Cleanup on unmount
  useEffect(() => {
    return () => {
      isScanningRef.current = false;
      if (animationFrameRef.current) {
        cancelAnimationFrame(animationFrameRef.current);
      }
      if (streamRef.current) {
        streamRef.current.getTracks().forEach(track => track.stop());
      }
    };
  }, []);

  return {
    isScanning,
    lastScannedCode,
    startScan,
    stopScan,
    scannerElementId,
    apiSupported,
    // Keep for API compatibility
    availableCameras: [] as { id: string; label: string }[],
    selectedCameraId: 'default' as string | null,
    switchCamera: async (_cameraId: string) => {},
  };
};
