// API Configuration for self-hosted backend
// Change this URL to point to your server
export const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

interface ApiOptions {
  method?: 'GET' | 'POST' | 'PUT' | 'PATCH' | 'DELETE';
  body?: any;
  headers?: Record<string, string>;
}

class ApiClient {
  private baseUrl: string;
  private accessToken: string | null = null;
  private refreshToken: string | null = null;

  constructor(baseUrl: string) {
    this.baseUrl = baseUrl;
    this.loadTokens();
  }

  private loadTokens() {
    this.accessToken = localStorage.getItem('accessToken');
    this.refreshToken = localStorage.getItem('refreshToken');
  }

  setTokens(accessToken: string, refreshToken: string) {
    this.accessToken = accessToken;
    this.refreshToken = refreshToken;
    localStorage.setItem('accessToken', accessToken);
    localStorage.setItem('refreshToken', refreshToken);
  }

  clearTokens() {
    this.accessToken = null;
    this.refreshToken = null;
    localStorage.removeItem('accessToken');
    localStorage.removeItem('refreshToken');
  }

  getAccessToken() {
    return this.accessToken;
  }

  isAuthenticated() {
    return !!this.accessToken;
  }

  private async refreshAccessToken(): Promise<boolean> {
    if (!this.refreshToken) return false;

    try {
      const response = await fetch(`${this.baseUrl}/auth/refresh`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ refreshToken: this.refreshToken }),
      });

      if (!response.ok) {
        this.clearTokens();
        return false;
      }

      const data = await response.json();
      this.setTokens(data.accessToken, data.refreshToken);
      return true;
    } catch {
      this.clearTokens();
      return false;
    }
  }

  async request<T>(endpoint: string, options: ApiOptions = {}): Promise<T> {
    const { method = 'GET', body, headers = {} } = options;

    const requestHeaders: Record<string, string> = {
      'Content-Type': 'application/json',
      ...headers,
    };

    if (this.accessToken) {
      requestHeaders['Authorization'] = `Bearer ${this.accessToken}`;
    }

    let response = await fetch(`${this.baseUrl}${endpoint}`, {
      method,
      headers: requestHeaders,
      body: body ? JSON.stringify(body) : undefined,
    });

    // Handle token expiration
    if (response.status === 401 && this.refreshToken) {
      const refreshed = await this.refreshAccessToken();
      if (refreshed) {
        requestHeaders['Authorization'] = `Bearer ${this.accessToken}`;
        response = await fetch(`${this.baseUrl}${endpoint}`, {
          method,
          headers: requestHeaders,
          body: body ? JSON.stringify(body) : undefined,
        });
      }
    }

    if (!response.ok) {
      let errorMessage = 'Request failed';
      try {
        const data = await response.json();
        errorMessage = data.error || errorMessage;
      } catch {
        errorMessage = response.statusText || `Error ${response.status}`;
      }
      throw new Error(errorMessage);
    }

    return response.json();
  }

  // Convenience methods
  get<T>(endpoint: string) {
    return this.request<T>(endpoint, { method: 'GET' });
  }

  post<T>(endpoint: string, body?: any) {
    return this.request<T>(endpoint, { method: 'POST', body });
  }

  put<T>(endpoint: string, body?: any) {
    return this.request<T>(endpoint, { method: 'PUT', body });
  }

  patch<T>(endpoint: string, body?: any) {
    return this.request<T>(endpoint, { method: 'PATCH', body });
  }

  delete<T>(endpoint: string) {
    return this.request<T>(endpoint, { method: 'DELETE' });
  }
}

export const api = new ApiClient(API_URL);

// Auth helpers
export const auth = {
  async signUp(email: string, password: string) {
    const data = await api.post<{ user: any; accessToken: string; refreshToken: string }>('/auth/signup', { email, password });
    api.setTokens(data.accessToken, data.refreshToken);
    return { user: data.user, error: null };
  },

  async signIn(email: string, password: string) {
    const data = await api.post<{ user: any; accessToken: string; refreshToken: string }>('/auth/login', { email, password });
    api.setTokens(data.accessToken, data.refreshToken);
    return { user: data.user, error: null };
  },

  async signOut() {
    try {
      await api.post('/auth/logout');
    } finally {
      api.clearTokens();
    }
  },

  async getUser() {
    if (!api.isAuthenticated()) return { user: null };
    try {
      const data = await api.get<{ user: any }>('/auth/me');
      return { user: data.user };
    } catch {
      return { user: null };
    }
  },

  isAuthenticated() {
    return api.isAuthenticated();
  },

  onAuthStateChange(callback: (event: string, session: any) => void) {
    // Check initial state
    const token = api.getAccessToken();
    if (token) {
      auth.getUser().then(({ user }) => {
        if (user) {
          callback('SIGNED_IN', { user });
        }
      });
    }

    // Listen for storage changes (cross-tab sync)
    const handler = (e: StorageEvent) => {
      if (e.key === 'accessToken') {
        if (e.newValue) {
          auth.getUser().then(({ user }) => {
            callback('SIGNED_IN', { user });
          });
        } else {
          callback('SIGNED_OUT', null);
        }
      }
    };

    window.addEventListener('storage', handler);
    return {
      data: { subscription: { unsubscribe: () => window.removeEventListener('storage', handler) } }
    };
  }
};

export default api;
