import api from '@/lib/api';

// Favorites API
export const favoritesApi = {
  getAll: () => api.get<any[]>('/favorites'),
  add: (itemId: string, itemType: string) => api.post('/favorites', { item_id: itemId, item_type: itemType }),
  remove: (itemType: string, itemId: string) => api.delete(`/favorites/${itemType}/${itemId}`),
  check: (itemType: string, itemId: string) => api.get<{ isFavorite: boolean }>(`/favorites/check/${itemType}/${itemId}`),
};

// Shopping List API
export const shoppingListApi = {
  getAll: () => api.get<any[]>('/shopping-list'),
  add: (itemId: string, itemType: string, quantity?: number, notes?: string) => 
    api.post('/shopping-list', { item_id: itemId, item_type: itemType, quantity, notes }),
  update: (id: string, updates: any) => api.patch(`/shopping-list/${id}`, updates),
  toggle: (id: string) => api.patch(`/shopping-list/${id}/toggle`),
  remove: (id: string) => api.delete(`/shopping-list/${id}`),
  clearPurchased: () => api.delete('/shopping-list/clear/purchased'),
};

// Recipes API
export const recipesApi = {
  getAll: () => api.get<any[]>('/recipes'),
  get: (id: string) => api.get<any>(`/recipes/${id}`),
  getShared: (token: string) => api.get<any>(`/recipes/shared/${token}`),
  create: (recipe: any) => api.post('/recipes', recipe),
  update: (id: string, updates: any) => api.patch(`/recipes/${id}`, updates),
  delete: (id: string) => api.delete(`/recipes/${id}`),
};

// Meal Plans API
export const mealPlansApi = {
  getAll: (startDate?: string, endDate?: string) => {
    const params = new URLSearchParams();
    if (startDate) params.set('start_date', startDate);
    if (endDate) params.set('end_date', endDate);
    return api.get<any[]>(`/meal-plans?${params.toString()}`);
  },
  create: (plan: any) => api.post('/meal-plans', plan),
  update: (id: string, updates: any) => api.patch(`/meal-plans/${id}`, updates),
  delete: (id: string) => api.delete(`/meal-plans/${id}`),
};

// Scan History API
export const scanHistoryApi = {
  getAll: (limit?: number) => api.get<any[]>(`/scan-history${limit ? `?limit=${limit}` : ''}`),
  add: (scan: any) => api.post('/scan-history', scan),
  remove: (id: string) => api.delete(`/scan-history/${id}`),
  clearAll: () => api.delete('/scan-history'),
};

// Community API
export const communityApi = {
  getPosts: (category?: string, limit?: number, offset?: number, userId?: string, userIds?: string[]) => {
    const params = new URLSearchParams();
    if (category) params.set('category', category);
    if (limit) params.set('limit', String(limit));
    if (offset) params.set('offset', String(offset));
    if (userId) params.set('user_id', userId);
    if (userIds?.length) params.set('user_ids', userIds.join(','));
    return api.get<any[]>(`/community/posts?${params.toString()}`);
  },
  getRepliesByUser: (userId: string) =>
    api.get<any[]>(`/community/replies?user_id=${encodeURIComponent(userId)}`),
  getAllReplies: (limit?: number) =>
    api.get<any[]>(`/community/replies${limit != null ? `?limit=${limit}` : ''}`),
  createPost: (post: any) => api.post('/community/posts', post),
  likePost: (id: string) => api.post<{ liked: boolean }>(`/community/posts/${id}/like`),
  getReplies: (postId: string) => api.get<any[]>(`/community/posts/${postId}/replies`),
  addReply: (postId: string, content: string, authorName: string) =>
    api.post(`/community/posts/${postId}/replies`, { content, author_name: authorName }),
  getFollows: () => api.get<any[]>('/community/follows'),
  follow: (userId: string) => api.post(`/community/follow/${userId}`),
  unfollow: (userId: string) => api.delete(`/community/follow/${userId}`),
  getStats: (userId: string) =>
    api.get<{ followersCount: number; followingCount: number }>(`/community/stats/${userId}`),
};

// User Profile API
export const userApi = {
  getProfile: () => api.get<any>('/users/profile'),
  updateProfile: (updates: any) => api.patch('/users/profile', updates),
  uploadAvatar: (base64Data: string) => api.post<{ avatar_url: string }>('/users/avatar', { data: base64Data }),
  searchProfiles: (query: string) => api.get<any[]>(`/users/search?q=${encodeURIComponent(query)}`),
  getHealthProfile: () => api.get<any>('/users/health-profile'),
  updateHealthProfile: (profile: any) => api.put('/users/health-profile', profile),
  getDietaryPreferences: () => api.get<any>('/users/dietary-preferences'),
  updateDietaryPreferences: (prefs: any) => api.put('/users/dietary-preferences', prefs),
};

// AI API
export const aiApi = {
  analyzeIngredients: (ingredients: string, analysisType?: string) =>
    api.post<{ analysis: string }>('/ai/analyze-ingredients', { ingredients, analysisType }),
  generateRecipes: (params: any) =>
    api.post<{ recipe: string }>('/ai/generate-recipes', params),
  generatePhytoestrogenFreeRecipes: (params: any) =>
    api.post<{ recipe: string }>('/ai/generate-recipes', { ...params, phytoestrogenFree: true }),
  extractIngredients: (recipes: { title: string; content: string; multiplier?: number }[]) =>
    api.post<{ categories: any[]; summary: any }>('/ai/extract-ingredients', { recipes }),
  analyzePhytoestrogens: (ingredients: string) =>
    api.post<{ analysis: string }>('/ai/analyze-phytoestrogens', { ingredients }),
  healthChat: (message: string, conversationHistory?: any[]) =>
    api.post<{ response: string }>('/ai/health-chat', { message, conversationHistory }),
  analyzeNutrition: (ingredients: string, servingSize?: string) =>
    api.post<{ nutrition: string }>('/ai/analyze-nutrition', { ingredients, servingSize }),
  lookupBarcode: (barcode: string) =>
    api.post<{ found: boolean; product?: any }>('/ai/lookup-barcode', { barcode }),
};

// Notifications API (simplified - in self-hosted, you'd add these backend routes)
export const notificationsApi = {
  getAll: () => api.get<any[]>('/notifications'),
  markAsRead: (id: string) => api.patch(`/notifications/${id}/read`),
  markAllAsRead: () => api.post('/notifications/mark-all-read'),
  delete: (id: string) => api.delete(`/notifications/${id}`),
  clearAll: () => api.delete('/notifications'),
};

// Follows API
export const followsApi = {
  getFollowing: () => communityApi.getFollows(),
  follow: (userId: string) => communityApi.follow(userId),
  unfollow: (userId: string) => communityApi.unfollow(userId),
  getStats: (userId: string) => communityApi.getStats(userId),
};

// Achievements API
export const achievementsApi = {
  getAll: (userId?: string) =>
    api.get<any[]>(userId ? `/achievements?user_id=${userId}` : '/achievements'),
  unlock: (achievementId: string, progress: number) =>
    api.post('/achievements', { achievement_id: achievementId, progress }),
};

// Phytoestrogen API
export const phytoestrogenApi = {
  getExposure: () => api.get<any[]>('/phytoestrogen/exposure'),
  logExposure: (entry: {
    phytoestrogen_id: string;
    phytoestrogen_name: string;
    potency: string;
    food_source: string;
    servings?: number;
    notes?: string;
    logged_at?: string;
  }) => api.post('/phytoestrogen/exposure', entry),
  deleteExposure: (id: string) => api.delete(`/phytoestrogen/exposure/${id}`),
  getGoals: () => api.get<any>('/phytoestrogen/goals'),
  saveGoals: (goals: { daily_limit?: number; weekly_limit?: number; show_alerts?: boolean; alert_at_percentage?: number }) =>
    api.put('/phytoestrogen/goals', goals),
};

// User by id (public profile)
export const userPublicApi = {
  getProfile: (userId: string) => api.get<any>(`/users/${userId}`),
};

// API base URL for assets (avatars, etc.)
export const API_BASE = (import.meta.env.VITE_API_URL || 'http://localhost:3001/api').replace(/\/api\/?$/, '');
