import { Link } from 'react-router-dom';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Progress } from '@/components/ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  ArrowLeft, 
  Trophy, 
  Star,
  Lock,
  Sparkles,
  Medal,
  Crown,
  Loader2
} from 'lucide-react';
import { useAchievements, ACHIEVEMENTS, Achievement } from '@/hooks/useAchievements';
import { useAuth } from '@/hooks/useAuth';
import { SocialShare } from '@/components/SocialShare';
import { Footer } from '@/components/Footer';

const rarityColors = {
  common: 'bg-gray-100 text-gray-700 border-gray-300',
  uncommon: 'bg-green-100 text-green-700 border-green-300',
  rare: 'bg-blue-100 text-blue-700 border-blue-300',
  epic: 'bg-purple-100 text-purple-700 border-purple-300',
  legendary: 'bg-amber-100 text-amber-700 border-amber-300',
};

const rarityGlow = {
  common: '',
  uncommon: '',
  rare: 'ring-2 ring-blue-200',
  epic: 'ring-2 ring-purple-300 shadow-purple-200 shadow-lg',
  legendary: 'ring-2 ring-amber-400 shadow-amber-200 shadow-xl animate-pulse',
};

const categoryLabels = {
  beginner: { label: 'Beginner', icon: Star },
  explorer: { label: 'Explorer', icon: Sparkles },
  health: { label: 'Health', icon: Medal },
  master: { label: 'Master', icon: Crown },
  special: { label: 'Special', icon: Trophy },
};

const AchievementCard = ({ 
  achievement, 
  isUnlocked, 
  progress 
}: { 
  achievement: Achievement; 
  isUnlocked: boolean; 
  progress: number;
}) => {
  const progressPercent = Math.min((progress / achievement.requirement) * 100, 100);
  
  return (
    <div 
      className={`
        relative p-4 rounded-xl border-2 transition-all duration-300
        ${isUnlocked 
          ? `${rarityColors[achievement.rarity]} ${rarityGlow[achievement.rarity]}` 
          : 'bg-muted/50 border-muted text-muted-foreground'
        }
      `}
    >
      {!isUnlocked && (
        <div className="absolute top-2 right-2">
          <Lock className="h-4 w-4" />
        </div>
      )}
      
      <div className="text-4xl mb-3 text-center">
        {isUnlocked ? achievement.icon : '🔒'}
      </div>
      
      <h3 className={`font-bold text-center mb-1 ${!isUnlocked && 'text-muted-foreground'}`}>
        {achievement.name}
      </h3>
      
      <p className="text-xs text-center mb-3 min-h-[32px]">
        {achievement.description}
      </p>
      
      {!isUnlocked && (
        <div className="space-y-1">
          <div className="flex justify-between text-xs">
            <span>Progress</span>
            <span>{progress}/{achievement.requirement}</span>
          </div>
          <Progress value={progressPercent} className="h-2" />
        </div>
      )}
      
      {isUnlocked && (
        <div className="text-center">
          <Badge variant="outline" className={rarityColors[achievement.rarity]}>
            {achievement.rarity.charAt(0).toUpperCase() + achievement.rarity.slice(1)}
          </Badge>
        </div>
      )}
    </div>
  );
};

const Achievements = () => {
  const { user, loading: authLoading } = useAuth();
  const { 
    achievements, 
    loading, 
    isAchievementUnlocked, 
    getAchievementProgress,
    getUnlockedCount,
    getTotalPoints 
  } = useAchievements();

  if (authLoading || loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  if (!user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        <Card className="max-w-md">
          <CardContent className="p-6 text-center">
            <Trophy className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
            <h2 className="text-xl font-semibold mb-2">Sign In Required</h2>
            <p className="text-muted-foreground mb-4">Sign in to track your achievements and earn badges.</p>
            <Link to="/auth">
              <Button>Sign In</Button>
            </Link>
          </CardContent>
        </Card>
      </div>
    );
  }

  const unlockedCount = getUnlockedCount();
  const totalPoints = getTotalPoints();
  const categories = ['beginner', 'explorer', 'health', 'master', 'special'] as const;

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      <div className="container mx-auto px-4 py-8 max-w-6xl">
        {/* Header */}
        <div className="flex items-center gap-4 mb-8">
          <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
            <ArrowLeft className="h-4 w-4" />
            Back
          </Link>
          <div className="flex-1">
            <h1 className="text-3xl font-bold flex items-center gap-3">
              <Trophy className="h-8 w-8 text-primary" />
              Achievements
            </h1>
            <p className="text-muted-foreground">Earn badges for healthy shopping habits</p>
          </div>
          <SocialShare 
            title="My TOBIN Health Achievements"
            text={`I've earned ${unlockedCount} achievements and ${totalPoints} points on TOBIN Health! 🏆`}
          />
        </div>

        {/* Stats Summary */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
          <Card className="bg-gradient-to-br from-primary/10 to-primary/5">
            <CardContent className="p-6 text-center">
              <Trophy className="h-10 w-10 mx-auto text-primary mb-2" />
              <p className="text-4xl font-bold">{unlockedCount}</p>
              <p className="text-sm text-muted-foreground">of {achievements.length} Unlocked</p>
            </CardContent>
          </Card>
          
          <Card className="bg-gradient-to-br from-amber-100 to-amber-50 dark:from-amber-950/30 dark:to-amber-900/20">
            <CardContent className="p-6 text-center">
              <Star className="h-10 w-10 mx-auto text-amber-500 mb-2" />
              <p className="text-4xl font-bold">{totalPoints}</p>
              <p className="text-sm text-muted-foreground">Total Points</p>
            </CardContent>
          </Card>
          
          <Card>
            <CardContent className="p-6 text-center">
              <Medal className="h-10 w-10 mx-auto text-blue-500 mb-2" />
              <p className="text-4xl font-bold">
                {Math.round((unlockedCount / achievements.length) * 100)}%
              </p>
              <p className="text-sm text-muted-foreground">Completion</p>
            </CardContent>
          </Card>
        </div>

        {/* Achievement Categories */}
        <Tabs defaultValue="all" className="space-y-6">
          <TabsList className="flex-wrap h-auto gap-2 p-2">
            <TabsTrigger value="all" className="gap-2">
              All
            </TabsTrigger>
            {categories.map(cat => {
              const { label, icon: Icon } = categoryLabels[cat];
              return (
                <TabsTrigger key={cat} value={cat} className="gap-2">
                  <Icon className="h-4 w-4" />
                  {label}
                </TabsTrigger>
              );
            })}
          </TabsList>

          <TabsContent value="all">
            <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
              {achievements.map(achievement => (
                <AchievementCard
                  key={achievement.id}
                  achievement={achievement}
                  isUnlocked={isAchievementUnlocked(achievement.id)}
                  progress={getAchievementProgress(achievement.id)}
                />
              ))}
            </div>
          </TabsContent>

          {categories.map(cat => (
            <TabsContent key={cat} value={cat}>
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4">
                {achievements
                  .filter(a => a.category === cat)
                  .map(achievement => (
                    <AchievementCard
                      key={achievement.id}
                      achievement={achievement}
                      isUnlocked={isAchievementUnlocked(achievement.id)}
                      progress={getAchievementProgress(achievement.id)}
                    />
                  ))}
              </div>
            </TabsContent>
          ))}
        </Tabs>

        {/* Rarity Legend */}
        <Card className="mt-8">
          <CardHeader>
            <CardTitle className="text-lg">Rarity Guide</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex flex-wrap gap-3">
              {(['common', 'uncommon', 'rare', 'epic', 'legendary'] as const).map(rarity => (
                <Badge key={rarity} className={`${rarityColors[rarity]} px-3 py-1`}>
                  {rarity.charAt(0).toUpperCase() + rarity.slice(1)}
                  <span className="ml-2 opacity-70">
                    {rarity === 'common' ? '10 pts' : 
                     rarity === 'uncommon' ? '25 pts' : 
                     rarity === 'rare' ? '50 pts' : 
                     rarity === 'epic' ? '100 pts' : '250 pts'}
                  </span>
                </Badge>
              ))}
            </div>
          </CardContent>
        </Card>
      </div>
      
      <Footer />
    </div>
  );
};

export default Achievements;
