import { useState } from 'react';
import { Link } from 'react-router-dom';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useScanHistory, ScanProductDetails } from '@/hooks/useScanHistory';
import { useAuth } from '@/hooks/useAuth';
import { useSubscription } from '@/hooks/useSubscription';
import { useBarcodeLookup, OpenFoodFactsProduct, AIAnalysis } from '@/hooks/useBarcodeLookup';
import { useCommunityProducts } from '@/hooks/useCommunityProducts';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Badge } from '@/components/ui/badge';
import { ScrollArea } from '@/components/ui/scroll-area';
import { 
  ArrowLeft, Camera, ScanLine, AlertTriangle, 
  History, Trash2, RotateCcw, Clock, Users, Crown 
} from 'lucide-react';
import { formatDistanceToNow } from 'date-fns';
import { UserMenu } from '@/components/UserMenu';
import { WebBarcodeScanner } from '@/components/WebBarcodeScanner';
import { ScannedProductCard } from '@/components/ScannedProductCard';
import { AddProductForm } from '@/components/AddProductForm';
import { foodDatabase } from '@/data/foodDatabase';
import { cosmeticsDatabase } from '@/data/cosmeticsDatabase';
import { toast } from 'sonner';
import foodPlaceholder from '@/assets/food-placeholder.png';
import cosmeticPlaceholder from '@/assets/cosmetic-placeholder.png';

// Local database for cosmetics (Open Food Facts is food-only)
const localBarcodeDatabase: Record<string, { id: string; type: 'food' | 'cosmetic'; name: string }> = {
  // Demo cosmetic barcodes - IDs must match cosmeticsDatabase exactly
  '3456789012345': { id: 'natural-deodorant', type: 'cosmetic', name: 'Natural Deodorant' },
  '4567890123456': { id: 'mineral-sunscreen', type: 'cosmetic', name: 'Mineral Sunscreen' },
  '381370012344': { id: 'conventional-perfume', type: 'cosmetic', name: 'Conventional Perfume' },
  '096200004567': { id: 'nail-polish', type: 'cosmetic', name: 'Nail Polish (Conventional)' },
  '071249123456': { id: 'anti-aging-cream', type: 'cosmetic', name: 'Anti-Aging Cream' },
  '074764234567': { id: 'conventional-lipstick', type: 'cosmetic', name: 'Lipstick (Conventional)' },
  '381519012345': { id: 'conventional-shampoo', type: 'cosmetic', name: 'Shampoo (Conventional)' },
  '305210234567': { id: 'conventional-lotion', type: 'cosmetic', name: 'Body Lotion (Conventional)' },
  '041167123456': { id: 'chemical-sunscreen', type: 'cosmetic', name: 'Chemical Sunscreen' },
  '012044234567': { id: 'conventional-deodorant', type: 'cosmetic', name: 'Antiperspirant/Deodorant' },
  '851201012345': { id: 'mineral-makeup', type: 'cosmetic', name: 'Mineral Makeup' },
  '857068123456': { id: 'natural-soap', type: 'cosmetic', name: 'Natural/Castile Soap' },
  '897017234567': { id: 'mineral-sunscreen', type: 'cosmetic', name: 'Mineral Sunscreen' },
  '898220012345': { id: 'coconut-oil', type: 'cosmetic', name: 'Pure Coconut Oil (Skincare)' },
  '856858123456': { id: 'natural-deodorant', type: 'cosmetic', name: 'Natural Deodorant' },
  '892884234567': { id: 'clean-mascara', type: 'cosmetic', name: 'Clean Mascara' },
};

interface LocalProduct {
  id: string;
  type: 'food' | 'cosmetic';
  name: string;
  barcode: string;
  details?: any;
}

const BarcodeScanner = () => {
  const { user } = useAuth();
  const { subscribed, loading: subscriptionLoading } = useSubscription();
  const { addItem, removeItem, isInList } = useShoppingList();
  const { removeFavorite, isFavorite } = useFavorites();
  const { history, loading: historyLoading, addToHistory, deleteFromHistory, clearHistory } = useScanHistory();
  const { lookupBarcode: lookupOpenFoodFacts, loading: apiLoading } = useBarcodeLookup();
  const { addProduct: addCommunityProduct, loading: savingProduct } = useCommunityProducts();
  
  const [manualCode, setManualCode] = useState('');
  const [showScanner, setShowScanner] = useState(false);
  const [currentBarcode, setCurrentBarcode] = useState('');
  const [localProduct, setLocalProduct] = useState<LocalProduct | null>(null);
  const [openFoodFactsProduct, setOpenFoodFactsProduct] = useState<OpenFoodFactsProduct | null>(null);
  const [productSource, setProductSource] = useState<'local' | 'openfoodfacts' | 'ai' | null>(null);
  const [apiSourceString, setApiSourceString] = useState<string | null>(null);
  const [isAiGenerated, setIsAiGenerated] = useState(false);
  const [aiAnalysis, setAiAnalysis] = useState<AIAnalysis | null>(null);
  const [productType, setProductType] = useState<'food' | 'cosmetic'>('food');
  const [notFound, setNotFound] = useState(false);
  const [notFoundError, setNotFoundError] = useState<string | null>(null);
  const [showAddForm, setShowAddForm] = useState(false);

  const lookupBarcode = async (code: string) => {
    setCurrentBarcode(code);
    setLocalProduct(null);
    setOpenFoodFactsProduct(null);
    setProductSource(null);
    setApiSourceString(null);
    setIsAiGenerated(false);
    setAiAnalysis(null);
    setProductType('food');
    setNotFound(false);
    setNotFoundError(null);
    setShowAddForm(false);

    // First check local database (for cosmetics)
    const localMatch = localBarcodeDatabase[code];
    if (localMatch) {
      const details = localMatch.type === 'food' 
        ? foodDatabase.find(f => f.id === localMatch.id)
        : cosmeticsDatabase.find(c => c.id === localMatch.id);
      
      setLocalProduct({ ...localMatch, barcode: code, details });
      setProductSource('local');
      if (user) {
        await addToHistory(code, localMatch.name, localMatch.type, localMatch.id, null);
      }
      return;
    }

    // Then try Open Food Facts / Open Beauty Facts API (with AI fallback)
    const result = await lookupOpenFoodFacts(code);
    if (result.found && result.product) {
      setOpenFoodFactsProduct(result.product);
      setProductSource(result.aiGenerated ? 'ai' : 'openfoodfacts');
      setApiSourceString(result.source || null);
      setIsAiGenerated(result.aiGenerated || false);
      setAiAnalysis(result.aiAnalysis || null);
      
      // Determine product type from API response (Open Beauty Facts = cosmetic)
      const detectedProductType = result.productType || 'food';
      setProductType(detectedProductType);

      const productDetails: ScanProductDetails = {
        brand: result.product.brand || undefined,
        categories: result.product.categories || undefined,
        ingredients: result.product.ingredients || undefined,
        imageUrl: result.product.imageUrl,
        nutrition: result.product.nutrition ? {
          calories: result.product.nutrition.calories,
          fat: result.product.nutrition.fat,
          carbs: result.product.nutrition.carbs,
          protein: result.product.nutrition.protein,
        } : undefined,
        nutriscoreGrade: result.product.nutriscoreGrade || undefined,
        novaGroup: result.product.novaGroup ?? undefined,
        aiAnalysis: result.aiAnalysis ? {
          safetyLevel: result.aiAnalysis.safetyRating === 'Safe' ? 'safe' : result.aiAnalysis.safetyRating === 'Moderate Concern' ? 'moderate_concern' : 'high_concern',
          summary: result.aiAnalysis.summary,
          concerns: result.aiAnalysis.concerns,
          ingredients: result.aiAnalysis.keyIngredients,
          saferAlternatives: result.aiAnalysis.alternatives,
          sources: result.aiAnalysis.sources || [],
          tobinScore: result.aiAnalysis.tobinScore ?? undefined,
          glycogenicIndex: result.aiAnalysis.glycogenicIndex ?? undefined,
          lipogenicIndex: result.aiAnalysis.lipogenicIndex ?? undefined,
          nutriscoreGrade: result.aiAnalysis.nutriscoreGrade ?? undefined,
          novaGroup: result.aiAnalysis.novaGroup ?? undefined,
          halal: result.aiAnalysis.halal ?? undefined,
          kosher: result.aiAnalysis.kosher ?? undefined,
          organic: result.aiAnalysis.organic ?? undefined,
          certificationNotes: result.aiAnalysis.certificationNotes ?? undefined,
        } : undefined,
        dietaryClassification: result.aiAnalysis && (result.aiAnalysis.halal || result.aiAnalysis.kosher || result.aiAnalysis.organic) ? {
          halal: result.aiAnalysis.halal ?? undefined,
          kosher: result.aiAnalysis.kosher ?? undefined,
          organic: result.aiAnalysis.organic ?? undefined,
          certificationNotes: result.aiAnalysis.certificationNotes ?? undefined,
        } : undefined,
        source: result.aiGenerated ? 'ai' : 'openfoodfacts',
        apiSourceString: result.source || undefined,
        isAiGenerated: result.aiGenerated || false,
      };
      if (user) {
        await addToHistory(code, result.product.name, detectedProductType, code, result.product.imageUrl, productDetails);
      }
    } else {
      setNotFound(true);
      setNotFoundError(result.error || null);
    }
  };

  const handleScan = (code: string) => {
    lookupBarcode(code);
    setShowScanner(false);
  };

  const handleManualLookup = () => {
    if (manualCode.trim()) {
      lookupBarcode(manualCode.trim());
    }
  };

  const handleRescan = (barcode: string) => {
    setManualCode(barcode);
    lookupBarcode(barcode);
  };

  const handleAddToList = async () => {
    if (localProduct) {
      await addItem(localProduct.id, localProduct.type);
      if (user) {
        await addToHistory(localProduct.barcode, localProduct.name, localProduct.type, localProduct.id, null);
      }
      setLocalProduct(null);
      setOpenFoodFactsProduct(null);
      setProductSource(null);
      setManualCode('');
    } else if (openFoodFactsProduct) {
      if (user) {
        const productDetails: ScanProductDetails = {
          brand: openFoodFactsProduct.brand || undefined,
          categories: openFoodFactsProduct.categories || undefined,
          ingredients: openFoodFactsProduct.ingredients || undefined,
          imageUrl: openFoodFactsProduct.imageUrl,
          nutrition: openFoodFactsProduct.nutrition ?? undefined,
          nutriscoreGrade: openFoodFactsProduct.nutriscoreGrade ?? undefined,
          novaGroup: openFoodFactsProduct.novaGroup ?? undefined,
          aiAnalysis: aiAnalysis ? {
            safetyLevel: aiAnalysis.safetyRating === 'Safe' ? 'safe' : aiAnalysis.safetyRating === 'Moderate Concern' ? 'moderate_concern' : 'high_concern',
            summary: aiAnalysis.summary,
            concerns: aiAnalysis.concerns,
            ingredients: aiAnalysis.keyIngredients,
            saferAlternatives: aiAnalysis.alternatives,
            sources: aiAnalysis.sources || [],
            tobinScore: aiAnalysis.tobinScore ?? undefined,
            glycogenicIndex: aiAnalysis.glycogenicIndex ?? undefined,
            lipogenicIndex: aiAnalysis.lipogenicIndex ?? undefined,
            nutriscoreGrade: aiAnalysis.nutriscoreGrade ?? undefined,
            novaGroup: aiAnalysis.novaGroup ?? undefined,
            halal: aiAnalysis.halal ?? undefined,
            kosher: aiAnalysis.kosher ?? undefined,
            organic: aiAnalysis.organic ?? undefined,
            certificationNotes: aiAnalysis.certificationNotes ?? undefined,
          } : undefined,
          dietaryClassification: aiAnalysis && (aiAnalysis.halal || aiAnalysis.kosher || aiAnalysis.organic) ? {
            halal: aiAnalysis.halal ?? undefined,
            kosher: aiAnalysis.kosher ?? undefined,
            organic: aiAnalysis.organic ?? undefined,
            certificationNotes: aiAnalysis.certificationNotes ?? undefined,
          } : undefined,
          source: isAiGenerated ? 'ai' : 'openfoodfacts',
          apiSourceString: apiSourceString ?? undefined,
          isAiGenerated: isAiGenerated || false,
        };
        await addToHistory(currentBarcode, openFoodFactsProduct.name, productType, currentBarcode, openFoodFactsProduct.imageUrl, productDetails);
      }
      setOpenFoodFactsProduct(null);
      setProductSource(null);
      setManualCode('');
    }
  };

  const getProductEmoji = (productType: 'food' | 'cosmetic') => {
    return productType === 'food' ? '🍽️' : '💄';
  };

  // Premium required for scanning
  if (!subscriptionLoading && !subscribed) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50">
        <div className="container mx-auto px-4 py-8 max-w-2xl">
          <div className="flex items-center justify-between mb-8">
            <Link to="/shopping-list" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
              <ArrowLeft className="h-4 w-4" />
              Back to Shopping List
            </Link>
            <UserMenu />
          </div>
          <Card className="border-2 border-primary/20 shadow-lg">
            <CardHeader className="text-center">
              <div className="mx-auto mb-4 w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center">
                <Crown className="h-8 w-8 text-primary" />
              </div>
              <CardTitle className="text-xl">Scanning is a Premium feature</CardTitle>
              <CardDescription className="text-base">
                Upgrade to Premium to scan barcodes and analyze products. Start with a 7-day free trial.
              </CardDescription>
            </CardHeader>
            <CardContent className="flex flex-col gap-3">
              <Button asChild className="w-full">
                <Link to="/subscription">Upgrade to Premium</Link>
              </Button>
              <Button asChild variant="outline" className="w-full">
                <Link to="/">Back to Home</Link>
              </Button>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50">
      <div className="container mx-auto px-4 py-8 max-w-2xl">
        <div className="flex items-center justify-between mb-8">
          <Link to="/shopping-list" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
            <ArrowLeft className="h-4 w-4" />
            Back to Shopping List
          </Link>
          <UserMenu />
        </div>

        <div className="text-center mb-8">
          <div className="inline-flex items-center justify-center w-16 h-16 bg-primary/10 rounded-full mb-4">
            <ScanLine className="h-8 w-8 text-primary" />
          </div>
          <h1 className="text-3xl font-bold text-foreground mb-2">Barcode Scanner</h1>
          <p className="text-muted-foreground">
            Scan product barcodes to quickly add items to your shopping list
          </p>
        </div>

        <div className="space-y-6">
          {/* Camera Scanner */}
          {showScanner ? (
            <WebBarcodeScanner 
              onScan={handleScan} 
              onClose={() => setShowScanner(false)} 
            />
          ) : (
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Camera className="h-5 w-5" />
                  Camera Scanner
                </CardTitle>
                <CardDescription>
                  Scan barcodes using your device camera
                </CardDescription>
              </CardHeader>
              <CardContent>
                <Button 
                  className="w-full gap-2" 
                  onClick={() => setShowScanner(true)}
                >
                  <Camera className="h-4 w-4" />
                  Open Camera Scanner
                </Button>
              </CardContent>
            </Card>
          )}

          {/* Manual Entry */}
          <Card>
            <CardHeader>
              <CardTitle>Manual Entry</CardTitle>
              <CardDescription>
                Enter a barcode number manually
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="space-y-2">
                <Label htmlFor="barcode">Barcode Number</Label>
                <div className="flex gap-2">
                  <Input
                    id="barcode"
                    placeholder="Enter barcode (e.g., 0123456789012)"
                    value={manualCode}
                    onChange={(e) => setManualCode(e.target.value)}
                    onKeyDown={(e) => e.key === 'Enter' && handleManualLookup()}
                  />
                  <Button onClick={handleManualLookup}>Lookup</Button>
                </div>
              </div>
              <div className="text-xs text-muted-foreground">
                <p className="font-medium mb-1">Try scanning real products or these demo cosmetic barcodes:</p>
                <ul className="space-y-0.5 grid grid-cols-2 gap-x-4">
                  <li>• 3017624010701 - Nutella</li>
                  <li>• 5449000000996 - Coca-Cola</li>
                  <li>• 3456789012345 - Natural Deodorant</li>
                  <li>• 897017234567 - Mineral Sunscreen</li>
                </ul>
                <p className="mt-2 text-primary font-medium">🌐 Now powered by Open Food Facts for real product data!</p>
              </div>
            </CardContent>
          </Card>

          {/* Scanned Product */}
          <ScannedProductCard
            localProduct={localProduct}
            openFoodFactsProduct={openFoodFactsProduct}
            source={productSource}
            apiSourceString={apiSourceString}
            isLoading={apiLoading}
            user={user}
            onAddToList={handleAddToList}
            currentBarcode={currentBarcode}
            isAiGenerated={isAiGenerated}
            aiAnalysis={aiAnalysis}
            productType={productType}
            onSaveAiProduct={async () => {
              if (!openFoodFactsProduct || !currentBarcode || !user) return;
              
              const success = await addCommunityProduct({
                barcode: currentBarcode,
                product_name: openFoodFactsProduct.name,
                brand: openFoodFactsProduct.brand || undefined,
                product_type: productType,
                categories: openFoodFactsProduct.categories || undefined,
                ingredients: openFoodFactsProduct.ingredients || undefined,
              });
              
              if (success) {
                setIsAiGenerated(false);
                setProductSource('openfoodfacts');
                toast.success('Product saved to community database!');
              }
            }}
          />

          {notFound && !apiLoading && (
            <>
              {showAddForm && user ? (
                <AddProductForm 
                  barcode={currentBarcode}
                  onSuccess={() => {
                    setShowAddForm(false);
                    setNotFound(false);
                    // Re-lookup the barcode to show the newly added product
                    lookupBarcode(currentBarcode);
                  }}
                  onCancel={() => setShowAddForm(false)}
                />
              ) : (
                <Card className="border-destructive/50 bg-destructive/5">
                  <CardContent className="pt-6">
                    <div className="flex items-start gap-3">
                      <AlertTriangle className="h-5 w-5 text-destructive mt-0.5" />
                      <div className="flex-1">
                        <p className="font-medium text-destructive">Product Not Found</p>
                        <p className="text-sm text-muted-foreground mb-4">
                          {notFoundError || "This barcode wasn't in our databases and couldn't be identified. You can add it below to help others."}
                        </p>
                        {user ? (
                          <Button 
                            onClick={() => setShowAddForm(true)}
                            variant="outline"
                            className="gap-2"
                          >
                            <Users className="h-4 w-4" />
                            Add This Product to Help Others
                          </Button>
                        ) : (
                          <p className="text-sm text-muted-foreground italic">
                            Sign in to contribute this product to our community database.
                          </p>
                        )}
                      </div>
                    </div>
                  </CardContent>
                </Card>
              )}
            </>
          )}

          {/* Scan History */}
          {user && (
            <Card>
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle className="flex items-center gap-2">
                    <History className="h-5 w-5" />
                    Scan History
                  </CardTitle>
                  {history.length > 0 && (
                    <Button 
                      variant="ghost" 
                      size="sm" 
                      className="text-destructive hover:text-destructive"
                      onClick={clearHistory}
                    >
                      <Trash2 className="h-4 w-4 mr-1" />
                      Clear All
                    </Button>
                  )}
                </div>
                <CardDescription>
                  Quickly re-scan previously scanned products. Use &quot;Clear All&quot; to remove all history data. Food items appear on your Food dashboard; cosmetics on Cosmetics.
                </CardDescription>
              </CardHeader>
              <CardContent>
                {historyLoading ? (
                  <div className="text-center py-4 text-muted-foreground">
                    Loading history...
                  </div>
                ) : history.length === 0 ? (
                  <div className="text-center py-4 text-muted-foreground">
                    <Clock className="h-8 w-8 mx-auto mb-2 opacity-50" />
                    <p>No scan history yet</p>
                    <p className="text-sm">Products you scan will appear here</p>
                  </div>
                ) : (
                  <ScrollArea className="h-[300px]">
                    <div className="space-y-2">
                      {history.map((item) => (
                        <div 
                          key={item.id}
                          className="flex items-center justify-between p-3 rounded-lg bg-muted/50 hover:bg-muted transition-colors"
                        >
                          <div className="flex items-center gap-3">
                            <img 
                              src={item.image_url || (item.product_type === 'cosmetic' ? cosmeticPlaceholder : foodPlaceholder)} 
                              alt={item.product_name}
                              className="w-12 h-12 object-cover rounded-lg bg-muted"
                            />
                            <div>
                              <p className="font-medium">{item.product_name}</p>
                              <div className="flex items-center gap-2 text-xs text-muted-foreground">
                                <Badge variant="outline" className="text-xs capitalize">
                                  {item.product_type}
                                </Badge>
                                <span>•</span>
                                <span>{formatDistanceToNow(new Date(item.scanned_at), { addSuffix: true })}</span>
                              </div>
                            </div>
                          </div>
                          <div className="flex items-center gap-1">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={() => handleRescan(item.barcode)}
                              title="Scan again"
                            >
                              <RotateCcw className="h-4 w-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={async () => {
                                const wasFav = isFavorite(item.barcode, item.product_type);
                                const inList = isInList(item.barcode, item.product_type);
                                const ok = await deleteFromHistory(item.id);
                                if (ok && wasFav) await removeFavorite(item.barcode, item.product_type);
                                if (ok && inList) await removeItem(item.barcode, item.product_type);
                              }}
                              title="Remove from history (also removes from Favorites and To Buy)"
                              className="text-muted-foreground hover:text-destructive"
                            >
                              <Trash2 className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      ))}
                    </div>
                  </ScrollArea>
                )}
              </CardContent>
            </Card>
          )}

        </div>
      </div>
    </div>
  );
};

export default BarcodeScanner;
