import { Link } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { 
  ArrowLeft, 
  FlaskConical, 
  AlertTriangle, 
  Shield, 
  ExternalLink,
  Baby,
  Brain,
  Heart,
  Dna,
  Droplets,
  Sparkles,
  BookOpen,
  Leaf
} from "lucide-react";

const ChemicalHazards = () => {
  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-b from-destructive/10 to-background py-8 px-4">
        <div className="max-w-4xl mx-auto">
          <Link to="/cosmetics">
            <Button variant="ghost" size="sm" className="mb-4 gap-2">
              <ArrowLeft className="h-4 w-4" />
              Back to Cosmetics
            </Button>
          </Link>
          
          <div className="flex items-center gap-3 mb-4">
            <FlaskConical className="h-8 w-8 text-destructive" />
            <h1 className="text-3xl font-display font-bold">Chemical Hazards Guide</h1>
          </div>
          <p className="text-muted-foreground max-w-2xl">
            Understanding endocrine-disrupting chemicals (EDCs) found in everyday products. 
            Learn about xenoestrogens, parabens, phthalates, and their documented health effects 
            based on CDC biomonitoring data and peer-reviewed research.
          </p>
        </div>
      </div>
      
      <main className="max-w-4xl mx-auto px-4 pb-16 space-y-12">
        {/* Phytoestrogen Scanner Link */}
        <Card className="bg-gradient-to-r from-category-warning/10 to-category-danger/10 border-category-warning">
          <CardContent className="flex flex-col sm:flex-row items-center justify-between gap-4 py-4">
            <div className="flex items-center gap-3">
              <Leaf className="h-8 w-8 text-category-warning" />
              <div>
                <h3 className="font-semibold">Phytoestrogen Scanner</h3>
                <p className="text-sm text-muted-foreground">Identify plant-based estrogen mimics in foods</p>
              </div>
            </div>
            <Link to="/phytoestrogen-scanner">
              <Button className="gap-2">
                <Leaf className="h-4 w-4" />
                Scan Foods
              </Button>
            </Link>
          </CardContent>
        </Card>
        
        {/* Quick Navigation */}
        <nav className="flex flex-wrap gap-2 py-4">
          <a href="#xenoestrogens">
            <Badge variant="outline" className="cursor-pointer hover:bg-muted">Xenoestrogens</Badge>
          </a>
          <a href="#parabens">
            <Badge variant="outline" className="cursor-pointer hover:bg-muted">Parabens</Badge>
          </a>
          <a href="#phthalates">
            <Badge variant="outline" className="cursor-pointer hover:bg-muted">Phthalates</Badge>
          </a>
          <a href="#bisphenols">
            <Badge variant="outline" className="cursor-pointer hover:bg-muted">Bisphenols</Badge>
          </a>
          <a href="#pfas">
            <Badge variant="outline" className="cursor-pointer hover:bg-muted">PFAS</Badge>
          </a>
          <a href="#protection">
            <Badge variant="outline" className="cursor-pointer hover:bg-muted">How to Protect Yourself</Badge>
          </a>
          <Link to="/phytoestrogen-scanner">
            <Badge variant="default" className="cursor-pointer bg-category-warning hover:bg-category-warning/80">Phytoestrogens</Badge>
          </Link>
        </nav>
        
        {/* What are EDCs */}
        <section>
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-category-warning" />
                What are Endocrine-Disrupting Chemicals?
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <p>
                Endocrine-disrupting chemicals (EDCs) are substances that interfere with the body's 
                hormonal system. They can mimic, block, or alter the production of hormones like 
                estrogen, testosterone, and thyroid hormones—even at very low doses.
              </p>
              <div className="grid sm:grid-cols-3 gap-4">
                <div className="bg-muted/50 rounded-lg p-4 text-center">
                  <Dna className="h-8 w-8 mx-auto mb-2 text-primary" />
                  <p className="font-medium">Hormone Mimicry</p>
                  <p className="text-sm text-muted-foreground">Mimic natural hormones</p>
                </div>
                <div className="bg-muted/50 rounded-lg p-4 text-center">
                  <Shield className="h-8 w-8 mx-auto mb-2 text-category-danger" />
                  <p className="font-medium">Receptor Blocking</p>
                  <p className="text-sm text-muted-foreground">Block hormone receptors</p>
                </div>
                <div className="bg-muted/50 rounded-lg p-4 text-center">
                  <FlaskConical className="h-8 w-8 mx-auto mb-2 text-category-warning" />
                  <p className="font-medium">Synthesis Disruption</p>
                  <p className="text-sm text-muted-foreground">Alter hormone production</p>
                </div>
              </div>
              <div className="bg-destructive/10 border border-destructive/30 rounded-lg p-4">
                <p className="text-sm">
                  <strong>CDC Finding:</strong> The CDC's Fourth National Report on Human Exposure 
                  to Environmental Chemicals detected measurable levels of hundreds of environmental 
                  chemicals in the U.S. population, including many EDCs.
                </p>
              </div>
            </CardContent>
          </Card>
        </section>
        
        {/* Xenoestrogens */}
        <section id="xenoestrogens">
          <Card className="border-l-4 border-l-category-danger">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Droplets className="h-5 w-5 text-category-danger" />
                Xenoestrogens
              </CardTitle>
              <p className="text-sm text-muted-foreground">
                Foreign estrogen-mimicking compounds that bind to estrogen receptors
              </p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div>
                <h4 className="font-semibold mb-2">What They Are</h4>
                <p className="text-muted-foreground">
                  Xenoestrogens are synthetic or natural chemicals that imitate estrogen in the body. 
                  Unlike natural estrogens, they are not easily broken down and can accumulate in 
                  fat tissue, leading to prolonged exposure and effects.
                </p>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Common Sources</h4>
                <div className="flex flex-wrap gap-2">
                  {["Plastics (BPA, BPS)", "Pesticides", "Parabens", "Phthalates", "Synthetic fragrances", 
                    "UV filters (oxybenzone)", "Triclosan", "Food packaging"].map((source) => (
                    <Badge key={source} variant="outline">{source}</Badge>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <Heart className="h-4 w-4 text-destructive" />
                  Health Effects
                </h4>
                <ul className="space-y-2 text-sm">
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Reproductive:</strong> Early puberty, reduced fertility, endometriosis, polycystic ovary syndrome (PCOS)</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Cancer:</strong> Increased risk of hormone-sensitive cancers (breast, ovarian, prostate)</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Metabolic:</strong> Weight gain, insulin resistance, thyroid disruption</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Developmental:</strong> Altered brain development, behavioral changes in children</span>
                  </li>
                </ul>
              </div>
              
              <div className="bg-muted/50 rounded-lg p-4">
                <p className="text-sm">
                  <BookOpen className="h-4 w-4 inline mr-1" />
                  <strong>Research Note:</strong> A 2023 study in <em>Environmental Health Perspectives</em> 
                  found that xenoestrogen exposure is associated with earlier onset of puberty in girls, 
                  with effects observed at exposure levels common in the U.S. population.
                </p>
              </div>
            </CardContent>
          </Card>
        </section>
        
        {/* Parabens */}
        <section id="parabens">
          <Card className="border-l-4 border-l-category-warning">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Sparkles className="h-5 w-5 text-category-warning" />
                Parabens
              </CardTitle>
              <p className="text-sm text-muted-foreground">
                Preservatives with estrogen-mimicking properties
              </p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div>
                <h4 className="font-semibold mb-2">What They Are</h4>
                <p className="text-muted-foreground">
                  Parabens are synthetic preservatives used to prevent bacterial and fungal growth 
                  in cosmetics, pharmaceuticals, and food. They are absorbed through the skin and 
                  have been detected in human tissues, including breast tumors.
                </p>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Types to Watch For</h4>
                <div className="grid sm:grid-cols-2 gap-3">
                  {[
                    { name: "Methylparaben", potency: "Low", use: "Most common" },
                    { name: "Ethylparaben", potency: "Low", use: "Common" },
                    { name: "Propylparaben", potency: "Moderate", use: "Common" },
                    { name: "Butylparaben", potency: "Higher", use: "More restricted" },
                    { name: "Isobutylparaben", potency: "Higher", use: "Less common" },
                    { name: "Benzylparaben", potency: "Highest", use: "Rare" },
                  ].map((paraben) => (
                    <div key={paraben.name} className="bg-muted/50 rounded-lg p-3">
                      <p className="font-medium text-sm">{paraben.name}</p>
                      <p className="text-xs text-muted-foreground">
                        Potency: {paraben.potency} | {paraben.use}
                      </p>
                    </div>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Common Products</h4>
                <div className="flex flex-wrap gap-2">
                  {["Shampoo", "Conditioner", "Body lotion", "Sunscreen", "Makeup", "Shaving products", 
                    "Toothpaste", "Deodorant", "Pharmaceuticals"].map((product) => (
                    <Badge key={product} variant="secondary">{product}</Badge>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <Heart className="h-4 w-4 text-destructive" />
                  Health Concerns
                </h4>
                <ul className="space-y-2 text-sm">
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Parabens have been detected in 99% of breast cancer tissue samples in some studies</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Associated with decreased sperm quality and testosterone levels in men</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>May increase skin aging when combined with UV exposure</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Prenatal exposure linked to behavioral issues and obesity in children</span>
                  </li>
                </ul>
              </div>
              
              <div className="bg-primary/10 border border-primary/30 rounded-lg p-4">
                <p className="text-sm">
                  <strong>CDC Biomonitoring:</strong> The CDC has detected parabens in the urine of 
                  over 90% of Americans tested. Methylparaben and propylparaben are the most 
                  commonly detected, with higher concentrations in women than men.
                </p>
              </div>
            </CardContent>
          </Card>
        </section>
        
        {/* Phthalates */}
        <section id="phthalates">
          <Card className="border-l-4 border-l-category-danger">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <FlaskConical className="h-5 w-5 text-category-danger" />
                Phthalates
              </CardTitle>
              <p className="text-sm text-muted-foreground">
                Plasticizers and fragrance carriers linked to reproductive harm
              </p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div>
                <h4 className="font-semibold mb-2">What They Are</h4>
                <p className="text-muted-foreground">
                  Phthalates are a group of chemicals used to make plastics flexible and to help 
                  fragrances last longer. They are not chemically bound to products, so they 
                  continuously release into the air, food, and skin.
                </p>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Key Phthalates</h4>
                <div className="space-y-3">
                  {[
                    { abbr: "DEHP", name: "Di(2-ethylhexyl) phthalate", use: "PVC plastics, medical tubing", concern: "Reproductive toxicity" },
                    { abbr: "DBP", name: "Dibutyl phthalate", use: "Nail polish, adhesives", concern: "Male reproductive development" },
                    { abbr: "DEP", name: "Diethyl phthalate", use: "Fragrances, personal care", concern: "Endocrine disruption" },
                    { abbr: "BBP", name: "Benzyl butyl phthalate", use: "Vinyl flooring, adhesives", concern: "Developmental toxicity" },
                  ].map((phthalate) => (
                    <div key={phthalate.abbr} className="bg-muted/50 rounded-lg p-3">
                      <div className="flex items-center gap-2 mb-1">
                        <Badge variant="destructive" className="text-xs">{phthalate.abbr}</Badge>
                        <span className="font-medium text-sm">{phthalate.name}</span>
                      </div>
                      <p className="text-xs text-muted-foreground">
                        <strong>Found in:</strong> {phthalate.use}<br />
                        <strong>Primary concern:</strong> {phthalate.concern}
                      </p>
                    </div>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <Baby className="h-4 w-4 text-primary" />
                  Reproductive & Developmental Effects
                </h4>
                <ul className="space-y-2 text-sm">
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Males:</strong> Decreased testosterone, reduced sperm count, genital abnormalities in infants</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Females:</strong> Early breast development, endometriosis, reduced fertility</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Children:</strong> ADHD, behavioral problems, asthma, allergies</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-danger mt-1">•</span>
                    <span><strong>Pregnancy:</strong> Preterm birth, altered brain development</span>
                  </li>
                </ul>
              </div>
              
              <div className="bg-destructive/10 border border-destructive/30 rounded-lg p-4">
                <p className="text-sm">
                  <strong>CDC Finding:</strong> Phthalate metabolites are found in the urine of the 
                  majority of Americans tested. MEHP, a metabolite of DEHP, was found in 75% of 
                  the population. Women of reproductive age have higher levels than men.
                </p>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Hidden Sources</h4>
                <p className="text-sm text-muted-foreground mb-2">
                  Phthalates are often not listed on labels. Look for these clues:
                </p>
                <div className="flex flex-wrap gap-2">
                  <Badge variant="outline" className="text-category-danger border-category-danger">
                    "Fragrance" or "Parfum"
                  </Badge>
                  <Badge variant="outline">Vinyl/PVC products</Badge>
                  <Badge variant="outline">Soft plastics</Badge>
                  <Badge variant="outline">Time-release medications</Badge>
                  <Badge variant="outline">Scented candles</Badge>
                </div>
              </div>
            </CardContent>
          </Card>
        </section>
        
        {/* Bisphenols */}
        <section id="bisphenols">
          <Card className="border-l-4 border-l-category-warning">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <AlertTriangle className="h-5 w-5 text-category-warning" />
                Bisphenols (BPA, BPS, BPF)
              </CardTitle>
              <p className="text-sm text-muted-foreground">
                Industrial chemicals with strong estrogenic activity
              </p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div>
                <h4 className="font-semibold mb-2">What They Are</h4>
                <p className="text-muted-foreground">
                  Bisphenol A (BPA) and its replacements (BPS, BPF) are used to make polycarbonate 
                  plastics and epoxy resins. Despite "BPA-free" labeling, replacement bisphenols 
                  often have similar or worse hormonal effects.
                </p>
              </div>
              
              <div className="grid sm:grid-cols-3 gap-4">
                <div className="bg-destructive/10 rounded-lg p-4 text-center">
                  <p className="font-bold text-lg">BPA</p>
                  <p className="text-xs text-muted-foreground">Original compound</p>
                  <Badge variant="destructive" className="mt-2">High concern</Badge>
                </div>
                <div className="bg-category-warning/10 rounded-lg p-4 text-center">
                  <p className="font-bold text-lg">BPS</p>
                  <p className="text-xs text-muted-foreground">"BPA-free" replacement</p>
                  <Badge className="bg-category-warning mt-2">Similar effects</Badge>
                </div>
                <div className="bg-category-warning/10 rounded-lg p-4 text-center">
                  <p className="font-bold text-lg">BPF</p>
                  <p className="text-xs text-muted-foreground">Another replacement</p>
                  <Badge className="bg-category-warning mt-2">Similar effects</Badge>
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Common Sources</h4>
                <div className="flex flex-wrap gap-2">
                  {["Canned food linings", "Thermal receipt paper", "Water bottles", "Food containers", 
                    "Dental sealants", "Medical devices", "Electronics"].map((source) => (
                    <Badge key={source} variant="secondary">{source}</Badge>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <Brain className="h-4 w-4 text-primary" />
                  Health Effects
                </h4>
                <ul className="space-y-2 text-sm">
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Cardiovascular disease and hypertension</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Type 2 diabetes and obesity</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Reproductive disorders in both sexes</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Neurodevelopmental effects in children</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-category-warning mt-1">•</span>
                    <span>Immune system dysfunction</span>
                  </li>
                </ul>
              </div>
              
              <div className="bg-primary/10 border border-primary/30 rounded-lg p-4">
                <p className="text-sm">
                  <strong>CDC Biomonitoring:</strong> BPA was detected in 93% of urine samples 
                  from people ages 6 and older. Cashiers who handle thermal receipts have 
                  significantly higher BPA levels than other workers.
                </p>
              </div>
            </CardContent>
          </Card>
        </section>
        
        {/* PFAS */}
        <section id="pfas">
          <Card className="border-l-4 border-l-destructive">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Shield className="h-5 w-5 text-destructive" />
                PFAS (Forever Chemicals)
              </CardTitle>
              <p className="text-sm text-muted-foreground">
                Per- and polyfluoroalkyl substances that never break down
              </p>
            </CardHeader>
            <CardContent className="space-y-6">
              <div>
                <h4 className="font-semibold mb-2">What They Are</h4>
                <p className="text-muted-foreground">
                  PFAS are a class of over 9,000 synthetic chemicals used for their water-, 
                  grease-, and stain-resistant properties. They are called "forever chemicals" 
                  because they don't break down in the environment or human body.
                </p>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2">Where They're Found</h4>
                <div className="grid sm:grid-cols-2 gap-2">
                  {[
                    "Non-stick cookware (Teflon)",
                    "Waterproof clothing/shoes",
                    "Stain-resistant fabrics",
                    "Food packaging (microwave popcorn bags)",
                    "Cosmetics (foundation, mascara)",
                    "Dental floss",
                    "Firefighting foam",
                    "Drinking water (contaminated areas)"
                  ].map((source) => (
                    <div key={source} className="flex items-center gap-2 text-sm">
                      <span className="text-destructive">•</span>
                      {source}
                    </div>
                  ))}
                </div>
              </div>
              
              <div>
                <h4 className="font-semibold mb-2 flex items-center gap-2">
                  <Heart className="h-4 w-4 text-destructive" />
                  Health Effects
                </h4>
                <ul className="space-y-2 text-sm">
                  <li className="flex items-start gap-2">
                    <span className="text-destructive mt-1">•</span>
                    <span><strong>Cancer:</strong> Kidney cancer, testicular cancer</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-destructive mt-1">•</span>
                    <span><strong>Thyroid:</strong> Thyroid disease and dysfunction</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-destructive mt-1">•</span>
                    <span><strong>Immune:</strong> Reduced vaccine response, immune suppression</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-destructive mt-1">•</span>
                    <span><strong>Reproductive:</strong> Pregnancy complications, developmental delays</span>
                  </li>
                  <li className="flex items-start gap-2">
                    <span className="text-destructive mt-1">•</span>
                    <span><strong>Liver:</strong> Liver damage, elevated cholesterol</span>
                  </li>
                </ul>
              </div>
              
              <div className="bg-destructive/10 border border-destructive/30 rounded-lg p-4">
                <p className="text-sm">
                  <strong>CDC Finding:</strong> PFAS have been detected in the blood of 97% of 
                  Americans. The chemicals accumulate over time, with a half-life of 3-8 years 
                  in the human body.
                </p>
              </div>
            </CardContent>
          </Card>
        </section>
        
        {/* Protection Section */}
        <section id="protection">
          <Card className="border-2 border-category-safe">
            <CardHeader>
              <CardTitle className="flex items-center gap-2 text-category-safe">
                <Shield className="h-5 w-5" />
                How to Protect Yourself
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="grid sm:grid-cols-2 gap-6">
                <div>
                  <h4 className="font-semibold mb-3">Personal Care</h4>
                  <ul className="space-y-2 text-sm">
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Choose "fragrance-free" products
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Look for paraben-free formulations
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Use mineral sunscreens (zinc oxide)
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Check EWG's Skin Deep database
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Use natural/organic cosmetics
                    </li>
                  </ul>
                </div>
                
                <div>
                  <h4 className="font-semibold mb-3">Food & Kitchen</h4>
                  <ul className="space-y-2 text-sm">
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Avoid plastic containers for hot food
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Use glass or stainless steel storage
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Choose fresh over canned foods
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Avoid non-stick cookware
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Filter drinking water
                    </li>
                  </ul>
                </div>
                
                <div>
                  <h4 className="font-semibold mb-3">Home Environment</h4>
                  <ul className="space-y-2 text-sm">
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Vacuum regularly (dust contains chemicals)
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Use unscented cleaning products
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Avoid air fresheners and scented candles
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Choose natural fiber furniture
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Ventilate when using chemical products
                    </li>
                  </ul>
                </div>
                
                <div>
                  <h4 className="font-semibold mb-3">Shopping Tips</h4>
                  <ul className="space-y-2 text-sm">
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Decline thermal receipts when possible
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Read ingredient labels carefully
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Buy organic when budget allows
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Avoid "anti-bacterial" products
                    </li>
                    <li className="flex items-start gap-2">
                      <span className="text-category-safe">✓</span>
                      Choose products with fewer ingredients
                    </li>
                  </ul>
                </div>
              </div>
            </CardContent>
          </Card>
        </section>
        
        <Separator />
        
        {/* References */}
        <section>
          <h3 className="font-semibold mb-4 flex items-center gap-2">
            <BookOpen className="h-5 w-5" />
            References & Resources
          </h3>
          <div className="space-y-3">
            <a 
              href="https://www.cdc.gov/exposurereport/" 
              target="_blank" 
              rel="noopener noreferrer"
              className="flex items-center gap-2 text-sm text-primary hover:underline"
            >
              <ExternalLink className="h-4 w-4" />
              CDC Fourth National Report on Human Exposure to Environmental Chemicals
            </a>
            <a 
              href="https://www.niehs.nih.gov/health/topics/agents/endocrine" 
              target="_blank" 
              rel="noopener noreferrer"
              className="flex items-center gap-2 text-sm text-primary hover:underline"
            >
              <ExternalLink className="h-4 w-4" />
              NIH National Institute of Environmental Health Sciences - Endocrine Disruptors
            </a>
            <a 
              href="https://www.ewg.org/skindeep/" 
              target="_blank" 
              rel="noopener noreferrer"
              className="flex items-center gap-2 text-sm text-primary hover:underline"
            >
              <ExternalLink className="h-4 w-4" />
              EWG Skin Deep Cosmetics Database
            </a>
            <a 
              href="https://www.fda.gov/cosmetics/cosmetic-ingredients/parabens-cosmetics" 
              target="_blank" 
              rel="noopener noreferrer"
              className="flex items-center gap-2 text-sm text-primary hover:underline"
            >
              <ExternalLink className="h-4 w-4" />
              FDA - Parabens in Cosmetics
            </a>
            <a 
              href="https://www.epa.gov/pfas" 
              target="_blank" 
              rel="noopener noreferrer"
              className="flex items-center gap-2 text-sm text-primary hover:underline"
            >
              <ExternalLink className="h-4 w-4" />
              EPA - PFAS Information
            </a>
          </div>
        </section>
        
        {/* Call to Action */}
        <div className="flex flex-wrap gap-4 justify-center pt-8">
          <Link to="/cosmetics">
            <Button size="lg" className="gap-2">
              <Sparkles className="h-4 w-4" />
              Browse Safer Cosmetics
            </Button>
          </Link>
          <Link to="/">
            <Button variant="outline" size="lg" className="gap-2">
              <ArrowLeft className="h-4 w-4" />
              Food Database
            </Button>
          </Link>
        </div>
      </main>
    </div>
  );
};

export default ChemicalHazards;
