import { useState, useEffect } from 'react';
import { Link } from 'react-router-dom';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Textarea } from '@/components/ui/textarea';
import { Input } from '@/components/ui/input';
import { Avatar, AvatarFallback } from '@/components/ui/avatar';
import { Collapsible, CollapsibleContent, CollapsibleTrigger } from '@/components/ui/collapsible';
import { 
  ArrowLeft, 
  Users, 
  MessageSquare, 
  Heart,
  Send,
  Lightbulb,
  Trophy,
  Loader2,
  Sparkles,
  Trash2,
  ChevronDown,
  ChevronUp,
  Reply
} from 'lucide-react';
import { useAuth } from '@/hooks/useAuth';
import { useCommunity, CommunityPost, PostReply } from '@/hooks/useCommunity';
import { SocialShare } from '@/components/SocialShare';
import { Footer } from '@/components/Footer';
import { CommunityLeaderboard } from '@/components/CommunityLeaderboard';
import { FollowingFeed } from '@/components/FollowingFeed';

const categoryIcons = {
  tip: Lightbulb,
  question: MessageSquare,
  achievement: Trophy,
  discovery: Sparkles,
};

const categoryColors = {
  tip: 'bg-green-100 text-green-700 border-green-300 dark:bg-green-950/30 dark:text-green-400',
  question: 'bg-blue-100 text-blue-700 border-blue-300 dark:bg-blue-950/30 dark:text-blue-400',
  achievement: 'bg-amber-100 text-amber-700 border-amber-300 dark:bg-amber-950/30 dark:text-amber-400',
  discovery: 'bg-purple-100 text-purple-700 border-purple-300 dark:bg-purple-950/30 dark:text-purple-400',
};

// Reply Section Component
const ReplySection = ({ 
  postId, 
  repliesCount,
  fetchReplies,
  createReply,
  deleteReply,
  formatTimeAgo,
  userId,
  subscribeToReplies
}: {
  postId: string;
  repliesCount: number;
  fetchReplies: (postId: string) => Promise<PostReply[]>;
  createReply: (postId: string, content: string) => Promise<boolean>;
  deleteReply: (replyId: string, postId: string) => Promise<boolean>;
  formatTimeAgo: (date: string) => string;
  userId?: string;
  subscribeToReplies: (postId: string, callback: (replies: PostReply[], eventType: string) => void) => () => void;
}) => {
  const [isOpen, setIsOpen] = useState(false);
  const [replies, setReplies] = useState<PostReply[]>([]);
  const [newReply, setNewReply] = useState('');
  const [loadingReplies, setLoadingReplies] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);

  // Subscribe to real-time reply updates when section is open
  useEffect(() => {
    if (!isOpen) return;

    // Initial load
    if (replies.length === 0) {
      loadReplies();
    }

    // Subscribe to real-time updates
    const unsubscribe = subscribeToReplies(postId, (updatedReplies, eventType) => {
      console.log('Real-time reply update:', eventType);
      setReplies(updatedReplies);
    });

    return () => {
      unsubscribe();
    };
  }, [isOpen, postId]);

  const loadReplies = async () => {
    setLoadingReplies(true);
    const data = await fetchReplies(postId);
    setReplies(data);
    setLoadingReplies(false);
  };

  const handleSubmitReply = async () => {
    if (!newReply.trim()) return;
    setIsSubmitting(true);
    const success = await createReply(postId, newReply.trim());
    if (success) {
      setNewReply('');
      // Real-time subscription will update replies automatically
    }
    setIsSubmitting(false);
  };

  const handleDeleteReply = async (replyId: string) => {
    await deleteReply(replyId, postId);
    // Real-time subscription will update replies automatically
  };

  return (
    <Collapsible open={isOpen} onOpenChange={setIsOpen}>
      <CollapsibleTrigger asChild>
        <button className="flex items-center gap-1 text-sm text-muted-foreground hover:text-foreground transition-colors">
          <MessageSquare className="h-4 w-4" />
          {repliesCount}
          {isOpen ? <ChevronUp className="h-3 w-3" /> : <ChevronDown className="h-3 w-3" />}
        </button>
      </CollapsibleTrigger>
      
      <CollapsibleContent className="mt-4 space-y-3 pl-2 border-l-2 border-muted">
        {loadingReplies ? (
          <div className="flex items-center gap-2 py-2">
            <Loader2 className="h-4 w-4 animate-spin" />
            <span className="text-sm text-muted-foreground">Loading replies...</span>
          </div>
        ) : (
          <>
            {replies.map(reply => (
              <div key={reply.id} className="flex gap-2 py-2">
                <Avatar className="h-6 w-6">
                  <AvatarFallback className="text-xs bg-muted">
                    {reply.author_name.substring(0, 2).toUpperCase()}
                  </AvatarFallback>
                </Avatar>
                <div className="flex-1 min-w-0">
                  <div className="flex items-center gap-2">
                    <span className="text-sm font-medium">{reply.author_name}</span>
                    <span className="text-xs text-muted-foreground">
                      {formatTimeAgo(reply.created_at)}
                    </span>
                    {userId === reply.user_id && (
                      <button 
                        onClick={() => handleDeleteReply(reply.id)}
                        className="text-muted-foreground hover:text-red-500 ml-auto"
                      >
                        <Trash2 className="h-3 w-3" />
                      </button>
                    )}
                  </div>
                  <p className="text-sm text-foreground/90">{reply.content}</p>
                </div>
              </div>
            ))}
            
            {replies.length === 0 && (
              <p className="text-sm text-muted-foreground py-2">No replies yet. Be the first to reply!</p>
            )}
            
            <div className="flex gap-2 pt-2">
              <Input
                placeholder="Write a reply..."
                value={newReply}
                onChange={(e) => setNewReply(e.target.value)}
                onKeyDown={(e) => e.key === 'Enter' && !e.shiftKey && handleSubmitReply()}
                className="flex-1 h-8 text-sm"
              />
              <Button 
                size="sm" 
                onClick={handleSubmitReply}
                disabled={!newReply.trim() || isSubmitting}
                className="h-8"
              >
                {isSubmitting ? (
                  <Loader2 className="h-3 w-3 animate-spin" />
                ) : (
                  <Reply className="h-3 w-3" />
                )}
              </Button>
            </div>
          </>
        )}
      </CollapsibleContent>
    </Collapsible>
  );
};

const Community = () => {
  const { user } = useAuth();
  const { 
    posts, 
    loading, 
    stats, 
    createPost, 
    toggleLike, 
    deletePost,
    formatTimeAgo,
    fetchReplies,
    createReply,
    deleteReply,
    subscribeToReplies
  } = useCommunity();
  
  const [newPost, setNewPost] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<CommunityPost['category']>('tip');
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handlePost = async () => {
    if (!newPost.trim()) return;
    
    setIsSubmitting(true);
    const success = await createPost(newPost.trim(), selectedCategory);
    if (success) {
      setNewPost('');
    }
    setIsSubmitting(false);
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      <div className="container mx-auto px-4 py-8 max-w-4xl">
        {/* Header */}
        <div className="flex items-center gap-4 mb-8">
          <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
            <ArrowLeft className="h-4 w-4" />
            Back
          </Link>
          <div className="flex-1">
            <h1 className="text-3xl font-bold flex items-center gap-3">
              <Users className="h-8 w-8 text-primary" />
              Community
            </h1>
            <p className="text-muted-foreground">Share tips, ask questions, and celebrate wins</p>
          </div>
          <SocialShare 
            title="Join the TOBIN Health Community" 
            text="Join our community of health-conscious shoppers! Share tips and discover healthier alternatives."
          />
        </div>

        {/* Stats and Leaderboard */}
        <div className="grid md:grid-cols-2 gap-6 mb-8">
          <div className="space-y-4">
            <div className="grid grid-cols-3 gap-4">
              <Card>
                <CardContent className="p-4 text-center">
                  <Users className="h-6 w-6 mx-auto text-primary mb-1" />
                  <p className="text-2xl font-bold">{stats.members.toLocaleString()}</p>
                  <p className="text-xs text-muted-foreground">Members</p>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4 text-center">
                  <MessageSquare className="h-6 w-6 mx-auto text-blue-500 mb-1" />
                  <p className="text-2xl font-bold">{stats.posts}</p>
                  <p className="text-xs text-muted-foreground">Posts</p>
                </CardContent>
              </Card>
              <Card>
                <CardContent className="p-4 text-center">
                  <Lightbulb className="h-6 w-6 mx-auto text-amber-500 mb-1" />
                  <p className="text-2xl font-bold">{stats.tips}</p>
                  <p className="text-xs text-muted-foreground">Tips Shared</p>
                </CardContent>
              </Card>
            </div>
          </div>
          
          <div className="space-y-6">
            <CommunityLeaderboard />
            <FollowingFeed />
          </div>
        </div>

        {/* Create Post */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="text-lg">Share with the Community</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="flex gap-2 flex-wrap">
              {(['tip', 'question', 'achievement', 'discovery'] as const).map(cat => {
                const Icon = categoryIcons[cat];
                return (
                  <Button
                    key={cat}
                    variant={selectedCategory === cat ? 'default' : 'outline'}
                    size="sm"
                    onClick={() => setSelectedCategory(cat)}
                    className="gap-2"
                  >
                    <Icon className="h-4 w-4" />
                    {cat.charAt(0).toUpperCase() + cat.slice(1)}
                  </Button>
                );
              })}
            </div>
            
            <Textarea
              placeholder={
                selectedCategory === 'tip' ? 'Share a health tip...' :
                selectedCategory === 'question' ? 'Ask the community...' :
                selectedCategory === 'achievement' ? 'Celebrate a win...' :
                'Share a discovery...'
              }
              value={newPost}
              onChange={(e) => setNewPost(e.target.value)}
              rows={3}
            />
            
            <div className="flex justify-between items-center">
              <p className="text-xs text-muted-foreground">
                {user ? `Posting as ${user.email?.split('@')[0]}` : 'Sign in to post'}
              </p>
              <Button 
                onClick={handlePost} 
                disabled={!newPost.trim() || isSubmitting || !user} 
                className="gap-2"
              >
                {isSubmitting ? (
                  <Loader2 className="h-4 w-4 animate-spin" />
                ) : (
                  <Send className="h-4 w-4" />
                )}
                Post
              </Button>
            </div>
          </CardContent>
        </Card>

        {/* Posts Feed */}
        {loading ? (
          <div className="flex justify-center py-12">
            <Loader2 className="h-8 w-8 animate-spin text-primary" />
          </div>
        ) : posts.length === 0 ? (
          <Card>
            <CardContent className="p-12 text-center">
              <MessageSquare className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
              <h3 className="text-xl font-semibold mb-2">No Posts Yet</h3>
              <p className="text-muted-foreground">
                Be the first to share something with the community!
              </p>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-4">
            {posts.map((post) => {
              const CategoryIcon = categoryIcons[post.category];
              const isOwner = user?.id === post.user_id;
              
              return (
                <Card key={post.id} className="overflow-hidden">
                  <CardContent className="p-4">
                    <div className="flex gap-3">
                      <Link to={`/user/${post.user_id}`}>
                        <Avatar className="cursor-pointer hover:ring-2 hover:ring-primary/50 transition-all">
                          <AvatarFallback className="bg-primary/10 text-primary">
                            {post.author_name.substring(0, 2).toUpperCase()}
                          </AvatarFallback>
                        </Avatar>
                      </Link>
                      
                      <div className="flex-1 min-w-0">
                        <div className="flex items-center gap-2 mb-1 flex-wrap">
                          <Link 
                            to={`/user/${post.user_id}`}
                            className="font-semibold hover:text-primary hover:underline transition-colors"
                          >
                            {post.author_name}
                          </Link>
                          <Badge variant="outline" className={`${categoryColors[post.category]} text-xs`}>
                            <CategoryIcon className="h-3 w-3 mr-1" />
                            {post.category}
                          </Badge>
                          <span className="text-xs text-muted-foreground">
                            {formatTimeAgo(post.created_at)}
                          </span>
                        </div>
                        
                        <p className="text-sm mb-3 whitespace-pre-wrap">{post.content}</p>
                        
                        <div className="flex items-center gap-4 flex-wrap">
                          <button 
                            onClick={() => toggleLike(post.id)}
                            className={`flex items-center gap-1 text-sm transition-colors ${
                              post.isLiked ? 'text-red-500' : 'text-muted-foreground hover:text-red-500'
                            }`}
                          >
                            <Heart className={`h-4 w-4 ${post.isLiked ? 'fill-current' : ''}`} />
                            {post.likes_count}
                          </button>
                          
                          <ReplySection
                            postId={post.id}
                            repliesCount={post.replies_count}
                            fetchReplies={fetchReplies}
                            createReply={createReply}
                            deleteReply={deleteReply}
                            formatTimeAgo={formatTimeAgo}
                            userId={user?.id}
                            subscribeToReplies={subscribeToReplies}
                          />
                          
                          <SocialShare 
                            title="TOBIN Health Community"
                            text={post.content.substring(0, 100)}
                            variant="ghost"
                            size="sm"
                          />
                          
                          {isOwner && (
                            <button 
                              onClick={() => deletePost(post.id)}
                              className="flex items-center gap-1 text-sm text-muted-foreground hover:text-red-500 ml-auto"
                            >
                              <Trash2 className="h-4 w-4" />
                            </button>
                          )}
                        </div>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              );
            })}
          </div>
        )}
      </div>
      
      <Footer />
    </div>
  );
};

export default Community;
