import { useState, useMemo } from "react";
import { cosmeticsDatabase, CosmeticItem } from "@/data/cosmeticsDatabase";
import { CosmeticCard } from "@/components/CosmeticCard";
import { CosmeticDetailModal } from "@/components/CosmeticDetailModal";
import { CosmeticCategoryFilter } from "@/components/CosmeticCategoryFilter";
import { TobinFilter, TobinFilters, SortOption, defaultTobinFilters } from "@/components/TobinFilter";
import { CosmeticsDashboard } from "@/components/CosmeticsDashboard";
import { CosmeticsSearchBar } from "@/components/CosmeticsSearchBar";
import { UserMenu } from "@/components/UserMenu";
import { Button } from "@/components/ui/button";
import { 
  ArrowLeft, 
  AlertTriangle, 
  ShieldCheck, 
  FlaskConical,
  Sparkles,
  BookOpen,
  ScanSearch
} from "lucide-react";
import { Link } from "react-router-dom";
import { useScanHistory } from "@/hooks/useScanHistory";
import { useAuth } from "@/hooks/useAuth";

type CosmeticCategory = CosmeticItem["category"] | null;

const Cosmetics = () => {
  const { user } = useAuth();
  const { history } = useScanHistory();
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<CosmeticCategory>(null);
  const [selectedRisk, setSelectedRisk] = useState<number | null>(null);
  const [selectedCosmetic, setSelectedCosmetic] = useState<CosmeticItem | null>(null);
  const [tobinFilters, setTobinFilters] = useState<TobinFilters>(defaultTobinFilters);
  const [sortBy, setSortBy] = useState<SortOption>('none');

  // Filter scan history for cosmetic products only (for stats)
  const myCosmetics = useMemo(() => {
    return history.filter(item => item.product_type === 'cosmetic');
  }, [history]);
  
  const filteredAndSortedCosmetics = useMemo(() => {
    let items = cosmeticsDatabase.filter((item) => {
      const matchesSearch = 
        item.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
        item.commonChemicals.some(c => c.toLowerCase().includes(searchQuery.toLowerCase()));
      const matchesCategory = selectedCategory === null || item.category === selectedCategory;
      const matchesRisk = selectedRisk === null || item.riskLevel === selectedRisk;
      
      // TOBIN filters
      const tobinScore = item.tobinScore;
      const matchesTobin = !tobinScore || (
        tobinScore.total <= tobinFilters.maxTotal &&
        tobinScore.T <= tobinFilters.maxT &&
        tobinScore.O <= tobinFilters.maxO &&
        tobinScore.B <= tobinFilters.maxB &&
        tobinScore.I <= tobinFilters.maxI &&
        tobinScore.N <= tobinFilters.maxN
      );
      
      return matchesSearch && matchesCategory && matchesRisk && matchesTobin;
    });
    
    // Sorting
    if (sortBy === 'tobin-asc') {
      items = [...items].sort((a, b) => (a.tobinScore?.total ?? 50) - (b.tobinScore?.total ?? 50));
    } else if (sortBy === 'tobin-desc') {
      items = [...items].sort((a, b) => (b.tobinScore?.total ?? 0) - (a.tobinScore?.total ?? 0));
    } else if (sortBy === 'name-asc') {
      items = [...items].sort((a, b) => a.name.localeCompare(b.name));
    } else if (sortBy === 'name-desc') {
      items = [...items].sort((a, b) => b.name.localeCompare(a.name));
    }
    
    return items;
  }, [searchQuery, selectedCategory, selectedRisk, tobinFilters, sortBy]);
  
  // Group by risk level for display
  const groupedCosmetics = useMemo(() => {
    if (selectedRisk !== null || sortBy !== 'none') return null;
    
    return {
      safe: filteredAndSortedCosmetics.filter(c => c.riskLevel === 4),
      low: filteredAndSortedCosmetics.filter(c => c.riskLevel === 3),
      moderate: filteredAndSortedCosmetics.filter(c => c.riskLevel === 2),
      high: filteredAndSortedCosmetics.filter(c => c.riskLevel === 1),
    };
  }, [filteredAndSortedCosmetics, selectedRisk, sortBy]);
  
  // Stats based on user's scanned cosmetics
  const stats = useMemo(() => {
    // Count products with AI analysis safety levels
    const highRisk = myCosmetics.filter(c => 
      c.product_details?.aiAnalysis?.safetyLevel === 'high_concern'
    ).length;
    const withXenoestrogens = myCosmetics.filter(c => 
      c.product_details?.aiAnalysis?.safetyLevel === 'high_concern' || 
      c.product_details?.aiAnalysis?.safetyLevel === 'moderate_concern'
    ).length;
    const safe = myCosmetics.filter(c => 
      c.product_details?.aiAnalysis?.safetyLevel === 'safe' || 
      c.product_details?.aiAnalysis?.safetyLevel === 'low_concern'
    ).length;
    
    return {
      total: myCosmetics.length,
      highRisk,
      withXenoestrogens,
      safe,
    };
  }, [myCosmetics]);
  
  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-b from-primary/10 to-background py-8 px-4">
        <div className="max-w-7xl mx-auto">
          <div className="flex items-center justify-between mb-4">
            <Link to="/">
              <Button variant="ghost" size="sm" className="gap-2">
                <ArrowLeft className="h-4 w-4" />
                Back to Foods
              </Button>
            </Link>
            <UserMenu />
          </div>
          
          <div className="flex items-center gap-3 mb-4">
            <Sparkles className="h-8 w-8 text-primary" />
            <h1 className="text-3xl font-display font-bold">Cosmetics & Personal Care</h1>
          </div>
          <p className="text-muted-foreground max-w-2xl mb-4">
            Explore chemical hazards in personal care products. Understand xenoestrogens, 
            endocrine disruptors, and find safer alternatives based on CDC and EWG data.
          </p>
          
          <div className="flex flex-wrap gap-2">
            <Link to="/chemical-hazards">
              <Button variant="outline" size="sm" className="gap-2">
                <BookOpen className="h-4 w-4" />
                Learn About Chemical Hazards
              </Button>
            </Link>
            <Link to="/ingredient-scanner">
              <Button variant="outline" size="sm" className="gap-2">
                <ScanSearch className="h-4 w-4" />
                Scan Ingredients
              </Button>
            </Link>
          </div>
          
          {/* Quick Stats */}
          <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mt-6">
            <div className="bg-card rounded-lg p-4 border">
              <div className="flex items-center gap-2 text-primary mb-1">
                <FlaskConical className="h-4 w-4" />
                <span className="text-sm font-medium">Products</span>
              </div>
              <p className="text-2xl font-bold">{stats.total}</p>
            </div>
            <div className="bg-card rounded-lg p-4 border">
              <div className="flex items-center gap-2 text-category-danger mb-1">
                <AlertTriangle className="h-4 w-4" />
                <span className="text-sm font-medium">High Risk</span>
              </div>
              <p className="text-2xl font-bold">{stats.highRisk}</p>
            </div>
            <div className="bg-card rounded-lg p-4 border">
              <div className="flex items-center gap-2 text-category-warning mb-1">
                <AlertTriangle className="h-4 w-4" />
                <span className="text-sm font-medium">Xenoestrogens</span>
              </div>
              <p className="text-2xl font-bold">{stats.withXenoestrogens}</p>
            </div>
            <div className="bg-card rounded-lg p-4 border">
              <div className="flex items-center gap-2 text-category-safe mb-1">
                <ShieldCheck className="h-4 w-4" />
                <span className="text-sm font-medium">Safe Options</span>
              </div>
              <p className="text-2xl font-bold">{stats.safe}</p>
            </div>
          </div>
        </div>
      </div>
      
      <main className="max-w-7xl mx-auto px-4 pb-16">
        {/* Search & Scan Bar */}
        <div className="py-6 space-y-4">
          <CosmeticsSearchBar
            value={searchQuery}
            onChange={setSearchQuery}
          />
          
          <CosmeticCategoryFilter
            selectedCategory={selectedCategory}
            onChange={setSelectedCategory}
            selectedRisk={selectedRisk}
            onRiskChange={setSelectedRisk}
          />
          
          <TobinFilter
            filters={tobinFilters}
            onChange={setTobinFilters}
            sortBy={sortBy}
            onSortChange={setSortBy}
          />
        </div>
        
        {/* Your Cosmetics Collection Dashboard */}
        {user && (
          <section className="mb-8">
            <CosmeticsDashboard 
              selectedCategory={selectedCategory}
              selectedRisk={selectedRisk}
            />
          </section>
        )}

        {/* Reference Database Header */}
        <div className="border-t pt-8 mb-6">
          <h2 className="text-xl font-semibold flex items-center gap-2 mb-2">
            <BookOpen className="h-5 w-5 text-primary" />
            Reference Database
          </h2>
          <p className="text-sm text-muted-foreground">
            Educational product categories with chemical hazard analysis based on CDC and EWG data.
          </p>
        </div>

        {/* Products Grid */}
        <div className="space-y-8">
          {selectedRisk !== null || sortBy !== 'none' ? (
            <div>
              <h2 className="text-xl font-semibold mb-4">
                {filteredAndSortedCosmetics.length} {selectedRisk === 1 ? "High Risk" : 
                  selectedRisk === 2 ? "Moderate Risk" :
                  selectedRisk === 3 ? "Low Risk" : 
                  selectedRisk === 4 ? "Safe" : ""} Products
                {sortBy !== 'none' && <span className="text-base font-normal text-muted-foreground ml-2">
                  (Sorted by {sortBy.includes('tobin') ? 'TOBIN score' : 'name'})
                </span>}
              </h2>
              <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-4">
                {filteredAndSortedCosmetics.map((cosmetic) => (
                  <CosmeticCard
                    key={cosmetic.id}
                    cosmetic={cosmetic}
                    onClick={() => setSelectedCosmetic(cosmetic)}
                  />
                ))}
              </div>
            </div>
          ) : searchQuery ? (
            <div>
              <h2 className="text-xl font-semibold mb-4">
                {filteredAndSortedCosmetics.length} Results for "{searchQuery}"
              </h2>
              <div className="grid sm:grid-cols-2 lg:grid-cols-3 gap-4">
                {filteredAndSortedCosmetics.map((cosmetic) => (
                  <CosmeticCard
                    key={cosmetic.id}
                    cosmetic={cosmetic}
                    onClick={() => setSelectedCosmetic(cosmetic)}
                  />
                ))}
              </div>
            </div>
          ) : groupedCosmetics && (
            <>
              {/* Safe Products First */}
              {groupedCosmetics.safe.length > 0 && (
                <div>
                  <h2 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <ShieldCheck className="h-5 w-5 text-category-safe" />
                    Safe & Clean Products
                    <span className="text-sm font-normal text-muted-foreground ml-2">
                      (No hormone disruptors)
                    </span>
                  </h2>
                  <div className="grid sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                    {groupedCosmetics.safe.map((cosmetic) => (
                      <CosmeticCard
                        key={cosmetic.id}
                        cosmetic={cosmetic}
                        onClick={() => setSelectedCosmetic(cosmetic)}
                      />
                    ))}
                  </div>
                </div>
              )}
              
              {/* Low Risk */}
              {groupedCosmetics.low.length > 0 && (
                <div>
                  <h2 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <span className="text-category-therapeutic">✓</span>
                    Low Risk Products
                    <span className="text-sm font-normal text-muted-foreground ml-2">
                      (Minimal concerns)
                    </span>
                  </h2>
                  <div className="grid sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                    {groupedCosmetics.low.map((cosmetic) => (
                      <CosmeticCard
                        key={cosmetic.id}
                        cosmetic={cosmetic}
                        onClick={() => setSelectedCosmetic(cosmetic)}
                      />
                    ))}
                  </div>
                </div>
              )}
              
              {/* Moderate Risk */}
              {groupedCosmetics.moderate.length > 0 && (
                <div>
                  <h2 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <AlertTriangle className="h-5 w-5 text-category-warning" />
                    Moderate Risk Products
                    <span className="text-sm font-normal text-muted-foreground ml-2">
                      (Contains chemicals of concern)
                    </span>
                  </h2>
                  <div className="grid sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                    {groupedCosmetics.moderate.map((cosmetic) => (
                      <CosmeticCard
                        key={cosmetic.id}
                        cosmetic={cosmetic}
                        onClick={() => setSelectedCosmetic(cosmetic)}
                      />
                    ))}
                  </div>
                </div>
              )}
              
              {/* High Risk */}
              {groupedCosmetics.high.length > 0 && (
                <div>
                  <h2 className="text-xl font-semibold mb-4 flex items-center gap-2">
                    <AlertTriangle className="h-5 w-5 text-category-danger" />
                    High Risk Products
                    <span className="text-sm font-normal text-muted-foreground ml-2">
                      (Significant chemical hazards)
                    </span>
                  </h2>
                  <div className="grid sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                    {groupedCosmetics.high.map((cosmetic) => (
                      <CosmeticCard
                        key={cosmetic.id}
                        cosmetic={cosmetic}
                        onClick={() => setSelectedCosmetic(cosmetic)}
                      />
                    ))}
                  </div>
                </div>
              )}
            </>
          )}
          
          {filteredAndSortedCosmetics.length === 0 && (
            <div className="text-center py-12">
              <FlaskConical className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
              <p className="text-muted-foreground">No products found matching your filters.</p>
            </div>
          )}
        </div>
      </main>
      
      {/* Detail Modal for Reference Database */}
      <CosmeticDetailModal
        cosmetic={selectedCosmetic}
        open={!!selectedCosmetic}
        onClose={() => setSelectedCosmetic(null)}
      />
    </div>
  );
};

export default Cosmetics;
