import { useState, useMemo } from "react";
import { Link } from "react-router-dom";
import { foodDatabase, FoodItem, getCategoryColor } from "@/data/foodDatabase";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { 
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { 
  ArrowLeft, 
  Scale, 
  Plus, 
  X, 
  Flame,
  Zap,
  TrendingUp,
  TrendingDown,
  Star,
  Leaf,
  AlertTriangle,
  Check,
  ScanBarcode,
  Camera,
  Search,
  Loader2
} from "lucide-react";
import { WebBarcodeScanner } from "@/components/WebBarcodeScanner";
import { useBarcodeLookup } from "@/hooks/useBarcodeLookup";
import { toast } from "sonner";

const getCategoryName = (category: number): string => {
  switch (category) {
    case 1: return "Avoid";
    case 2: return "Conditional";
    case 3: return "Safe";
    case 4: return "Therapeutic";
    default: return "Unknown";
  }
};

const FoodComparison = () => {
  const [selectedFoods, setSelectedFoods] = useState<FoodItem[]>([]);
  const [openPopover, setOpenPopover] = useState<number | null>(null);
  const [scannerOpen, setScannerOpen] = useState(false);
  const [scanSlot, setScanSlot] = useState<number>(0);
  const [barcodeInput, setBarcodeInput] = useState("");
  const [loadingSlot, setLoadingSlot] = useState<number | null>(null);
  const { lookupBarcode, loading: barcodeLoading } = useBarcodeLookup();

  const addFood = (food: FoodItem, slot: number) => {
    const newFoods = [...selectedFoods];
    newFoods[slot] = food;
    setSelectedFoods(newFoods);
    setOpenPopover(null);
  };

  const removeFood = (index: number) => {
    setSelectedFoods(selectedFoods.filter((_, i) => i !== index));
  };

  const handleBarcodeScanned = async (barcode: string, slot: number) => {
    setScannerOpen(false);
    setLoadingSlot(slot);
    
    try {
      const result = await lookupBarcode(barcode);
      
      if (result.found && result.product) {
        // Create a FoodItem from the scanned product
        const nutrition = result.product.nutrition;
        const scannedFood: FoodItem = {
          id: `scanned-${barcode}`,
          name: result.product.name || "Scanned Product",
          emoji: "📦",
          category: 2, // Default to conditional
          categoryName: "Conditional",
          netCarbs: nutrition?.carbs || 0,
          fiber: nutrition?.fiber || 0,
          protein: nutrition?.protein || 0,
          fat: nutrition?.fat || 0,
          absorptionCoefficient: 0.8,
          insulinSecretionFactor: 1.0,
          fatCarbCombinationFactor: 1.0,
          dnlConstant: 0.15,
          ampkScore: 0.5,
          mitoEfficiency: 0.5,
          glycogenicIndex: nutrition?.sugar ? Math.min(100, Math.round(nutrition.sugar * 2)) : 50,
          lipogenicIndex: nutrition?.fat ? Math.min(3, nutrition.fat / 10) : 1.0,
          mechanism: `${result.product.brand || 'Product'} - Scanned via barcode. Metabolic profile estimated from nutrition data.`,
        };
        
        addFood(scannedFood, slot);
        toast.success(`Added "${result.product.name}" to comparison`);
      } else {
        toast.error("Product not found. Try searching manually.");
      }
    } finally {
      setLoadingSlot(null);
    }
  };

  const openScanner = (slot: number) => {
    setScanSlot(slot);
    setScannerOpen(true);
    setOpenPopover(null);
  };

  const handleManualBarcodeSubmit = async (slot: number) => {
    const trimmedBarcode = barcodeInput.trim();
    if (!trimmedBarcode) {
      toast.error("Please enter a barcode number");
      return;
    }
    if (!/^\d{8,14}$/.test(trimmedBarcode)) {
      toast.error("Invalid barcode format. Please enter 8-14 digits.");
      return;
    }
    setOpenPopover(null);
    setBarcodeInput("");
    await handleBarcodeScanned(trimmedBarcode, slot);
  };

  const availableFoods = useMemo(() => {
    const selectedIds = selectedFoods.map(f => f.id);
    return foodDatabase.filter(f => !selectedIds.includes(f.id));
  }, [selectedFoods]);

  const getComparisonColor = (value: number, metric: 'li' | 'gi', isLower: boolean) => {
    if (selectedFoods.length < 2) return "";
    
    const values = selectedFoods.map(f => metric === 'li' ? f.lipogenicIndex : f.glycogenicIndex);
    const best = isLower ? Math.min(...values) : Math.max(...values);
    const worst = isLower ? Math.max(...values) : Math.min(...values);
    
    if (value === best) return "text-category-safe font-bold";
    if (value === worst) return "text-category-danger";
    return "text-category-warning";
  };

  const getBestFood = () => {
    if (selectedFoods.length < 2) return null;
    return selectedFoods.reduce((best, current) => 
      current.category > best.category ? current : best
    );
  };

  const bestFood = getBestFood();

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-b from-primary/10 to-background py-8 px-4">
        <div className="max-w-6xl mx-auto">
          <Link to="/">
            <Button variant="ghost" size="sm" className="mb-4 gap-2">
              <ArrowLeft className="h-4 w-4" />
              Back to Foods
            </Button>
          </Link>
          
          <div className="flex items-center gap-3 mb-4">
            <Scale className="h-8 w-8 text-primary" />
            <h1 className="text-3xl font-display font-bold">Food Comparison</h1>
          </div>
          <p className="text-muted-foreground max-w-2xl">
            Compare the metabolic profiles of 2-3 foods side-by-side. See how their 
            Lipogenic Index (LI), Glycemic Index (GI*), and dietary classifications stack up.
          </p>
        </div>
      </div>
      
      <main className="max-w-6xl mx-auto px-4 pb-16 space-y-8">
        {/* Food Selection */}
        <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
          {[0, 1, 2].map((slot) => (
            <div key={slot}>
              {selectedFoods[slot] ? (
                <Card className="relative border-2 border-primary/20 h-full">
                  <Button
                    variant="ghost"
                    size="icon"
                    className="absolute top-2 right-2 h-6 w-6"
                    onClick={() => removeFood(slot)}
                  >
                    <X className="h-4 w-4" />
                  </Button>
                  <CardContent className="pt-6 text-center">
                    <span className="text-4xl mb-2 block">{selectedFoods[slot].emoji}</span>
                    <h3 className="font-semibold">{selectedFoods[slot].name}</h3>
                    <Badge 
                      className={`mt-2 ${getCategoryColor(selectedFoods[slot].category)}`}
                    >
                      {getCategoryName(selectedFoods[slot].category)}
                    </Badge>
                  </CardContent>
                </Card>
              ) : loadingSlot === slot ? (
                <Card className="relative border-2 border-primary/30 h-full animate-pulse">
                  <CardContent className="pt-6 text-center flex flex-col items-center justify-center h-32">
                    <Loader2 className="h-8 w-8 text-primary animate-spin mb-2" />
                    <p className="text-sm text-muted-foreground font-medium">Looking up product...</p>
                    <div className="flex gap-1 mt-2">
                      <span className="w-1.5 h-1.5 bg-primary rounded-full animate-bounce" style={{ animationDelay: '0ms' }} />
                      <span className="w-1.5 h-1.5 bg-primary rounded-full animate-bounce" style={{ animationDelay: '150ms' }} />
                      <span className="w-1.5 h-1.5 bg-primary rounded-full animate-bounce" style={{ animationDelay: '300ms' }} />
                    </div>
                  </CardContent>
                </Card>
              ) : (
                <Popover open={openPopover === slot} onOpenChange={(open) => setOpenPopover(open ? slot : null)}>
                  <PopoverTrigger asChild>
                    <Button 
                      variant="outline" 
                      className="w-full h-32 border-dashed border-2 flex flex-col gap-2"
                    >
                      <Plus className="h-6 w-6" />
                      <span>Add Food {slot + 1}</span>
                    </Button>
                  </PopoverTrigger>
                  <PopoverContent className="w-80 p-0" align="start">
                    {/* Barcode Entry Section */}
                    <div className="p-3 border-b space-y-2">
                      <div className="flex items-center gap-2 text-sm font-medium text-muted-foreground">
                        <ScanBarcode className="h-4 w-4" />
                        Enter Barcode
                      </div>
                      <div className="flex gap-2">
                        <Input
                          placeholder="e.g. 5901234123457"
                          value={barcodeInput}
                          onChange={(e) => setBarcodeInput(e.target.value.replace(/\D/g, '').slice(0, 14))}
                          onKeyDown={(e) => {
                            if (e.key === 'Enter') {
                              handleManualBarcodeSubmit(slot);
                            }
                          }}
                          className="flex-1"
                          disabled={barcodeLoading}
                        />
                        <Button 
                          size="sm" 
                          onClick={() => handleManualBarcodeSubmit(slot)}
                          disabled={barcodeLoading || !barcodeInput.trim()}
                        >
                          <Search className="h-4 w-4" />
                        </Button>
                      </div>
                      <Button
                        variant="outline"
                        size="sm"
                        className="w-full gap-2"
                        onClick={() => openScanner(slot)}
                        disabled={barcodeLoading}
                      >
                        <Camera className="h-4 w-4" />
                        Scan with Camera
                      </Button>
                    </div>
                    <Command>
                      <CommandInput placeholder="Search foods..." />
                      <CommandList>
                        <CommandEmpty>No food found.</CommandEmpty>
                        <CommandGroup heading="From Database">
                          {availableFoods.map((food) => (
                            <CommandItem
                              key={food.id}
                              value={food.name}
                              onSelect={() => addFood(food, slot)}
                              className="cursor-pointer"
                            >
                              <span className="mr-2">{food.emoji}</span>
                              {food.name}
                            </CommandItem>
                          ))}
                        </CommandGroup>
                      </CommandList>
                    </Command>
                  </PopoverContent>
                </Popover>
              )}
            </div>
          ))}
        </div>

        {/* Comparison Table */}
        {selectedFoods.length >= 2 && (
          <>
            {/* Best Choice Banner */}
            {bestFood && (
              <div className="bg-category-safe/10 border border-category-safe/30 rounded-lg p-4 flex items-center gap-3">
                <Check className="h-6 w-6 text-category-safe" />
                <div>
                  <p className="font-semibold text-category-safe">Best Choice: {bestFood.name}</p>
                  <p className="text-sm text-muted-foreground">
                    Based on metabolic category and overall health impact
                  </p>
                </div>
              </div>
            )}

            {/* Metabolic Indices */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Flame className="h-5 w-5 text-category-danger" />
                  Metabolic Indices
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b">
                        <th className="text-left py-3 px-2 font-medium">Metric</th>
                        {selectedFoods.map((food, idx) => (
                          <th key={idx} className="text-center py-3 px-2 font-medium">
                            <span className="mr-1">{food.emoji}</span>
                            {food.name}
                          </th>
                        ))}
                      </tr>
                    </thead>
                    <tbody>
                      <tr className="border-b">
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <Flame className="h-4 w-4 text-category-danger" />
                            Lipogenic Index (LI)
                          </div>
                          <p className="text-xs text-muted-foreground">Lower is better</p>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2">
                            <span className={`text-xl ${getComparisonColor(food.lipogenicIndex, 'li', true)}`}>
                              {food.lipogenicIndex.toFixed(1)}
                            </span>
                          </td>
                        ))}
                      </tr>
                      <tr className="border-b">
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <Zap className="h-4 w-4 text-category-warning" />
                            Glycemic Index (GI*)
                          </div>
                          <p className="text-xs text-muted-foreground">Lower is better</p>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2">
                            <span className={`text-xl ${getComparisonColor(food.glycogenicIndex, 'gi', true)}`}>
                              {food.glycogenicIndex}
                            </span>
                          </td>
                        ))}
                      </tr>
                      <tr className="border-b">
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <TrendingUp className="h-4 w-4 text-primary" />
                            Category
                          </div>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2">
                            <Badge className={getCategoryColor(food.category)}>
                              {getCategoryName(food.category)}
                            </Badge>
                          </td>
                        ))}
                      </tr>
                      <tr>
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <TrendingDown className="h-4 w-4 text-muted-foreground" />
                            Insulin Impact
                          </div>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2 text-sm">
                            {food.glycogenicIndex > 70 ? (
                              <span className="text-category-danger">High spike</span>
                            ) : food.glycogenicIndex > 40 ? (
                              <span className="text-category-warning">Moderate</span>
                            ) : (
                              <span className="text-category-safe">Low/Stable</span>
                            )}
                          </td>
                        ))}
                      </tr>
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>

            {/* Dietary Classifications */}
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Star className="h-5 w-5 text-primary" />
                  Dietary Classifications
                </CardTitle>
              </CardHeader>
              <CardContent>
                <div className="overflow-x-auto">
                  <table className="w-full">
                    <thead>
                      <tr className="border-b">
                        <th className="text-left py-3 px-2 font-medium">Classification</th>
                        {selectedFoods.map((food, idx) => (
                          <th key={idx} className="text-center py-3 px-2 font-medium">
                            <span className="mr-1">{food.emoji}</span>
                            {food.name}
                          </th>
                        ))}
                      </tr>
                    </thead>
                    <tbody>
                      <tr className="border-b">
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <Star className="h-4 w-4 text-category-safe" />
                            Halal Status
                          </div>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2">
                            {food.dietaryClassification ? (
                              <Badge variant={
                                food.dietaryClassification.halal === "halal" ? "default" :
                                food.dietaryClassification.halal === "haram" ? "destructive" : "secondary"
                              }>
                                {food.dietaryClassification.halal.charAt(0).toUpperCase() + 
                                 food.dietaryClassification.halal.slice(1)}
                              </Badge>
                            ) : (
                              <span className="text-muted-foreground text-sm">N/A</span>
                            )}
                          </td>
                        ))}
                      </tr>
                      <tr className="border-b">
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <Star className="h-4 w-4 text-primary" />
                            Kosher Status
                          </div>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2">
                            {food.dietaryClassification ? (
                              <Badge variant={
                                food.dietaryClassification.kosher === "kosher" ? "default" :
                                food.dietaryClassification.kosher === "pareve" ? "outline" :
                                food.dietaryClassification.kosher === "not_kosher" ? "destructive" : "secondary"
                              }>
                                {food.dietaryClassification.kosher === "not_kosher" ? "Not Kosher" :
                                 food.dietaryClassification.kosher.charAt(0).toUpperCase() + 
                                 food.dietaryClassification.kosher.slice(1)}
                              </Badge>
                            ) : (
                              <span className="text-muted-foreground text-sm">N/A</span>
                            )}
                          </td>
                        ))}
                      </tr>
                      <tr>
                        <td className="py-3 px-2">
                          <div className="flex items-center gap-2">
                            <Leaf className="h-4 w-4 text-category-therapeutic" />
                            Organic Status
                          </div>
                        </td>
                        {selectedFoods.map((food, idx) => (
                          <td key={idx} className="text-center py-3 px-2">
                            {food.dietaryClassification ? (
                              <Badge variant={
                                food.dietaryClassification.organic === "organic" ? "default" :
                                food.dietaryClassification.organic === "varies" ? "secondary" : "outline"
                              }>
                                {food.dietaryClassification.organic === "non_organic" ? "Non-Organic" :
                                 food.dietaryClassification.organic.charAt(0).toUpperCase() + 
                                 food.dietaryClassification.organic.slice(1)}
                              </Badge>
                            ) : (
                              <span className="text-muted-foreground text-sm">N/A</span>
                            )}
                          </td>
                        ))}
                      </tr>
                    </tbody>
                  </table>
                </div>
              </CardContent>
            </Card>

            {/* Chemical Concerns */}
            {selectedFoods.some(f => f.chemicalContaminants && f.chemicalContaminants.length > 0) && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <AlertTriangle className="h-5 w-5 text-category-warning" />
                    Chemical Concerns
                  </CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="grid gap-4" style={{ gridTemplateColumns: `repeat(${selectedFoods.length}, 1fr)` }}>
                    {selectedFoods.map((food, idx) => (
                      <div key={idx} className="space-y-2">
                        <p className="font-medium text-sm">{food.emoji} {food.name}</p>
                        {food.chemicalContaminants && food.chemicalContaminants.length > 0 ? (
                          <div className="flex flex-wrap gap-1">
                            {food.chemicalContaminants.map((chem, cidx) => (
                              <Badge key={cidx} variant="outline" className="text-xs text-category-warning border-category-warning">
                                {chem}
                              </Badge>
                            ))}
                          </div>
                        ) : (
                          <p className="text-sm text-category-safe flex items-center gap-1">
                            <Check className="h-3 w-3" /> No major concerns
                          </p>
                        )}
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Key Benefits */}
            <Card>
              <CardHeader>
                <CardTitle>Key Benefits & Considerations</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid gap-4" style={{ gridTemplateColumns: `repeat(${selectedFoods.length}, 1fr)` }}>
                  {selectedFoods.map((food, idx) => (
                    <div key={idx} className="space-y-2">
                      <p className="font-medium text-sm flex items-center gap-1">
                        {food.emoji} {food.name}
                      </p>
                      {food.benefits && food.benefits.length > 0 && (
                        <ul className="text-sm space-y-1">
                          {food.benefits.slice(0, 3).map((benefit, bidx) => (
                            <li key={bidx} className="flex items-start gap-1">
                              <Check className="h-3 w-3 text-category-safe mt-1 flex-shrink-0" />
                              <span>{benefit}</span>
                            </li>
                          ))}
                        </ul>
                      )}
                      {food.synergyWarning && (
                        <div className="pt-2 border-t">
                          <p className="text-xs text-category-warning flex items-center gap-1">
                            <AlertTriangle className="h-3 w-3" />
                            {food.synergyWarning}
                          </p>
                        </div>
                      )}
                    </div>
                  ))}
                </div>
              </CardContent>
            </Card>
          </>
        )}

        {/* Empty State */}
        {selectedFoods.length < 2 && (
          <Card className="border-dashed">
            <CardContent className="py-12 text-center">
              <Scale className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
              <h3 className="font-semibold mb-2">Select Foods to Compare</h3>
              <p className="text-muted-foreground text-sm max-w-md mx-auto">
                Add at least 2 foods using the buttons above to see a detailed 
                side-by-side comparison of their metabolic profiles.
              </p>
            </CardContent>
          </Card>
        )}
      </main>

      {/* Barcode Scanner Dialog */}
      <Dialog open={scannerOpen} onOpenChange={setScannerOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle className="flex items-center gap-2">
              <ScanBarcode className="h-5 w-5" />
              Scan Barcode to Add Food
            </DialogTitle>
          </DialogHeader>
          <WebBarcodeScanner
            onScan={(barcode) => handleBarcodeScanned(barcode, scanSlot)}
            onClose={() => setScannerOpen(false)}
          />
          {barcodeLoading && (
            <div className="text-center text-sm text-muted-foreground py-2">
              Looking up product...
            </div>
          )}
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default FoodComparison;
