import { useState, useMemo } from "react";
import { Link } from "react-router-dom";
import { foodDatabase, FoodItem } from "@/data/foodDatabase";
import { HeroSection } from "@/components/HeroSection";
import { SearchBar } from "@/components/SearchBar";
import { CategoryFilter } from "@/components/CategoryFilter";
import { DietaryFilter, DietaryFilters } from "@/components/DietaryFilter";
import { TobinFilter, TobinFilters, SortOption, defaultTobinFilters } from "@/components/TobinFilter";
import { FoodCard } from "@/components/FoodCard";
import { FoodDetailModal } from "@/components/FoodDetailModal";
import { EquationDisplay } from "@/components/EquationDisplay";
import { MealGuide } from "@/components/MealGuide";
import { HealthChatbot } from "@/components/HealthChatbot";
import { StatsOverview } from "@/components/StatsOverview";
import { UserDashboard } from "@/components/UserDashboard";
import { Footer } from "@/components/Footer";
import { Button } from "@/components/ui/button";
import { Scale, ScanBarcode, Leaf } from "lucide-react";
import { useAuth } from "@/hooks/useAuth";

const Index = () => {
  const { user } = useAuth();
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<number | null>(null);
  const [selectedFood, setSelectedFood] = useState<FoodItem | null>(null);
  const [dietaryFilters, setDietaryFilters] = useState<DietaryFilters>({
    halal: "all",
    kosher: "all",
    organic: "all",
    phytoestrogen: "all",
  });
  const [tobinFilters, setTobinFilters] = useState<TobinFilters>(defaultTobinFilters);
  const [sortBy, setSortBy] = useState<SortOption>('none');
  
  const filteredAndSortedFoods = useMemo(() => {
    let foods = foodDatabase.filter((food) => {
      const matchesSearch = food.name.toLowerCase().includes(searchQuery.toLowerCase());
      const matchesCategory = selectedCategory === null || food.category === selectedCategory;
      
      // Dietary filters
      const matchesHalal = dietaryFilters.halal === "all" || 
        food.dietaryClassification?.halal === dietaryFilters.halal;
      const matchesKosher = dietaryFilters.kosher === "all" || 
        food.dietaryClassification?.kosher === dietaryFilters.kosher;
      const matchesOrganic = dietaryFilters.organic === "all" || 
        food.dietaryClassification?.organic === dietaryFilters.organic;
      
      // Phytoestrogen filter
      let matchesPhytoestrogen = true;
      if (dietaryFilters.phytoestrogen === "hide") {
        // Hide all foods with phytoestrogens
        matchesPhytoestrogen = !food.estrogenicProfile?.hasPhytoestrogens;
      } else if (dietaryFilters.phytoestrogen === "low_only") {
        // Show foods with no phytoestrogens or low potency only
        matchesPhytoestrogen = !food.estrogenicProfile?.hasPhytoestrogens || 
          food.estrogenicProfile?.estrogenicPotency === "low" ||
          food.estrogenicProfile?.estrogenicPotency === "none";
      } else if (dietaryFilters.phytoestrogen === "show_estrogenic") {
        // Only show foods that have phytoestrogens
        matchesPhytoestrogen = food.estrogenicProfile?.hasPhytoestrogens === true;
      }
      
      // TOBIN filters
      const tobinScore = food.tobinScore;
      const matchesTobin = !tobinScore || (
        tobinScore.total <= tobinFilters.maxTotal &&
        tobinScore.T <= tobinFilters.maxT &&
        tobinScore.O <= tobinFilters.maxO &&
        tobinScore.B <= tobinFilters.maxB &&
        tobinScore.I <= tobinFilters.maxI &&
        tobinScore.N <= tobinFilters.maxN
      );
      
      return matchesSearch && matchesCategory && matchesHalal && matchesKosher && matchesOrganic && matchesPhytoestrogen && matchesTobin;
    });
    
    // Sorting
    if (sortBy === 'tobin-asc') {
      foods = [...foods].sort((a, b) => (a.tobinScore?.total ?? 50) - (b.tobinScore?.total ?? 50));
    } else if (sortBy === 'tobin-desc') {
      foods = [...foods].sort((a, b) => (b.tobinScore?.total ?? 0) - (a.tobinScore?.total ?? 0));
    } else if (sortBy === 'name-asc') {
      foods = [...foods].sort((a, b) => a.name.localeCompare(b.name));
    } else if (sortBy === 'name-desc') {
      foods = [...foods].sort((a, b) => b.name.localeCompare(a.name));
    }
    
    return foods;
  }, [searchQuery, selectedCategory, dietaryFilters, tobinFilters, sortBy]);
  
  // Group foods by category for display
  const groupedFoods = useMemo(() => {
    if (selectedCategory !== null || sortBy !== 'none') return null;
    
    return {
      therapeutic: filteredAndSortedFoods.filter(f => f.category === 4),
      safe: filteredAndSortedFoods.filter(f => f.category === 3),
      conditional: filteredAndSortedFoods.filter(f => f.category === 2),
      avoid: filteredAndSortedFoods.filter(f => f.category === 1),
    };
  }, [filteredAndSortedFoods, selectedCategory, sortBy]);
  
  return (
    <div className="min-h-screen bg-background p-[5px]">
      <HeroSection />
      
      <main className="max-w-7xl mx-auto space-y-4 sm:space-y-6 md:space-y-8">
        {/* Search Section */}
        <section className="space-y-3 sm:space-y-4">
          <SearchBar 
            value={searchQuery}
            onChange={setSearchQuery}
          />
          
          <CategoryFilter 
            selectedCategory={selectedCategory}
            onChange={setSelectedCategory}
          />
          
          <DietaryFilter
            filters={dietaryFilters}
            onChange={setDietaryFilters}
          />
          
          <TobinFilter
            filters={tobinFilters}
            onChange={setTobinFilters}
            sortBy={sortBy}
            onSortChange={setSortBy}
          />
        </section>
        
        {/* User Dashboard - Shows personal items when logged in */}
        {user && (
          <section>
            <UserDashboard />
          </section>
        )}
        
        {/* Stats Overview */}
        <section className="flex flex-col lg:flex-row items-start lg:items-center justify-between gap-4">
          <StatsOverview />
          <div className="flex flex-wrap gap-2 w-full lg:w-auto">
            <Link to="/barcode-scanner" className="flex-1 sm:flex-none">
              <Button variant="outline" className="gap-2 w-full sm:w-auto">
                <ScanBarcode className="h-4 w-4" />
                <span className="hidden xs:inline">Scan</span> Barcode
              </Button>
            </Link>
            <Link to="/tobin-comparison" className="flex-1 sm:flex-none">
              <Button variant="outline" className="gap-2 w-full sm:w-auto">
                <Scale className="h-4 w-4" />
                Compare<span className="hidden xs:inline"> Items</span>
              </Button>
            </Link>
            <Link to="/phytoestrogen-scanner" className="flex-1 sm:flex-none">
              <Button variant="outline" className="gap-2 w-full sm:w-auto border-category-warning text-category-warning hover:bg-category-warning/10">
                <Leaf className="h-4 w-4" />
                Phytoestrogens
              </Button>
            </Link>
          </div>
        </section>
        
        {/* Food Database Reference Section */}
        <section className="space-y-4 sm:space-y-6 md:space-y-8">
          <div className="flex items-center justify-between">
            <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold">
              {user ? "📚 Food Database Reference" : "🍽️ Browse Foods"}
            </h2>
            {user && (
              <p className="text-sm text-muted-foreground hidden md:block">
                Browse our complete food database for research and reference
              </p>
            )}
          </div>
          
          {selectedCategory !== null || sortBy !== 'none' ? (
            // Flat list when category is selected or sorting is active
            <div>
              <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold mb-3 sm:mb-4 md:mb-6">
                {filteredAndSortedFoods.length} {selectedCategory === 1 ? 'Foods to Avoid' : 
                  selectedCategory === 2 ? 'Conditional Foods' :
                  selectedCategory === 3 ? 'Safe Foods' : 
                  selectedCategory === 4 ? 'Therapeutic Foods' : 'Foods'}
                {sortBy !== 'none' && <span className="text-xs sm:text-sm md:text-base font-normal text-muted-foreground ml-1 sm:ml-2">
                  (Sorted by {sortBy.includes('tobin') ? 'TOBIN score' : 'name'})
                </span>}
              </h2>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4">
                {filteredAndSortedFoods.map((food) => (
                  <FoodCard
                    key={food.id}
                    food={food}
                    onClick={() => setSelectedFood(food)}
                  />
                ))}
              </div>
            </div>
          ) : searchQuery ? (
            // Search results
            <div>
              <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold mb-3 sm:mb-4 md:mb-6">
                {filteredAndSortedFoods.length} Results for "{searchQuery}"
              </h2>
              <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3 sm:gap-4">
                {filteredAndSortedFoods.map((food) => (
                  <FoodCard
                    key={food.id}
                    food={food}
                    onClick={() => setSelectedFood(food)}
                  />
                ))}
              </div>
            </div>
          ) : groupedFoods && (
            // Grouped by category
            <>
              {/* Therapeutic Foods First */}
              {groupedFoods.therapeutic.length > 0 && (
                <div>
                  <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold mb-2 sm:mb-3 md:mb-4 flex flex-wrap items-center gap-1 sm:gap-2">
                    <span className="text-category-therapeutic">✨</span>
                    Therapeutic Foods
                    <span className="text-xs sm:text-sm font-normal text-muted-foreground ml-1 sm:ml-2">
                      (Promotes fat loss)
                    </span>
                  </h2>
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3 sm:gap-4">
                    {groupedFoods.therapeutic.map((food) => (
                      <FoodCard
                        key={food.id}
                        food={food}
                        onClick={() => setSelectedFood(food)}
                      />
                    ))}
                  </div>
                </div>
              )}
              
              {/* Safe Foods */}
              {groupedFoods.safe.length > 0 && (
                <div>
                  <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold mb-2 sm:mb-3 md:mb-4 flex flex-wrap items-center gap-1 sm:gap-2">
                    <span className="text-category-safe">✓</span>
                    Metabolically Safe Foods
                    <span className="text-xs sm:text-sm font-normal text-muted-foreground ml-1 sm:ml-2">
                      (Foundation foods)
                    </span>
                  </h2>
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3 sm:gap-4">
                    {groupedFoods.safe.map((food) => (
                      <FoodCard
                        key={food.id}
                        food={food}
                        onClick={() => setSelectedFood(food)}
                      />
                    ))}
                  </div>
                </div>
              )}
              
              {/* Conditional Foods */}
              {groupedFoods.conditional.length > 0 && (
                <div>
                  <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold mb-2 sm:mb-3 md:mb-4 flex flex-wrap items-center gap-1 sm:gap-2">
                    <span className="text-category-warning">⚠️</span>
                    Conditional Foods
                    <span className="text-xs sm:text-sm font-normal text-muted-foreground ml-1 sm:ml-2">
                      (Context-dependent)
                    </span>
                  </h2>
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3 sm:gap-4">
                    {groupedFoods.conditional.map((food) => (
                      <FoodCard
                        key={food.id}
                        food={food}
                        onClick={() => setSelectedFood(food)}
                      />
                    ))}
                  </div>
                </div>
              )}
              
              {/* Avoid Foods */}
              {groupedFoods.avoid.length > 0 && (
                <div>
                  <h2 className="text-lg sm:text-xl md:text-2xl font-display font-bold mb-2 sm:mb-3 md:mb-4 flex flex-wrap items-center gap-1 sm:gap-2">
                    <span className="text-category-danger">❌</span>
                    Foods to Avoid
                    <span className="text-xs sm:text-sm font-normal text-muted-foreground ml-1 sm:ml-2">
                      (Rapid weight gain)
                    </span>
                  </h2>
                  <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3 sm:gap-4">
                    {groupedFoods.avoid.map((food) => (
                      <FoodCard
                        key={food.id}
                        food={food}
                        onClick={() => setSelectedFood(food)}
                      />
                    ))}
                  </div>
                </div>
              )}
            </>
          )}
        </section>
        
        {/* Educational Section */}
        <section className="grid grid-cols-1 lg:grid-cols-2 gap-4 sm:gap-6">
          <EquationDisplay />
          <MealGuide />
        </section>
      </main>
      
      {/* Food Detail Modal */}
      <FoodDetailModal
        food={selectedFood}
        open={!!selectedFood}
        onClose={() => setSelectedFood(null)}
      />
      
      {/* Health Chatbot */}
      <HealthChatbot />
      
      <Footer />
    </div>
  );
};

export default Index;
