import { useState, useRef } from "react";
import { Link } from "react-router-dom";
import { Button } from "@/components/ui/button";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { 
  ArrowLeft, 
  Search, 
  FlaskConical, 
  AlertTriangle,
  Shield,
  Loader2,
  Sparkles,
  Copy,
  RotateCcw,
  Skull,
  Flame,
  Activity,
  Bug,
  Pill,
  Crown
} from "lucide-react";
import { toast } from "@/hooks/use-toast";
import { UserMenu } from "@/components/UserMenu";
import { useSubscription } from "@/hooks/useSubscription";
import { aiApi } from "@/lib/apiServices";

const productTypes = [
  { value: "skincare", label: "Skincare (Lotion, Cream, Serum)" },
  { value: "makeup", label: "Makeup (Foundation, Lipstick)" },
  { value: "haircare", label: "Haircare (Shampoo, Conditioner)" },
  { value: "sunscreen", label: "Sunscreen" },
  { value: "deodorant", label: "Deodorant/Antiperspirant" },
  { value: "fragrance", label: "Fragrance/Perfume" },
  { value: "oral_care", label: "Oral Care (Toothpaste, Mouthwash)" },
  { value: "nail_care", label: "Nail Care (Polish, Remover)" },
  { value: "baby_product", label: "Baby Product" },
  { value: "food", label: "Food/Beverage" },
  { value: "supplement", label: "Supplement/Vitamin" },
  { value: "cleaning", label: "Cleaning Product" },
];

const exampleIngredients = `Water, Glycerin, Cetearyl Alcohol, Caprylic/Capric Triglyceride, Butyrospermum Parkii (Shea) Butter, Dimethicone, Cetyl Alcohol, Phenoxyethanol, Methylparaben, Propylparaben, Fragrance (Parfum), Tocopheryl Acetate, Sodium Hydroxide, Carbomer, Disodium EDTA, Oxybenzone`;

const tobinCategories = [
  { key: 'T', name: 'Toxins', icon: Skull, color: 'text-red-500', bgColor: 'bg-red-500', description: 'Direct toxic load from chemicals' },
  { key: 'O', name: 'Oxidative Stress', icon: Flame, color: 'text-orange-500', bgColor: 'bg-orange-500', description: 'Pro-oxidant & inflammatory triggers' },
  { key: 'B', name: 'Biological/Hormonal', icon: Activity, color: 'text-purple-500', bgColor: 'bg-purple-500', description: 'Endocrine & hormone disruptors' },
  { key: 'I', name: 'Inflammatory', icon: Bug, color: 'text-yellow-500', bgColor: 'bg-yellow-500', description: 'Allergens & immune triggers' },
  { key: 'N', name: 'Nutrient Depleting', icon: Pill, color: 'text-blue-500', bgColor: 'bg-blue-500', description: 'Mineral & vitamin antagonists' },
];

const IngredientScanner = () => {
  const { subscribed, loading: subscriptionLoading } = useSubscription();
  const [ingredients, setIngredients] = useState("");
  const [productType, setProductType] = useState("skincare");
  const [analysis, setAnalysis] = useState("");
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const abortControllerRef = useRef<AbortController | null>(null);

  const analyzeIngredients = async () => {
    if (!ingredients.trim()) {
      toast({
        title: "No ingredients",
        description: "Please enter an ingredient list to analyze.",
        variant: "destructive",
      });
      return;
    }

    if (ingredients.trim().length < 10) {
      toast({
        title: "Too short",
        description: "Please enter a complete ingredient list.",
        variant: "destructive",
      });
      return;
    }

    setIsAnalyzing(true);
    setAnalysis("");
    try {
      const analysisType = productTypes.find(p => p.value === productType)?.label || productType;
      const data = await aiApi.analyzeIngredients(ingredients.trim(), analysisType);
      setAnalysis(data.analysis || "");
    } catch (error) {
      console.error("Analysis error:", error);
      toast({
        title: "Analysis failed",
        description: error instanceof Error ? error.message : "Could not analyze ingredients",
        variant: "destructive",
      });
    } finally {
      setIsAnalyzing(false);
    }
  };

  const handleReset = () => {
    if (abortControllerRef.current) {
      abortControllerRef.current.abort();
    }
    setIngredients("");
    setAnalysis("");
    setIsAnalyzing(false);
  };

  const handlePasteExample = () => {
    setIngredients(exampleIngredients);
  };

  const handleCopyAnalysis = () => {
    navigator.clipboard.writeText(analysis);
    toast({
      title: "Copied",
      description: "Analysis copied to clipboard",
    });
  };

  // Parse TOBIN scores from analysis
  const parseTobinScores = (text: string): Record<string, number> => {
    const scores: Record<string, number> = {};
    const patterns = [
      /T\s*[-–]\s*Toxins?\s*\|\s*(\d+)\/10/i,
      /O\s*[-–]\s*Oxidative\s*Stress\s*\|\s*(\d+)\/10/i,
      /B\s*[-–]\s*Biological\/Hormonal\s*\|\s*(\d+)\/10/i,
      /I\s*[-–]\s*Inflammatory\s*\|\s*(\d+)\/10/i,
      /N\s*[-–]\s*Nutrient\s*Depleting\s*\|\s*(\d+)\/10/i,
    ];
    
    const keys = ['T', 'O', 'B', 'I', 'N'];
    patterns.forEach((pattern, idx) => {
      const match = text.match(pattern);
      if (match) {
        scores[keys[idx]] = parseInt(match[1], 10);
      }
    });
    
    return scores;
  };

  const tobinScores = parseTobinScores(analysis);
  const overallScore = Object.values(tobinScores).reduce((a, b) => a + b, 0);
  const hasScores = Object.keys(tobinScores).length > 0;

  // Enhanced markdown-like rendering
  const renderAnalysis = (text: string) => {
    const lines = text.split("\n");
    return lines.map((line, idx) => {
      // Skip table rows - we render them separately
      if (line.includes("|") && (line.includes("T -") || line.includes("O -") || line.includes("B -") || line.includes("I -") || line.includes("N -") || line.includes("OVERALL"))) {
        return null;
      }
      if (line.trim().startsWith("|") && line.includes("---")) {
        return null;
      }
      if (line.trim().startsWith("| Category")) {
        return null;
      }
      
      // Headers
      if (line.startsWith("### ")) {
        return <h3 key={idx} className="text-lg font-bold mt-6 mb-3 flex items-center gap-2">{line.slice(4)}</h3>;
      }
      if (line.startsWith("## ")) {
        return <h2 key={idx} className="text-xl font-bold mt-6 mb-3">{line.slice(3)}</h2>;
      }
      if (line.startsWith("# ")) {
        return <h1 key={idx} className="text-2xl font-bold mt-6 mb-3">{line.slice(2)}</h1>;
      }
      
      // Bold text
      let processed = line.replace(/\*\*(.*?)\*\*/g, '<strong class="text-foreground">$1</strong>');
      
      // Numbered lists
      if (/^\d+\.\s/.test(line.trim())) {
        return (
          <li 
            key={idx} 
            className="ml-4 mb-2 list-decimal"
            dangerouslySetInnerHTML={{ __html: processed.replace(/^\d+\.\s*/, '') }}
          />
        );
      }
      
      // Bullet points
      if (line.trim().startsWith("- ") || line.trim().startsWith("• ")) {
        return (
          <li 
            key={idx} 
            className="ml-4 mb-1"
            dangerouslySetInnerHTML={{ __html: processed.replace(/^[-•]\s*/, '') }}
          />
        );
      }
      
      // Empty lines
      if (!line.trim()) {
        return <br key={idx} />;
      }
      
      // Regular text
      return <p key={idx} className="mb-2" dangerouslySetInnerHTML={{ __html: processed }} />;
    });
  };

  const getScoreColor = (score: number) => {
    if (score <= 2) return 'bg-green-500';
    if (score <= 4) return 'bg-yellow-500';
    if (score <= 6) return 'bg-orange-500';
    return 'bg-red-500';
  };

  const getOverallRating = (score: number) => {
    if (score <= 10) return { label: 'Excellent', color: 'text-green-600', bg: 'bg-green-100' };
    if (score <= 20) return { label: 'Good', color: 'text-emerald-600', bg: 'bg-emerald-100' };
    if (score <= 30) return { label: 'Moderate Concern', color: 'text-yellow-600', bg: 'bg-yellow-100' };
    if (score <= 40) return { label: 'High Concern', color: 'text-orange-600', bg: 'bg-orange-100' };
    return { label: 'Severe Concern', color: 'text-red-600', bg: 'bg-red-100' };
  };

  // Premium required for scanning
  if (!subscriptionLoading && !subscribed) {
    return (
      <div className="min-h-screen bg-background">
        <div className="bg-gradient-to-b from-primary/10 to-background py-8 px-4">
          <div className="max-w-4xl mx-auto">
            <div className="flex items-center justify-between mb-4">
              <Link to="/cosmetics">
                <Button variant="ghost" size="sm" className="gap-2">
                  <ArrowLeft className="h-4 w-4" />
                  Back to Cosmetics
                </Button>
              </Link>
              <UserMenu />
            </div>
          </div>
        </div>
        <main className="max-w-md mx-auto px-4 py-8">
          <Card className="border-2 border-primary/20 shadow-lg">
            <CardHeader className="text-center">
              <div className="mx-auto mb-4 w-16 h-16 bg-primary/10 rounded-full flex items-center justify-center">
                <Crown className="h-8 w-8 text-primary" />
              </div>
              <CardTitle className="text-xl">Ingredient scanning is a Premium feature</CardTitle>
              <CardDescription className="text-base">
                Upgrade to Premium to analyze ingredients with the TOBIN framework. Start with a 7-day free trial.
              </CardDescription>
            </CardHeader>
            <CardContent className="flex flex-col gap-3">
              <Button asChild className="w-full">
                <Link to="/subscription">Upgrade to Premium</Link>
              </Button>
              <Button asChild variant="outline" className="w-full">
                <Link to="/cosmetics">Back to Cosmetics</Link>
              </Button>
            </CardContent>
          </Card>
        </main>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <div className="bg-gradient-to-b from-primary/10 to-background py-8 px-4">
        <div className="max-w-4xl mx-auto">
          <div className="flex items-center justify-between mb-4">
            <Link to="/cosmetics">
              <Button variant="ghost" size="sm" className="gap-2">
                <ArrowLeft className="h-4 w-4" />
                Back to Cosmetics
              </Button>
            </Link>
            <UserMenu />
          </div>
          
          <div className="flex items-center gap-3 mb-4">
            <Search className="h-8 w-8 text-primary" />
            <h1 className="text-3xl font-display font-bold">TOBIN Ingredient Scanner</h1>
          </div>
          <p className="text-muted-foreground max-w-2xl">
            Analyze ingredients using the <strong>TOBIN framework</strong>: Toxins, Oxidative stress, 
            Biological/Hormonal imbalance, Inflammatory triggers, and Nutrient depletion.
          </p>
        </div>
      </div>
      
      <main className="max-w-4xl mx-auto px-4 pb-16 space-y-6">
        {/* TOBIN Legend */}
        <Card className="bg-gradient-to-r from-muted/50 to-muted/30">
          <CardContent className="pt-6">
            <h3 className="font-semibold mb-4 flex items-center gap-2">
              <Shield className="h-5 w-5 text-primary" />
              TOBIN Framework
            </h3>
            <div className="grid grid-cols-2 md:grid-cols-5 gap-3">
              {tobinCategories.map((cat) => (
                <div key={cat.key} className="flex flex-col items-center text-center p-2 rounded-lg bg-background/50">
                  <cat.icon className={`h-6 w-6 ${cat.color} mb-1`} />
                  <span className="font-bold text-sm">{cat.key}</span>
                  <span className="text-xs text-muted-foreground">{cat.name}</span>
                </div>
              ))}
            </div>
          </CardContent>
        </Card>

        {/* Input Section */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FlaskConical className="h-5 w-5 text-primary" />
              Enter Ingredients
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid sm:grid-cols-2 gap-4">
              <div>
                <label className="text-sm font-medium mb-2 block">Product Type</label>
                <Select value={productType} onValueChange={setProductType}>
                  <SelectTrigger>
                    <SelectValue placeholder="Select product type" />
                  </SelectTrigger>
                  <SelectContent>
                    {productTypes.map((type) => (
                      <SelectItem key={type.value} value={type.value}>
                        {type.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
              <div className="flex items-end">
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={handlePasteExample}
                  className="gap-2"
                >
                  <Sparkles className="h-4 w-4" />
                  Try Example
                </Button>
              </div>
            </div>
            
            <div>
              <label className="text-sm font-medium mb-2 block">
                Ingredient List
                <span className="text-muted-foreground font-normal ml-2">
                  (Copy from product label)
                </span>
              </label>
              <Textarea
                placeholder="Paste ingredient list here... e.g., Water, Glycerin, Cetearyl Alcohol, Fragrance, Methylparaben..."
                value={ingredients}
                onChange={(e) => setIngredients(e.target.value)}
                className="min-h-[150px] font-mono text-sm"
              />
              <p className="text-xs text-muted-foreground mt-1">
                {ingredients.length} characters
              </p>
            </div>
            
            <div className="flex gap-2">
              <Button 
                onClick={analyzeIngredients} 
                disabled={isAnalyzing || !ingredients.trim()}
                className="gap-2"
              >
                {isAnalyzing ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    Analyzing with TOBIN...
                  </>
                ) : (
                  <>
                    <Search className="h-4 w-4" />
                    Analyze with TOBIN
                  </>
                )}
              </Button>
              <Button 
                variant="outline" 
                onClick={handleReset}
                className="gap-2"
              >
                <RotateCcw className="h-4 w-4" />
                Reset
              </Button>
            </div>
          </CardContent>
        </Card>
        
        {/* TOBIN Score Visual */}
        {hasScores && (
          <Card className="border-2 border-primary/30">
            <CardHeader>
              <CardTitle className="flex items-center justify-between">
                <span className="flex items-center gap-2">
                  📊 TOBIN Score Summary
                </span>
                <Badge className={`${getOverallRating(overallScore).bg} ${getOverallRating(overallScore).color} border-0`}>
                  {getOverallRating(overallScore).label}: {overallScore}/50
                </Badge>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              {tobinCategories.map((cat) => {
                const score = tobinScores[cat.key] || 0;
                return (
                  <div key={cat.key} className="space-y-1">
                    <div className="flex items-center justify-between text-sm">
                      <div className="flex items-center gap-2">
                        <cat.icon className={`h-4 w-4 ${cat.color}`} />
                        <span className="font-medium">{cat.key} - {cat.name}</span>
                      </div>
                      <span className={`font-bold ${score > 5 ? 'text-red-600' : score > 3 ? 'text-yellow-600' : 'text-green-600'}`}>
                        {score}/10
                      </span>
                    </div>
                    <Progress value={score * 10} className="h-2" />
                  </div>
                );
              })}
            </CardContent>
          </Card>
        )}

        {/* Results Section */}
        {(analysis || isAnalyzing) && (
          <Card className="border-2 border-primary/20">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-2">
                  <Shield className="h-5 w-5 text-primary" />
                  Detailed TOBIN Analysis
                </CardTitle>
                {analysis && !isAnalyzing && (
                  <Button variant="ghost" size="sm" onClick={handleCopyAnalysis} className="gap-2">
                    <Copy className="h-4 w-4" />
                    Copy
                  </Button>
                )}
              </div>
            </CardHeader>
            <CardContent>
              {isAnalyzing && !analysis && (
                <div className="flex items-center gap-3 text-muted-foreground">
                  <Loader2 className="h-5 w-5 animate-spin" />
                  Analyzing ingredients with TOBIN framework...
                </div>
              )}
              
              {analysis && (
                <div className="prose prose-sm max-w-none dark:prose-invert">
                  {renderAnalysis(analysis)}
                </div>
              )}
            </CardContent>
          </Card>
        )}
        
        {/* Info Cards */}
        {!analysis && !isAnalyzing && (
          <div className="grid sm:grid-cols-2 gap-4">
            <Card>
              <CardContent className="pt-6">
                <AlertTriangle className="h-8 w-8 text-category-warning mb-3" />
                <h3 className="font-semibold mb-2">What TOBIN Analyzes</h3>
                <ul className="text-sm text-muted-foreground space-y-1">
                  <li>• <strong>T</strong>oxins: Heavy metals, PFAS, pesticides</li>
                  <li>• <strong>O</strong>xidative: Pro-oxidants, UV reactives</li>
                  <li>• <strong>B</strong>iological: Endocrine disruptors, xenoestrogens</li>
                  <li>• <strong>I</strong>nflammatory: Allergens, sensitizers</li>
                  <li>• <strong>N</strong>utrient: Mineral chelators, absorption blockers</li>
                </ul>
              </CardContent>
            </Card>
            
            <Card>
              <CardContent className="pt-6">
                <Shield className="h-8 w-8 text-category-safe mb-3" />
                <h3 className="font-semibold mb-2">Score Interpretation</h3>
                <ul className="text-sm text-muted-foreground space-y-1">
                  <li>• <span className="text-green-600 font-medium">0-10/50</span>: Excellent - Very safe</li>
                  <li>• <span className="text-emerald-600 font-medium">11-20/50</span>: Good - Minor concerns</li>
                  <li>• <span className="text-yellow-600 font-medium">21-30/50</span>: Moderate - Use caution</li>
                  <li>• <span className="text-orange-600 font-medium">31-40/50</span>: High - Consider alternatives</li>
                  <li>• <span className="text-red-600 font-medium">41-50/50</span>: Severe - Avoid if possible</li>
                </ul>
              </CardContent>
            </Card>
          </div>
        )}
        
        {/* Disclaimer */}
        <div className="bg-muted/50 rounded-lg p-4 text-sm text-muted-foreground">
          <p>
            <strong>Disclaimer:</strong> The TOBIN score provides educational guidance based on scientific 
            research and regulatory data. Individual sensitivities vary. This analysis is not medical advice. 
            Always patch test new products and consult healthcare professionals for specific concerns.
          </p>
        </div>
      </main>
    </div>
  );
};

export default IngredientScanner;
