import { useState, useMemo } from "react";
import { useNavigate } from "react-router-dom";
import { 
  Search, 
  ArrowLeft, 
  AlertTriangle, 
  Leaf, 
  Info, 
  ChevronDown, 
  ChevronUp,
  ShieldAlert,
  Beaker,
  Apple,
  Filter,
  FileDown,
  TrendingUp,
  ChefHat
} from "lucide-react";
import { generatePhytoestrogenPdf } from "@/utils/phytoestrogenPdfExport";
import { PhytoestrogenTracker } from "@/components/PhytoestrogenTracker";
import { PhytoestrogenFreeRecipes } from "@/components/PhytoestrogenFreeRecipes";
import { Input } from "@/components/ui/input";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/ui/accordion";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { 
  phytoestrogensDatabase, 
  getPhytoestrogensInFood,
  getPhytoestrogensByClass,
  getPhytoestrogensByPotency,
  PhytoestrogensData
} from "@/data/chemicalDatabase";
import { cn } from "@/lib/utils";

const potencyConfig = {
  weak: { label: "Weak", color: "bg-yellow-500/20 text-yellow-700 border-yellow-500/30" },
  moderate: { label: "Moderate", color: "bg-orange-500/20 text-orange-700 border-orange-500/30" },
  strong: { label: "Strong", color: "bg-red-500/20 text-red-700 border-red-500/30" },
};

const classConfig: Record<string, { label: string; icon: typeof Leaf }> = {
  isoflavone: { label: "Isoflavones", icon: Leaf },
  lignan: { label: "Lignans", icon: Leaf },
  coumestan: { label: "Coumestans", icon: Leaf },
  stilbene: { label: "Stilbenes", icon: Leaf },
  flavonoid: { label: "Flavonoids", icon: Leaf },
  mycoestrogen: { label: "Mycoestrogens", icon: ShieldAlert },
  other: { label: "Other", icon: Beaker },
};

export default function PhytoestrogenScanner() {
  const navigate = useNavigate();
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedClass, setSelectedClass] = useState<string>("all");
  const [selectedPotency, setSelectedPotency] = useState<string>("all");
  const [expandedItem, setExpandedItem] = useState<string | null>(null);

  // Search results for food search
  const foodSearchResults = useMemo(() => {
    if (searchQuery.length < 2) return [];
    return getPhytoestrogensInFood(searchQuery);
  }, [searchQuery]);

  // Filtered database for browsing
  const filteredDatabase = useMemo(() => {
    let results = [...phytoestrogensDatabase];
    
    if (selectedClass !== "all") {
      results = results.filter(p => p.class === selectedClass);
    }
    
    if (selectedPotency !== "all") {
      results = results.filter(p => p.potency === selectedPotency);
    }
    
    return results.sort((a, b) => {
      const potencyOrder = { strong: 0, moderate: 1, weak: 2 };
      return potencyOrder[a.potency] - potencyOrder[b.potency];
    });
  }, [selectedClass, selectedPotency]);

  // Group by class for display
  const groupedByClass = useMemo(() => {
    const groups: Record<string, PhytoestrogensData[]> = {};
    filteredDatabase.forEach(p => {
      if (!groups[p.class]) groups[p.class] = [];
      groups[p.class].push(p);
    });
    return groups;
  }, [filteredDatabase]);

  // Stats
  const stats = useMemo(() => ({
    total: phytoestrogensDatabase.length,
    strong: phytoestrogensDatabase.filter(p => p.potency === "strong").length,
    moderate: phytoestrogensDatabase.filter(p => p.potency === "moderate").length,
    weak: phytoestrogensDatabase.filter(p => p.potency === "weak").length,
  }), []);

  const renderPhytoestrogenCard = (phyto: PhytoestrogensData, showFoodSources = true) => {
    const isExpanded = expandedItem === phyto.id;
    const config = potencyConfig[phyto.potency];
    const classInfo = classConfig[phyto.class];

    return (
      <Card 
        key={phyto.id} 
        className={cn(
          "transition-all cursor-pointer hover:shadow-md",
          isExpanded && "ring-2 ring-primary/50"
        )}
        onClick={() => setExpandedItem(isExpanded ? null : phyto.id)}
      >
        <CardHeader className="p-3 sm:p-4 pb-2">
          <div className="flex items-start justify-between gap-2">
            <div className="flex-1 min-w-0">
              <CardTitle className="text-sm sm:text-base flex items-center gap-2 flex-wrap">
                <span className="truncate">{phyto.name}</span>
                <Badge variant="outline" className={cn("text-[10px] sm:text-xs shrink-0", config.color)}>
                  {config.label}
                </Badge>
              </CardTitle>
              <CardDescription className="text-xs mt-1 flex items-center gap-1">
                <classInfo.icon className="h-3 w-3" />
                {classInfo.label}
              </CardDescription>
            </div>
            <div className="flex items-center gap-1">
              <Badge variant="destructive" className="text-[10px] shrink-0">
                AVOID
              </Badge>
              {isExpanded ? (
                <ChevronUp className="h-4 w-4 text-muted-foreground" />
              ) : (
                <ChevronDown className="h-4 w-4 text-muted-foreground" />
              )}
            </div>
          </div>
        </CardHeader>
        
        {isExpanded && (
          <CardContent className="p-3 sm:p-4 pt-0 space-y-3">
            {showFoodSources && (
              <div>
                <p className="text-xs font-medium text-muted-foreground mb-1.5 flex items-center gap-1">
                  <Apple className="h-3 w-3" />
                  Found in:
                </p>
                <div className="flex flex-wrap gap-1">
                  {phyto.foodSources.map((source, idx) => (
                    <Badge key={idx} variant="secondary" className="text-[10px]">
                      {source}
                    </Badge>
                  ))}
                </div>
              </div>
            )}
            
            <div>
              <p className="text-xs font-medium text-muted-foreground mb-1.5 flex items-center gap-1">
                <AlertTriangle className="h-3 w-3" />
                Health Concerns:
              </p>
              <ul className="space-y-1">
                {phyto.healthConcerns.map((concern, idx) => (
                  <li key={idx} className="text-xs text-muted-foreground flex items-start gap-1.5">
                    <span className="text-destructive mt-0.5">•</span>
                    {concern}
                  </li>
                ))}
              </ul>
            </div>
            
            {phyto.notes && (
              <div className="p-2 bg-muted/50 rounded-lg">
                <p className="text-xs text-muted-foreground flex items-start gap-1.5">
                  <Info className="h-3 w-3 mt-0.5 shrink-0" />
                  {phyto.notes}
                </p>
              </div>
            )}
          </CardContent>
        )}
      </Card>
    );
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="sticky top-0 z-40 bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60 border-b">
        <div className="container max-w-4xl mx-auto px-4 py-3">
          <div className="flex items-center gap-3">
            <Button variant="ghost" size="icon" onClick={() => navigate(-1)}>
              <ArrowLeft className="h-5 w-5" />
            </Button>
            <div>
              <h1 className="text-lg sm:text-xl font-bold flex items-center gap-2">
                <Leaf className="h-5 w-5 text-destructive" />
                Phytoestrogen Scanner
              </h1>
              <p className="text-xs text-muted-foreground">
                Check foods for estrogenic compounds
              </p>
            </div>
          </div>
        </div>
      </header>

      <main className="container max-w-4xl mx-auto px-4 py-4 space-y-4">
        {/* Stats Overview */}
        <Card className="bg-gradient-to-r from-destructive/10 to-orange-500/10 border-destructive/20">
          <CardContent className="p-4">
            <div className="flex items-center justify-between gap-3 mb-3">
              <div className="flex items-center gap-3">
                <ShieldAlert className="h-6 w-6 text-destructive" />
                <div>
                  <h2 className="font-semibold text-sm">All Phytoestrogens: AVOID</h2>
                  <p className="text-xs text-muted-foreground">
                    Database of {stats.total} estrogenic compounds found in food
                  </p>
                </div>
              </div>
              <Button 
                variant="outline" 
                size="sm" 
                onClick={generatePhytoestrogenPdf}
                className="gap-2 shrink-0"
              >
                <FileDown className="h-4 w-4" />
                <span className="hidden sm:inline">Export PDF</span>
              </Button>
            </div>
            <div className="grid grid-cols-3 gap-2 text-center">
              <div className="p-2 bg-background/50 rounded-lg">
                <p className="text-lg font-bold text-destructive">{stats.strong}</p>
                <p className="text-[10px] text-muted-foreground">Strong</p>
              </div>
              <div className="p-2 bg-background/50 rounded-lg">
                <p className="text-lg font-bold text-category-warning">{stats.moderate}</p>
                <p className="text-[10px] text-muted-foreground">Moderate</p>
              </div>
              <div className="p-2 bg-background/50 rounded-lg">
                <p className="text-lg font-bold text-yellow-600">{stats.weak}</p>
                <p className="text-[10px] text-muted-foreground">Weak</p>
              </div>
            </div>
          </CardContent>
        </Card>

        <Tabs defaultValue="tracker" className="w-full">
          <TabsList className="grid w-full grid-cols-4">
            <TabsTrigger value="tracker" className="gap-1.5">
              <TrendingUp className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Tracker</span>
              <span className="sm:hidden">Track</span>
            </TabsTrigger>
            <TabsTrigger value="recipes" className="gap-1.5">
              <ChefHat className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Recipes</span>
              <span className="sm:hidden">Cook</span>
            </TabsTrigger>
            <TabsTrigger value="search" className="gap-1.5">
              <Search className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Search Food</span>
              <span className="sm:hidden">Search</span>
            </TabsTrigger>
            <TabsTrigger value="browse" className="gap-1.5">
              <Beaker className="h-3.5 w-3.5" />
              <span className="hidden sm:inline">Browse</span>
              <span className="sm:hidden">DB</span>
            </TabsTrigger>
          </TabsList>

          {/* Tracker Tab */}
          <TabsContent value="tracker" className="mt-4">
            <PhytoestrogenTracker />
          </TabsContent>

          {/* Recipes Tab */}
          <TabsContent value="recipes" className="mt-4">
            <PhytoestrogenFreeRecipes />
          </TabsContent>

          {/* Search Tab */}
          <TabsContent value="search" className="mt-4 space-y-4">
            <div className="relative">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Enter a food name (e.g., soy, flaxseed, beer)..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pl-10"
              />
            </div>

            {searchQuery.length >= 2 ? (
              foodSearchResults.length > 0 ? (
                <div className="space-y-3">
                  <div className="flex items-center justify-between">
                    <p className="text-sm text-muted-foreground">
                      Found <span className="font-semibold text-foreground">{foodSearchResults.length}</span> phytoestrogens in "{searchQuery}"
                    </p>
                    <Badge variant="destructive" className="text-xs">
                      Avoid All
                    </Badge>
                  </div>
                  
                  <ScrollArea className="h-[500px]">
                    <div className="space-y-2 pr-4">
                      {foodSearchResults
                        .sort((a, b) => {
                          const order = { strong: 0, moderate: 1, weak: 2 };
                          return order[a.potency] - order[b.potency];
                        })
                        .map(phyto => renderPhytoestrogenCard(phyto, false))}
                    </div>
                  </ScrollArea>
                </div>
              ) : (
                <Card className="bg-green-500/10 border-green-500/20">
                  <CardContent className="p-6 text-center">
                    <Leaf className="h-10 w-10 mx-auto text-green-600 mb-3" />
                    <h3 className="font-semibold text-green-700 mb-1">No Phytoestrogens Found</h3>
                    <p className="text-sm text-muted-foreground">
                      "{searchQuery}" doesn't match any known phytoestrogen sources in our database
                    </p>
                  </CardContent>
                </Card>
              )
            ) : (
              <Card className="bg-muted/30">
                <CardContent className="p-6 text-center">
                  <Search className="h-10 w-10 mx-auto text-muted-foreground mb-3" />
                  <h3 className="font-semibold mb-1">Search for Foods</h3>
                  <p className="text-sm text-muted-foreground">
                    Enter a food name to see which phytoestrogens it contains
                  </p>
                  <div className="flex flex-wrap justify-center gap-2 mt-4">
                    {["Soy", "Flaxseed", "Beer", "Tofu", "Green tea"].map(food => (
                      <Badge 
                        key={food}
                        variant="outline" 
                        className="cursor-pointer hover:bg-primary/10"
                        onClick={() => setSearchQuery(food)}
                      >
                        {food}
                      </Badge>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </TabsContent>

          {/* Browse Tab */}
          <TabsContent value="browse" className="mt-4 space-y-4">
            {/* Filters */}
            <div className="flex flex-col sm:flex-row gap-2">
              <Select value={selectedClass} onValueChange={setSelectedClass}>
                <SelectTrigger className="w-full sm:w-[180px]">
                  <Filter className="h-3.5 w-3.5 mr-2" />
                  <SelectValue placeholder="Filter by class" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Classes</SelectItem>
                  {Object.entries(classConfig).map(([key, { label }]) => (
                    <SelectItem key={key} value={key}>{label}</SelectItem>
                  ))}
                </SelectContent>
              </Select>

              <Select value={selectedPotency} onValueChange={setSelectedPotency}>
                <SelectTrigger className="w-full sm:w-[180px]">
                  <AlertTriangle className="h-3.5 w-3.5 mr-2" />
                  <SelectValue placeholder="Filter by potency" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All Potencies</SelectItem>
                  <SelectItem value="strong">Strong</SelectItem>
                  <SelectItem value="moderate">Moderate</SelectItem>
                  <SelectItem value="weak">Weak</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <p className="text-sm text-muted-foreground">
              Showing {filteredDatabase.length} of {stats.total} compounds
            </p>

            <ScrollArea className="h-[500px]">
              <Accordion type="single" collapsible className="space-y-2">
                {Object.entries(groupedByClass).map(([className, items]) => {
                  const classInfo = classConfig[className];
                  const strongCount = items.filter(i => i.potency === "strong").length;
                  const moderateCount = items.filter(i => i.potency === "moderate").length;
                  
                  return (
                    <AccordionItem key={className} value={className} className="border rounded-lg px-3">
                      <AccordionTrigger className="hover:no-underline py-3">
                        <div className="flex items-center gap-2 text-left">
                          <classInfo.icon className="h-4 w-4 text-primary" />
                          <span className="font-medium">{classInfo.label}</span>
                          <Badge variant="secondary" className="text-xs ml-2">
                            {items.length}
                          </Badge>
                          {strongCount > 0 && (
                            <Badge variant="destructive" className="text-[10px]">
                              {strongCount} strong
                            </Badge>
                          )}
                        </div>
                      </AccordionTrigger>
                      <AccordionContent className="pb-3">
                        <div className="space-y-2 pt-2">
                          {items.map(phyto => renderPhytoestrogenCard(phyto))}
                        </div>
                      </AccordionContent>
                    </AccordionItem>
                  );
                })}
              </Accordion>
            </ScrollArea>
          </TabsContent>
        </Tabs>

        {/* Educational Info */}
        <Card className="mt-4">
          <CardHeader className="p-4 pb-2">
            <CardTitle className="text-sm flex items-center gap-2">
              <Info className="h-4 w-4 text-primary" />
              About Phytoestrogens
            </CardTitle>
          </CardHeader>
          <CardContent className="p-4 pt-2 text-xs text-muted-foreground space-y-2">
            <p>
              <strong>Phytoestrogens</strong> are plant-derived compounds that can bind to estrogen receptors 
              in the body and mimic or modulate estrogenic activity.
            </p>
            <p>
              <strong>Why Avoid?</strong> While often marketed as "natural" and beneficial, phytoestrogens 
              can disrupt hormonal balance, affect thyroid function, and may be problematic for individuals 
              with estrogen-sensitive conditions.
            </p>
            <p>
              <strong>Potency varies:</strong> Strong phytoestrogens like genistein (soy) and coumestrol (alfalfa) 
              have significant estrogenic effects, while weak ones like quercetin have minimal activity but 
              can accumulate with regular consumption.
            </p>
          </CardContent>
        </Card>
      </main>
    </div>
  );
}