import { useEffect } from 'react';
import { useNavigate, Link } from 'react-router-dom';
import { useAuth } from '@/hooks/useAuth';
import { useFavorites } from '@/contexts/FavoritesContext';
import { useDietaryPreferences } from '@/hooks/useDietaryPreferences';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Label } from '@/components/ui/label';
import { Switch } from '@/components/ui/switch';
import { Slider } from '@/components/ui/slider';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { ArrowLeft, Heart, Settings, LogOut, Loader2, Utensils, Sparkles, Activity, User } from 'lucide-react';
import { foodDatabase } from '@/data/foodDatabase';
import { cosmeticsDatabase } from '@/data/cosmeticsDatabase';
import HealthProfileSettings from '@/components/HealthProfileSettings';
import { ProfileCustomization } from '@/components/ProfileCustomization';

const Profile = () => {
  const navigate = useNavigate();
  const { user, loading: authLoading, signOut } = useAuth();
  const { favorites, loading: favoritesLoading } = useFavorites();
  const { preferences, loading: preferencesLoading, updatePreferences } = useDietaryPreferences();

  useEffect(() => {
    if (!authLoading && !user) {
      navigate('/auth');
    }
  }, [user, authLoading, navigate]);

  const handleSignOut = async () => {
    await signOut();
    navigate('/');
  };

  if (authLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50">
        <Loader2 className="h-8 w-8 animate-spin text-emerald-600" />
      </div>
    );
  }

  if (!user) return null;

  const favoriteFoods = favorites
    .filter(f => f.item_type === 'food')
    .map(f => foodDatabase.find(food => food.id === f.item_id))
    .filter(Boolean);

  const favoriteCosmetics = favorites
    .filter(f => f.item_type === 'cosmetic')
    .map(f => cosmeticsDatabase.find(c => c.id === f.item_id))
    .filter(Boolean);

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50">
      <div className="container mx-auto px-4 py-8 max-w-4xl">
        <div className="flex items-center justify-between mb-8">
          <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
            <ArrowLeft className="h-4 w-4" />
            Back to Home
          </Link>
          <Button variant="outline" onClick={handleSignOut} className="gap-2">
            <LogOut className="h-4 w-4" />
            Sign Out
          </Button>
        </div>

        <div className="mb-8">
          <h1 className="text-3xl font-bold text-foreground mb-2">My Profile</h1>
          <p className="text-muted-foreground">{user.email}</p>
        </div>

        <Tabs defaultValue="profile" className="space-y-6">
          <TabsList className="flex-wrap">
            <TabsTrigger value="profile" className="gap-2">
              <User className="h-4 w-4" />
              Profile
            </TabsTrigger>
            <TabsTrigger value="favorites" className="gap-2">
              <Heart className="h-4 w-4" />
              Favorites
            </TabsTrigger>
            <TabsTrigger value="health" className="gap-2">
              <Activity className="h-4 w-4" />
              Health
            </TabsTrigger>
            <TabsTrigger value="preferences" className="gap-2">
              <Settings className="h-4 w-4" />
              Preferences
            </TabsTrigger>
          </TabsList>

          <TabsContent value="profile">
            <ProfileCustomization />
          </TabsContent>

          <TabsContent value="favorites" className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Utensils className="h-5 w-5" />
                  Favorite Foods
                </CardTitle>
                <CardDescription>
                  Your saved food items
                </CardDescription>
              </CardHeader>
              <CardContent>
                {favoritesLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                  </div>
                ) : favoriteFoods.length === 0 ? (
                  <p className="text-muted-foreground text-center py-8">
                    No favorite foods yet. Browse foods and click the heart icon to save them!
                  </p>
                ) : (
                  <div className="grid gap-3">
                    {favoriteFoods.map((food) => (
                      <div key={food!.id} className="flex items-center gap-3 p-3 rounded-lg bg-muted/50">
                        <span className="text-2xl">{food!.emoji}</span>
                        <div className="flex-1">
                          <p className="font-medium">{food!.name}</p>
                          <p className="text-sm text-muted-foreground">{food!.category}</p>
                        </div>
                        <Badge variant="outline">GI: {food!.glycogenicIndex}</Badge>
                        <Badge variant="outline">LI: {food!.lipogenicIndex}</Badge>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle className="flex items-center gap-2">
                  <Sparkles className="h-5 w-5" />
                  Favorite Cosmetics
                </CardTitle>
                <CardDescription>
                  Your saved cosmetic items
                </CardDescription>
              </CardHeader>
              <CardContent>
                {favoritesLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                  </div>
                ) : favoriteCosmetics.length === 0 ? (
                  <p className="text-muted-foreground text-center py-8">
                    No favorite cosmetics yet. Browse cosmetics and click the heart icon to save them!
                  </p>
                ) : (
                  <div className="grid gap-3">
                    {favoriteCosmetics.map((cosmetic) => (
                      <div key={cosmetic!.id} className="flex items-center gap-3 p-3 rounded-lg bg-muted/50">
                        <span className="text-2xl">{cosmetic!.emoji}</span>
                        <div className="flex-1">
                          <p className="font-medium">{cosmetic!.name}</p>
                          <p className="text-sm text-muted-foreground">{cosmetic!.category}</p>
                        </div>
                        <Badge 
                          variant="outline"
                          className={
                            cosmetic!.riskLevel <= 2 ? 'border-green-500 text-green-700' :
                            cosmetic!.riskLevel <= 3 ? 'border-yellow-500 text-yellow-700' :
                            'border-red-500 text-red-700'
                          }
                        >
                          Risk: {cosmetic!.riskLevel}/5
                        </Badge>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>

          <TabsContent value="health">
            <HealthProfileSettings />
          </TabsContent>

          <TabsContent value="preferences">
            <Card>
              <CardHeader>
                <CardTitle>Dietary Preferences</CardTitle>
                <CardDescription>
                  Set your dietary requirements and these will be applied when browsing
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {preferencesLoading ? (
                  <div className="flex justify-center py-8">
                    <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                  </div>
                ) : (
                  <>
                    <div className="grid gap-4 md:grid-cols-3">
                      <div className="space-y-2">
                        <Label>Halal Preference</Label>
                        <Select 
                          value={preferences.halal_preference}
                          onValueChange={(value) => updatePreferences({ halal_preference: value })}
                        >
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="all">All Foods</SelectItem>
                            <SelectItem value="halal">Halal Only</SelectItem>
                            <SelectItem value="mushbooh">Include Mushbooh</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>

                      <div className="space-y-2">
                        <Label>Kosher Preference</Label>
                        <Select 
                          value={preferences.kosher_preference}
                          onValueChange={(value) => updatePreferences({ kosher_preference: value })}
                        >
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="all">All Foods</SelectItem>
                            <SelectItem value="kosher">Kosher Only</SelectItem>
                            <SelectItem value="pareve">Pareve Only</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>

                      <div className="space-y-2">
                        <Label>Organic Preference</Label>
                        <Select 
                          value={preferences.organic_preference}
                          onValueChange={(value) => updatePreferences({ organic_preference: value })}
                        >
                          <SelectTrigger>
                            <SelectValue />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="all">All Foods</SelectItem>
                            <SelectItem value="organic">Organic Only</SelectItem>
                            <SelectItem value="transitional">Include Transitional</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>
                    </div>

                    <div className="space-y-4 pt-4 border-t">
                      <div className="flex items-center justify-between">
                        <div>
                          <Label>Avoid Xenoestrogens</Label>
                          <p className="text-sm text-muted-foreground">
                            Filter out cosmetics with xenoestrogenic compounds
                          </p>
                        </div>
                        <Switch 
                          checked={preferences.avoid_xenoestrogens}
                          onCheckedChange={(checked) => updatePreferences({ avoid_xenoestrogens: checked })}
                        />
                      </div>

                      <div className="space-y-3">
                        <div className="flex items-center justify-between">
                          <Label>Maximum Cosmetic Risk Level</Label>
                          <Badge variant="outline">{preferences.max_risk_level}/5</Badge>
                        </div>
                        <Slider
                          value={[preferences.max_risk_level]}
                          onValueChange={([value]) => updatePreferences({ max_risk_level: value })}
                          max={5}
                          min={1}
                          step={1}
                        />
                        <p className="text-sm text-muted-foreground">
                          Only show cosmetics with risk level at or below this threshold
                        </p>
                      </div>
                    </div>
                  </>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </div>
    </div>
  );
};

export default Profile;
