import { useState } from "react";
import { Link } from "react-router-dom";
import { ArrowLeft, ChefHat, Sparkles, Users, Leaf, Clock, RefreshCw, Copy, Check, BookmarkPlus } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";
import { ScrollArea } from "@/components/ui/scroll-area";
import { useRecipeGenerator } from "@/hooks/useRecipeGenerator";
import { useSavedRecipes } from "@/hooks/useSavedRecipes";
import { RecipeCookbook } from "@/components/RecipeCookbook";
import { toast } from "sonner";

const mealTypeLabels: Record<string, string> = {
  daily: "Full Day Plan",
  breakfast: "Breakfast",
  lunch: "Lunch",
  dinner: "Dinner",
  snacks: "Healthy Snacks",
  weekly: "Weekly Prep",
};

export default function Recipes() {
  const { content, isGenerating, error, generateRecipes, reset } = useRecipeGenerator();
  const { saveRecipe } = useSavedRecipes();
  const [mealType, setMealType] = useState("daily");
  const [servings, setServings] = useState("2");
  const [dietaryRestrictions, setDietaryRestrictions] = useState("");
  const [preferences, setPreferences] = useState("");
  const [copied, setCopied] = useState(false);
  const [saving, setSaving] = useState(false);

  const handleGenerate = () => {
    generateRecipes({
      mealType,
      servings: parseInt(servings),
      dietaryRestrictions: dietaryRestrictions || undefined,
      preferences: preferences || undefined,
    });
  };

  const handleCopy = async () => {
    if (content) {
      await navigator.clipboard.writeText(content);
      setCopied(true);
      toast.success("Recipes copied to clipboard!");
      setTimeout(() => setCopied(false), 2000);
    }
  };

  const handleNewPlan = () => {
    reset();
  };

  const handleSaveRecipe = async () => {
    if (!content) return;
    setSaving(true);
    const title = `${mealTypeLabels[mealType]} - ${new Date().toLocaleDateString()}`;
    await saveRecipe(title, content, mealType, parseInt(servings));
    setSaving(false);
  };

  return (
    <div className="min-h-screen bg-background">
      {/* Header */}
      <header className="sticky top-0 z-40 bg-background/95 backdrop-blur-md border-b border-border">
        <div className="max-w-4xl mx-auto px-3 sm:px-4 py-3 sm:py-4 flex items-center gap-3 sm:gap-4">
          <Link to="/">
            <Button variant="ghost" size="icon" className="h-8 w-8 sm:h-9 sm:w-9">
              <ArrowLeft className="h-4 w-4 sm:h-5 sm:w-5" />
            </Button>
          </Link>
          <div className="flex items-center gap-2">
            <div className="p-1.5 sm:p-2 rounded-lg bg-gradient-to-br from-primary/20 to-primary/10">
              <ChefHat className="h-5 w-5 sm:h-6 sm:w-6 text-primary" />
            </div>
            <div>
              <h1 className="text-lg sm:text-xl font-display font-bold">Recipe Generator</h1>
              <p className="text-xs sm:text-sm text-muted-foreground">Low LI & GI* meal plans</p>
            </div>
          </div>
        </div>
      </header>

      <main className="max-w-4xl mx-auto px-3 sm:px-4 py-4 sm:py-6 space-y-4 sm:space-y-6 pb-20 md:pb-8">
        {/* Info Card */}
        <Card className="bg-gradient-to-br from-category-therapeutic/10 to-category-safe/10 border-category-therapeutic/20">
          <CardContent className="p-3 sm:p-4">
            <div className="flex items-start gap-3">
              <div className="p-2 rounded-full bg-category-therapeutic/20 shrink-0">
                <Sparkles className="h-4 w-4 sm:h-5 sm:w-5 text-category-therapeutic" />
              </div>
              <div>
                <h3 className="font-semibold text-sm sm:text-base mb-1">AI-Powered Metabolic Recipes</h3>
                <p className="text-xs sm:text-sm text-muted-foreground">
                  Generate delicious meal plans using therapeutic foods with low Lipogenic Index (LI) and 
                  low Glycemic Index (GI*) to support fat loss and metabolic health.
                </p>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Configuration Form */}
        {!content && !isGenerating && (
          <Card>
            <CardHeader className="p-4 sm:p-6 pb-3 sm:pb-4">
              <CardTitle className="text-base sm:text-lg">Customize Your Meal Plan</CardTitle>
              <CardDescription className="text-xs sm:text-sm">
                Set your preferences for personalized recipes
              </CardDescription>
            </CardHeader>
            <CardContent className="p-4 sm:p-6 pt-0 space-y-4">
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="mealType" className="text-sm flex items-center gap-2">
                    <Clock className="h-3.5 w-3.5 text-muted-foreground" />
                    Meal Plan Type
                  </Label>
                  <Select value={mealType} onValueChange={setMealType}>
                    <SelectTrigger id="mealType">
                      <SelectValue placeholder="Select type" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="daily">Full Day Plan</SelectItem>
                      <SelectItem value="breakfast">Breakfast Focus</SelectItem>
                      <SelectItem value="lunch">Lunch Focus</SelectItem>
                      <SelectItem value="dinner">Dinner Focus</SelectItem>
                      <SelectItem value="snacks">Healthy Snacks</SelectItem>
                      <SelectItem value="weekly">Weekly Prep</SelectItem>
                    </SelectContent>
                  </Select>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="servings" className="text-sm flex items-center gap-2">
                    <Users className="h-3.5 w-3.5 text-muted-foreground" />
                    Servings
                  </Label>
                  <Select value={servings} onValueChange={setServings}>
                    <SelectTrigger id="servings">
                      <SelectValue placeholder="Select servings" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="1">1 serving</SelectItem>
                      <SelectItem value="2">2 servings</SelectItem>
                      <SelectItem value="4">4 servings</SelectItem>
                      <SelectItem value="6">6 servings</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>

              <div className="space-y-2">
                <Label htmlFor="dietary" className="text-sm flex items-center gap-2">
                  <Leaf className="h-3.5 w-3.5 text-muted-foreground" />
                  Dietary Restrictions (optional)
                </Label>
                <Input
                  id="dietary"
                  placeholder="e.g., dairy-free, nut-free, pescatarian..."
                  value={dietaryRestrictions}
                  onChange={(e) => setDietaryRestrictions(e.target.value)}
                  className="text-sm"
                />
              </div>

              <div className="space-y-2">
                <Label htmlFor="preferences" className="text-sm">
                  Additional Preferences (optional)
                </Label>
                <Textarea
                  id="preferences"
                  placeholder="e.g., quick prep time, Mediterranean-style, high protein..."
                  value={preferences}
                  onChange={(e) => setPreferences(e.target.value)}
                  className="text-sm min-h-[80px] resize-none"
                />
              </div>

              <Button 
                onClick={handleGenerate} 
                className="w-full gap-2"
                size="lg"
              >
                <Sparkles className="h-4 w-4" />
                Generate Meal Plan
              </Button>
            </CardContent>
          </Card>
        )}

        {/* Loading State */}
        {isGenerating && !content && (
          <Card>
            <CardContent className="p-8 sm:p-12 flex flex-col items-center justify-center text-center">
              <div className="relative mb-4">
                <div className="h-16 w-16 rounded-full bg-primary/10 flex items-center justify-center">
                  <ChefHat className="h-8 w-8 text-primary animate-pulse" />
                </div>
                <div className="absolute inset-0 rounded-full border-2 border-primary/30 border-t-primary animate-spin" />
              </div>
              <h3 className="font-semibold text-base sm:text-lg mb-2">Creating Your Meal Plan</h3>
              <p className="text-sm text-muted-foreground">
                Our AI chef is crafting delicious low LI/GI* recipes for you...
              </p>
            </CardContent>
          </Card>
        )}

        {/* Error State */}
        {error && (
          <Card className="border-destructive/50 bg-destructive/5">
            <CardContent className="p-4 sm:p-6">
              <p className="text-sm text-destructive mb-3">{error}</p>
              <Button variant="outline" onClick={handleGenerate} className="gap-2">
                <RefreshCw className="h-4 w-4" />
                Try Again
              </Button>
            </CardContent>
          </Card>
        )}

        {/* Generated Content */}
        {(content || isGenerating) && content && (
          <Card>
            <CardHeader className="p-4 sm:p-6 pb-2 flex flex-row items-center justify-between gap-2">
              <div>
                <CardTitle className="text-base sm:text-lg flex items-center gap-2">
                  <ChefHat className="h-5 w-5 text-primary" />
                  Your Meal Plan
                </CardTitle>
                <CardDescription className="text-xs sm:text-sm">
                  {isGenerating ? "Generating..." : "Ready to cook!"}
                </CardDescription>
              </div>
              <div className="flex gap-2 flex-wrap">
                <Button 
                  variant="default" 
                  size="sm" 
                  onClick={handleSaveRecipe}
                  disabled={isGenerating || saving}
                  className="gap-1.5"
                >
                  <BookmarkPlus className="h-3.5 w-3.5" />
                  <span className="hidden sm:inline">{saving ? "Saving..." : "Save to Cookbook"}</span>
                  <span className="sm:hidden">{saving ? "..." : "Save"}</span>
                </Button>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={handleCopy}
                  disabled={isGenerating}
                  className="gap-1.5"
                >
                  {copied ? <Check className="h-3.5 w-3.5" /> : <Copy className="h-3.5 w-3.5" />}
                  <span className="hidden sm:inline">{copied ? "Copied" : "Copy"}</span>
                </Button>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={handleNewPlan}
                  disabled={isGenerating}
                  className="gap-1.5"
                >
                  <RefreshCw className="h-3.5 w-3.5" />
                  <span className="hidden sm:inline">New Plan</span>
                </Button>
              </div>
            </CardHeader>
            <CardContent className="p-4 sm:p-6 pt-2">
              <ScrollArea className="h-[60vh] sm:h-[65vh]">
                <div className="prose prose-sm dark:prose-invert max-w-none pr-4">
                  <div className="whitespace-pre-wrap text-sm leading-relaxed">
                    {content}
                    {isGenerating && (
                      <span className="inline-block w-2 h-4 bg-primary/50 animate-pulse ml-0.5" />
                    )}
                  </div>
                </div>
              </ScrollArea>
            </CardContent>
          </Card>
        )}

        {/* Tips Section */}
        {!content && !isGenerating && (
          <Card className="bg-muted/30">
            <CardContent className="p-4 sm:p-6">
              <h3 className="font-semibold text-sm sm:text-base mb-3">💡 Tips for Metabolic Health</h3>
              <ul className="text-xs sm:text-sm text-muted-foreground space-y-2">
                <li className="flex items-start gap-2">
                  <span className="text-category-therapeutic">✓</span>
                  Focus on therapeutic foods with LI &lt; 0.5 for optimal fat burning
                </li>
                <li className="flex items-start gap-2">
                  <span className="text-category-safe">✓</span>
                  Combine protein with healthy fats for satiety
                </li>
                <li className="flex items-start gap-2">
                  <span className="text-primary">✓</span>
                  Include plenty of non-starchy vegetables in every meal
                </li>
                <li className="flex items-start gap-2">
                  <span className="text-primary">✓</span>
                  Avoid seed oils and processed foods for best results
                </li>
              </ul>
            </CardContent>
          </Card>
        )}

        {/* Personal Cookbook */}
        <RecipeCookbook />
      </main>
    </div>
  );
}
