import { useState, useEffect } from "react";
import { useParams, Link } from "react-router-dom";
import { ArrowLeft, ChefHat, Clock, Users, Flame, Beef, Wheat, Droplets, Leaf } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Badge } from "@/components/ui/badge";
import { Skeleton } from "@/components/ui/skeleton";
import { recipesApi } from "@/lib/apiServices";

interface NutritionalInfo {
  calories: number;
  protein: number;
  carbs: number;
  fat: number;
  fiber: number;
  sugar: number;
  sodium: number;
  highlights: string[];
  warnings: string[];
}

interface SharedRecipe {
  id: string;
  title: string;
  content: string;
  meal_type: string;
  servings: number;
  created_at: string;
  nutritional_info: NutritionalInfo | null;
}

const mealTypeLabels: Record<string, string> = {
  daily: "Full Day",
  breakfast: "Breakfast",
  lunch: "Lunch",
  dinner: "Dinner",
  snacks: "Snacks",
  weekly: "Weekly",
};

export default function SharedRecipe() {
  const { token } = useParams<{ token: string }>();
  const [recipe, setRecipe] = useState<SharedRecipe | null>(null);
  const [loading, setLoading] = useState(true);
  const [notFound, setNotFound] = useState(false);

  useEffect(() => {
    const fetchSharedRecipe = async () => {
      if (!token) {
        setNotFound(true);
        setLoading(false);
        return;
      }
      try {
        const data = await recipesApi.getShared(token);
        setRecipe({
          ...data,
          nutritional_info: data.nutritional_info as unknown as NutritionalInfo | null,
        });
      } catch (e) {
        console.error('Error fetching shared recipe:', e);
        setNotFound(true);
      }
      setLoading(false);
    };
    fetchSharedRecipe();
  }, [token]);

  if (loading) {
    return (
      <div className="min-h-screen bg-background">
        <header className="sticky top-0 z-40 bg-background/95 backdrop-blur-md border-b border-border">
          <div className="max-w-4xl mx-auto px-4 py-4 flex items-center gap-4">
            <Skeleton className="h-10 w-10 rounded-lg" />
            <div className="space-y-2">
              <Skeleton className="h-5 w-48" />
              <Skeleton className="h-4 w-32" />
            </div>
          </div>
        </header>
        <main className="max-w-4xl mx-auto px-4 py-6">
          <Card>
            <CardContent className="p-6 space-y-4">
              <Skeleton className="h-6 w-3/4" />
              <Skeleton className="h-4 w-full" />
              <Skeleton className="h-4 w-full" />
              <Skeleton className="h-4 w-2/3" />
            </CardContent>
          </Card>
        </main>
      </div>
    );
  }

  if (notFound || !recipe) {
    return (
      <div className="min-h-screen bg-background flex items-center justify-center">
        <Card className="max-w-md mx-4">
          <CardContent className="p-8 text-center">
            <ChefHat className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
            <h1 className="text-xl font-bold mb-2">Recipe Not Found</h1>
            <p className="text-muted-foreground mb-6">
              This recipe may have been removed or the link is no longer valid.
            </p>
            <Link to="/">
              <Button>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Go Home
              </Button>
            </Link>
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-background pb-20 md:pb-8">
      {/* Header */}
      <header className="sticky top-0 z-40 bg-background/95 backdrop-blur-md border-b border-border">
        <div className="max-w-4xl mx-auto px-3 sm:px-4 py-3 sm:py-4 flex items-center gap-3 sm:gap-4">
          <Link to="/">
            <Button variant="ghost" size="icon" className="h-8 w-8 sm:h-9 sm:w-9">
              <ArrowLeft className="h-4 w-4 sm:h-5 sm:w-5" />
            </Button>
          </Link>
          <div className="flex items-center gap-2">
            <div className="p-1.5 sm:p-2 rounded-lg bg-gradient-to-br from-primary/20 to-primary/10">
              <ChefHat className="h-5 w-5 sm:h-6 sm:w-6 text-primary" />
            </div>
            <div>
              <h1 className="text-lg sm:text-xl font-display font-bold">Shared Recipe</h1>
              <p className="text-xs sm:text-sm text-muted-foreground">Low LI & GI* meal plan</p>
            </div>
          </div>
        </div>
      </header>

      <main className="max-w-4xl mx-auto px-3 sm:px-4 py-4 sm:py-6 space-y-4 sm:space-y-6">
        {/* Recipe Info */}
        <Card>
          <CardHeader className="p-4 sm:p-6">
            <CardTitle className="text-lg sm:text-xl">{recipe.title}</CardTitle>
            <CardDescription className="flex items-center gap-4 mt-2">
              <span className="flex items-center gap-1">
                <Clock className="h-4 w-4" />
                {mealTypeLabels[recipe.meal_type] || recipe.meal_type}
              </span>
              <span className="flex items-center gap-1">
                <Users className="h-4 w-4" />
                {recipe.servings} servings
              </span>
            </CardDescription>
          </CardHeader>
        </Card>

        {/* Nutritional Info */}
        {recipe.nutritional_info && (
          <Card className="bg-gradient-to-br from-category-therapeutic/10 to-category-safe/10 border-category-therapeutic/20">
            <CardHeader className="p-4 pb-2">
              <CardTitle className="text-sm sm:text-base flex items-center gap-2">
                <Leaf className="h-4 w-4 text-category-therapeutic" />
                Nutritional Information
                <Badge variant="secondary" className="ml-auto text-xs">per serving</Badge>
              </CardTitle>
            </CardHeader>
            <CardContent className="p-4 pt-2">
              <div className="grid grid-cols-2 sm:grid-cols-4 gap-3 mb-4">
                <div className="text-center p-3 bg-background/50 rounded-lg">
                  <Flame className="h-5 w-5 mx-auto text-orange-500 mb-1" />
                  <p className="text-lg font-bold">{recipe.nutritional_info.calories}</p>
                  <p className="text-xs text-muted-foreground">Calories</p>
                </div>
                <div className="text-center p-3 bg-background/50 rounded-lg">
                  <Beef className="h-5 w-5 mx-auto text-red-500 mb-1" />
                  <p className="text-lg font-bold">{recipe.nutritional_info.protein}g</p>
                  <p className="text-xs text-muted-foreground">Protein</p>
                </div>
                <div className="text-center p-3 bg-background/50 rounded-lg">
                  <Wheat className="h-5 w-5 mx-auto text-amber-500 mb-1" />
                  <p className="text-lg font-bold">{recipe.nutritional_info.carbs}g</p>
                  <p className="text-xs text-muted-foreground">Carbs</p>
                </div>
                <div className="text-center p-3 bg-background/50 rounded-lg">
                  <Droplets className="h-5 w-5 mx-auto text-yellow-500 mb-1" />
                  <p className="text-lg font-bold">{recipe.nutritional_info.fat}g</p>
                  <p className="text-xs text-muted-foreground">Fat</p>
                </div>
              </div>
              {recipe.nutritional_info.highlights.length > 0 && (
                <div className="flex flex-wrap gap-2">
                  {recipe.nutritional_info.highlights.map((highlight, i) => (
                    <Badge key={i} variant="outline" className="text-xs bg-category-therapeutic/10">
                      ✓ {highlight}
                    </Badge>
                  ))}
                </div>
              )}
            </CardContent>
          </Card>
        )}

        {/* Recipe Content */}
        <Card>
          <CardContent className="p-4 sm:p-6">
            <ScrollArea className="h-[60vh]">
              <div className="prose prose-sm dark:prose-invert max-w-none pr-4">
                <div className="whitespace-pre-wrap text-sm leading-relaxed">
                  {recipe.content}
                </div>
              </div>
            </ScrollArea>
          </CardContent>
        </Card>

        {/* CTA */}
        <Card className="bg-muted/30">
          <CardContent className="p-4 sm:p-6 text-center">
            <ChefHat className="h-10 w-10 mx-auto text-primary mb-3" />
            <h3 className="font-semibold mb-2">Want to create your own meal plans?</h3>
            <p className="text-sm text-muted-foreground mb-4">
              Generate personalized low LI/GI* recipes for metabolic health
            </p>
            <Link to="/recipes">
              <Button>Try Recipe Generator</Button>
            </Link>
          </CardContent>
        </Card>
      </main>
    </div>
  );
}
