import { useEffect, useMemo } from 'react';
import { useNavigate, Link } from 'react-router-dom';
import { useAuth } from '@/hooks/useAuth';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { useScanHistory } from '@/hooks/useScanHistory';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { ArrowLeft, ShoppingCart, Trash2, Loader2, Plus, Minus, ScanLine, Activity, Sparkles, Package } from 'lucide-react';
import { foodDatabase } from '@/data/foodDatabase';
import { cosmeticsDatabase } from '@/data/cosmeticsDatabase';
import { UserMenu } from '@/components/UserMenu';

interface ScannedProductItem {
  id: string;
  item_id: string;
  item_type: 'food' | 'cosmetic';
  quantity: number;
  is_purchased: boolean;
  product_name: string;
  image_url?: string | null;
}

const ShoppingList = () => {
  const navigate = useNavigate();
  const { user, loading: authLoading } = useAuth();
  const { items, loading, togglePurchased, updateQuantity, removeItem, clearPurchased } = useShoppingList();
  const { history: scanHistory } = useScanHistory();

  useEffect(() => {
    if (!authLoading && !user) {
      navigate('/auth');
    }
  }, [user, authLoading, navigate]);

  // Resolve products - check local DB first, then scan history
  const resolvedItems = useMemo(() => {
    const foods: Array<typeof items[0] & { details: typeof foodDatabase[0] }> = [];
    const cosmetics: Array<typeof items[0] & { details: typeof cosmeticsDatabase[0] }> = [];
    const scannedProducts: ScannedProductItem[] = [];

    items.forEach(item => {
      if (item.item_type === 'food') {
        const localFood = foodDatabase.find(f => f.id === item.item_id);
        if (localFood) {
          foods.push({ ...item, details: localFood });
        } else {
          // Check scan history for this barcode/product
          const scanned = scanHistory.find(s => s.barcode === item.item_id || s.product_id === item.item_id);
          if (scanned) {
            scannedProducts.push({
              id: item.id,
              item_id: item.item_id,
              item_type: scanned.product_type as 'food' | 'cosmetic',
              quantity: item.quantity,
              is_purchased: item.is_purchased,
              product_name: scanned.product_name,
              image_url: scanned.image_url
            });
          }
        }
      } else {
        const localCosmetic = cosmeticsDatabase.find(c => c.id === item.item_id);
        if (localCosmetic) {
          cosmetics.push({ ...item, details: localCosmetic });
        } else {
          // Check scan history for this barcode/product
          const scanned = scanHistory.find(s => s.barcode === item.item_id || s.product_id === item.item_id);
          if (scanned) {
            scannedProducts.push({
              id: item.id,
              item_id: item.item_id,
              item_type: scanned.product_type as 'food' | 'cosmetic',
              quantity: item.quantity,
              is_purchased: item.is_purchased,
              product_name: scanned.product_name,
              image_url: scanned.image_url
            });
          }
        }
      }
    });

    return { foods, cosmetics, scannedProducts };
  }, [items, scanHistory]);

  if (authLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50">
        <Loader2 className="h-8 w-8 animate-spin text-emerald-600" />
      </div>
    );
  }

  if (!user) return null;

  const { foods: foodItems, cosmetics: cosmeticItems, scannedProducts: scannedItems } = resolvedItems;

  const purchasedCount = items.filter(i => i.is_purchased).length;
  const totalCount = items.length;

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50">
      <div className="container mx-auto px-4 py-8 max-w-4xl">
        <div className="flex items-center justify-between mb-8">
          <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
            <ArrowLeft className="h-4 w-4" />
            Back to Home
          </Link>
          <UserMenu />
        </div>

        <div className="flex items-center justify-between mb-8">
          <div>
            <h1 className="text-3xl font-bold text-foreground mb-2 flex items-center gap-3">
              <ShoppingCart className="h-8 w-8 text-primary" />
              Shopping List
            </h1>
            <p className="text-muted-foreground">
              {purchasedCount} of {totalCount} items purchased
            </p>
          </div>
          <div className="flex gap-2">
            <Link to="/tobin-tracker">
              <Button variant="outline" className="gap-2">
                <Activity className="h-4 w-4" />
                Exposure Tracker
              </Button>
            </Link>
            <Link to="/barcode-scanner">
              <Button variant="outline" className="gap-2">
                <ScanLine className="h-4 w-4" />
                Scan Barcode
              </Button>
            </Link>
            {purchasedCount > 0 && (
              <Button variant="outline" onClick={clearPurchased} className="gap-2">
                <Trash2 className="h-4 w-4" />
                Clear Purchased
              </Button>
            )}
          </div>
        </div>

        {loading ? (
          <div className="flex justify-center py-16">
            <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
          </div>
        ) : items.length === 0 ? (
          <Card>
            <CardContent className="flex flex-col items-center justify-center py-16">
              <ShoppingCart className="h-16 w-16 text-muted-foreground mb-4" />
              <h2 className="text-xl font-semibold mb-2">Your shopping list is empty</h2>
              <p className="text-muted-foreground text-center mb-6">
                Browse foods and cosmetics and click the cart icon to add them here!
              </p>
              <div className="flex gap-3">
                <Link to="/">
                  <Button>Browse Foods</Button>
                </Link>
                <Link to="/cosmetics">
                  <Button variant="outline">Browse Cosmetics</Button>
                </Link>
              </div>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-6">
            {/* Foods Section */}
            {foodItems.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    🥗 Foods
                    <Badge variant="secondary">{foodItems.length}</Badge>
                  </CardTitle>
                  <CardDescription>Food items to purchase</CardDescription>
                </CardHeader>
                <CardContent className="space-y-3">
                  {foodItems.map((item) => (
                    <div
                      key={item.id}
                      className={`flex items-center gap-4 p-4 rounded-lg border transition-colors ${
                        item.is_purchased ? 'bg-muted/50 opacity-60' : 'bg-background'
                      }`}
                    >
                      <Checkbox
                        checked={item.is_purchased}
                        onCheckedChange={(checked) => togglePurchased(item.id, !!checked)}
                      />
                      <span className="text-2xl">{item.details!.emoji}</span>
                      <div className="flex-1">
                        <p className={`font-medium ${item.is_purchased ? 'line-through' : ''}`}>
                          {item.details!.name}
                        </p>
                        <p className="text-sm text-muted-foreground">{item.details!.categoryName}</p>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() => updateQuantity(item.id, Math.max(1, item.quantity - 1))}
                          disabled={item.quantity <= 1}
                        >
                          <Minus className="h-3 w-3" />
                        </Button>
                        <span className="w-8 text-center font-medium">{item.quantity}</span>
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() => updateQuantity(item.id, item.quantity + 1)}
                        >
                          <Plus className="h-3 w-3" />
                        </Button>
                      </div>
                      <Button
                        variant="ghost"
                        size="icon"
                        className="h-8 w-8 text-muted-foreground hover:text-destructive"
                        onClick={() => removeItem(item.item_id, 'food')}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                </CardContent>
              </Card>
            )}

            {/* Cosmetics Section */}
            {cosmeticItems.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    ✨ Cosmetics
                    <Badge variant="secondary">{cosmeticItems.length}</Badge>
                  </CardTitle>
                  <CardDescription>Personal care products to purchase</CardDescription>
                </CardHeader>
                <CardContent className="space-y-3">
                  {cosmeticItems.map((item) => (
                    <div
                      key={item.id}
                      className={`flex items-center gap-4 p-4 rounded-lg border transition-colors ${
                        item.is_purchased ? 'bg-muted/50 opacity-60' : 'bg-background'
                      }`}
                    >
                      <Checkbox
                        checked={item.is_purchased}
                        onCheckedChange={(checked) => togglePurchased(item.id, !!checked)}
                      />
                      <span className="text-2xl">{item.details!.emoji}</span>
                      <div className="flex-1">
                        <p className={`font-medium ${item.is_purchased ? 'line-through' : ''}`}>
                          {item.details!.name}
                        </p>
                        <p className="text-sm text-muted-foreground capitalize">
                          {item.details!.category.replace('_', ' ')}
                        </p>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() => updateQuantity(item.id, Math.max(1, item.quantity - 1))}
                          disabled={item.quantity <= 1}
                        >
                          <Minus className="h-3 w-3" />
                        </Button>
                        <span className="w-8 text-center font-medium">{item.quantity}</span>
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() => updateQuantity(item.id, item.quantity + 1)}
                        >
                          <Plus className="h-3 w-3" />
                        </Button>
                      </div>
                      <Button
                        variant="ghost"
                        size="icon"
                        className="h-8 w-8 text-muted-foreground hover:text-destructive"
                        onClick={() => removeItem(item.item_id, 'cosmetic')}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                </CardContent>
              </Card>
            )}

            {/* Scanned Products Section */}
            {scannedItems.length > 0 && (
              <Card className="border-amber-200">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Sparkles className="h-5 w-5 text-amber-500" />
                    Scanned Products
                    <Badge variant="secondary">{scannedItems.length}</Badge>
                  </CardTitle>
                  <CardDescription>Products added from barcode scanning</CardDescription>
                </CardHeader>
                <CardContent className="space-y-3">
                  {scannedItems.map((item) => (
                    <div
                      key={item.id}
                      className={`flex items-center gap-4 p-4 rounded-lg border transition-colors ${
                        item.is_purchased ? 'bg-muted/50 opacity-60' : 'bg-background'
                      }`}
                    >
                      <Checkbox
                        checked={item.is_purchased}
                        onCheckedChange={(checked) => togglePurchased(item.id, !!checked)}
                      />
                      <div className="h-10 w-10 rounded-lg bg-muted flex items-center justify-center">
                        <Package className="h-5 w-5 text-muted-foreground" />
                      </div>
                      <div className="flex-1">
                        <p className={`font-medium ${item.is_purchased ? 'line-through' : ''}`}>
                          {item.product_name}
                        </p>
                        <div className="flex items-center gap-2 mt-0.5">
                          <Badge variant="outline" className="text-xs capitalize">
                            {item.item_type}
                          </Badge>
                          <Badge variant="secondary" className="text-xs gap-1">
                            <Sparkles className="h-2.5 w-2.5" />
                            Scanned
                          </Badge>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() => updateQuantity(item.id, Math.max(1, item.quantity - 1))}
                          disabled={item.quantity <= 1}
                        >
                          <Minus className="h-3 w-3" />
                        </Button>
                        <span className="w-8 text-center font-medium">{item.quantity}</span>
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() => updateQuantity(item.id, item.quantity + 1)}
                        >
                          <Plus className="h-3 w-3" />
                        </Button>
                      </div>
                      <Button
                        variant="ghost"
                        size="icon"
                        className="h-8 w-8 text-muted-foreground hover:text-destructive"
                        onClick={() => removeItem(item.item_id, item.item_type)}
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                    </div>
                  ))}
                </CardContent>
              </Card>
            )}
          </div>
        )}
      </div>
    </div>
  );
};

export default ShoppingList;
