import { Link } from 'react-router-dom';
import { useAuth } from '@/hooks/useAuth';
import { useSubscription, SUBSCRIPTION_TIERS } from '@/hooks/useSubscription';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardFooter, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Progress } from '@/components/ui/progress';
import { UserMenu } from '@/components/UserMenu';
import { 
  ArrowLeft, Check, Loader2, Crown, Calendar, Sparkles, 
  Shield, Zap, Star, Gift, CreditCard, Settings
} from 'lucide-react';
import { format, differenceInDays } from 'date-fns';

const Subscription = () => {
  const { user } = useAuth();
  const { 
    subscribed, 
    is_trial, 
    subscription_end, 
    trial_end,
    loading, 
    createCheckout, 
    openCustomerPortal,
    checkSubscription,
  } = useSubscription();

  const premium = SUBSCRIPTION_TIERS.premium;

  // Calculate trial progress
  const getTrialProgress = () => {
    if (!is_trial || !trial_end) return null;
    const endDate = new Date(trial_end);
    const daysLeft = differenceInDays(endDate, new Date());
    const progress = Math.max(0, Math.min(100, ((7 - daysLeft) / 7) * 100));
    return { daysLeft: Math.max(0, daysLeft), progress };
  };

  const trialInfo = getTrialProgress();

  return (
    <div className="h-screen overflow-hidden bg-gradient-to-br from-emerald-50 via-white to-amber-50 flex flex-col">
      {/* Decorative background elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <div className="absolute -top-40 -right-40 w-80 h-80 bg-primary/5 rounded-full blur-3xl" />
        <div className="absolute top-1/2 -left-40 w-80 h-80 bg-amber-200/20 rounded-full blur-3xl" />
      </div>

      <div className="relative container mx-auto px-4 py-4 max-w-5xl flex-1 flex flex-col">
        {/* Header */}
        <div className="flex items-center justify-between mb-4">
          <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors text-sm">
            <ArrowLeft className="h-4 w-4" />
            Back
          </Link>
          <UserMenu />
        </div>

        {/* Hero - Ultra Compact */}
        <div className="text-center mb-4 animate-fade-in">
          <div className="inline-flex items-center justify-center w-12 h-12 bg-gradient-to-br from-primary to-primary/70 rounded-full mb-2 shadow-lg shadow-primary/25">
            <Crown className="h-6 w-6 text-primary-foreground" />
          </div>
          <h1 className="text-2xl font-bold text-foreground tracking-tight">Premium Subscription</h1>
        </div>

        {/* Main Content - Side by Side */}
        <div className="flex-1 flex items-center justify-center gap-6">
          {/* Left: Current Status Card */}
          {user && subscribed && (
            <Card className="relative overflow-hidden border-0 shadow-xl bg-gradient-to-br from-primary via-primary to-primary/80 w-full max-w-sm animate-fade-in self-stretch flex flex-col">
              <div className="absolute inset-0 opacity-10">
                <div className="absolute top-0 right-0 w-32 h-32 bg-white rounded-full -translate-y-1/2 translate-x-1/2" />
              </div>
              
              <CardContent className="relative pt-5 pb-5 flex-1 flex flex-col justify-between">
                <div>
                  <div className="flex items-center gap-3 mb-4">
                    <div className="w-10 h-10 rounded-full bg-white/20 backdrop-blur flex items-center justify-center">
                      <Crown className="h-5 w-5 text-primary-foreground" />
                    </div>
                    <div>
                      <p className="text-primary-foreground/70 text-xs font-medium">Your Plan</p>
                      <h2 className="text-lg font-bold text-primary-foreground">Premium</h2>
                    </div>
                  </div>

                  <div className="flex flex-wrap gap-2 mb-4">
                    <Badge 
                      className={`${is_trial 
                        ? 'bg-amber-400/90 text-amber-950 hover:bg-amber-400' 
                        : 'bg-white/20 text-primary-foreground hover:bg-white/30'
                      } border-0 px-2 py-0.5 text-xs`}
                    >
                      {is_trial ? (
                        <><Gift className="h-3 w-3 mr-1" />Free Trial</>
                      ) : (
                        <><Shield className="h-3 w-3 mr-1" />Active</>
                      )}
                    </Badge>
                    <Badge className="bg-white/10 text-primary-foreground hover:bg-white/20 border-0 px-2 py-0.5 text-xs">
                      <Zap className="h-3 w-3 mr-1" />All Features
                    </Badge>
                  </div>

                  {is_trial && trialInfo ? (
                    <div className="bg-white/10 backdrop-blur rounded-lg p-3 space-y-2 mb-4">
                      <div className="flex items-center justify-between">
                        <span className="text-primary-foreground/70 text-xs">Trial Progress</span>
                        <span className="text-primary-foreground font-bold text-sm">{trialInfo.daysLeft} days left</span>
                      </div>
                      <Progress value={trialInfo.progress} className="h-1.5 bg-white/20" />
                      {trial_end && (
                        <div className="flex items-center gap-1 text-primary-foreground/70 text-xs">
                          <Calendar className="h-3 w-3" />
                          Ends {format(new Date(trial_end), 'MMM d, yyyy')}
                        </div>
                      )}
                    </div>
                  ) : subscription_end && (
                    <div className="bg-white/10 backdrop-blur rounded-lg p-3 space-y-1 mb-4">
                      <div className="flex items-center gap-1 text-primary-foreground/70 text-xs">
                        <CreditCard className="h-3 w-3" />Next billing
                      </div>
                      <p className="text-primary-foreground font-bold text-sm">
                        {format(new Date(subscription_end), 'MMM d, yyyy')}
                      </p>
                      <p className="text-primary-foreground/60 text-xs">${premium.price}/month</p>
                    </div>
                  )}
                </div>

                <Button 
                  variant="secondary" 
                  size="sm"
                  onClick={openCustomerPortal} 
                  disabled={loading}
                  className="w-full bg-white/20 hover:bg-white/30 text-primary-foreground border-0"
                >
                  {loading ? <Loader2 className="h-4 w-4 mr-2 animate-spin" /> : <Settings className="h-4 w-4 mr-2" />}
                  Manage Subscription
                </Button>
              </CardContent>
            </Card>
          )}

          {/* Right: Pricing Card */}
          <Card className={`w-full max-w-sm animate-fade-in self-stretch flex flex-col ${subscribed ? 'opacity-60' : 'border-2 border-primary/20 shadow-2xl shadow-primary/10'}`}>
            <CardHeader className="text-center pb-2 pt-4">
              <Badge className="mx-auto mb-2 bg-gradient-to-r from-amber-400 to-orange-400 text-amber-950 border-0 px-2.5 py-0.5 text-xs font-semibold shadow">
                <Star className="h-3 w-3 mr-1 fill-current" />7-Day Free Trial
              </Badge>
              <CardTitle className="text-xl font-bold">{premium.name}</CardTitle>
              <CardDescription className="text-xs">Everything for your health journey</CardDescription>
            </CardHeader>
            
            <CardContent className="text-center pb-3 flex-1">
              <div className="mb-3">
                <div className="flex items-baseline justify-center gap-1">
                  <span className="text-3xl font-bold tracking-tight">${premium.price}</span>
                  <span className="text-muted-foreground text-sm">/{premium.interval}</span>
                </div>
                <p className="text-xs text-muted-foreground">After 7-day free trial</p>
              </div>
              
              <div className="bg-muted/50 rounded-lg p-3">
                <ul className="space-y-2 text-left">
                  {premium.features.slice(0, 5).map((feature, index) => (
                    <li key={index} className="flex items-start gap-2">
                      <div className="w-4 h-4 rounded-full bg-primary/10 flex items-center justify-center flex-shrink-0">
                        <Check className="h-2.5 w-2.5 text-primary" />
                      </div>
                      <span className="text-xs">{feature}</span>
                    </li>
                  ))}
                </ul>
              </div>
            </CardContent>
            
            <CardFooter className="flex flex-col gap-2 pb-4 pt-0">
              {!user ? (
                <>
                  <Button asChild size="sm" className="w-full font-semibold shadow-lg shadow-primary/25">
                    <Link to="/auth">
                      <Sparkles className="h-4 w-4 mr-2" />Sign Up for Free Trial
                    </Link>
                  </Button>
                  <p className="text-[10px] text-muted-foreground text-center">Create an account to start</p>
                </>
              ) : subscribed ? (
                <Button variant="outline" size="sm" className="w-full" disabled>
                  <Check className="h-4 w-4 mr-2" />Currently Subscribed
                </Button>
              ) : (
                <>
                  <Button size="sm" className="w-full font-semibold shadow-lg shadow-primary/25" onClick={createCheckout} disabled={loading}>
                    {loading ? <Loader2 className="h-4 w-4 mr-2 animate-spin" /> : <Sparkles className="h-4 w-4 mr-2" />}
                    Start Free Trial
                  </Button>
                  <p className="text-[10px] text-muted-foreground text-center">Cancel anytime • No charge until trial ends</p>
                </>
              )}
            </CardFooter>
          </Card>
        </div>

        {/* Footer */}
        <div className="flex justify-center gap-6 py-3 text-muted-foreground">
          <div className="flex items-center gap-1.5 text-xs">
            <Shield className="h-3.5 w-3.5" /><span>Secure</span>
          </div>
          <div className="flex items-center gap-1.5 text-xs">
            <Zap className="h-3.5 w-3.5" /><span>Instant Access</span>
          </div>
          <div className="flex items-center gap-1.5 text-xs">
            <Gift className="h-3.5 w-3.5" /><span>Cancel Anytime</span>
          </div>
          {user && (
            <Button variant="ghost" size="sm" onClick={checkSubscription} disabled={loading} className="text-muted-foreground/60 hover:text-muted-foreground h-auto py-0 px-2 text-xs">
              {loading && <Loader2 className="h-3 w-3 mr-1 animate-spin" />}Refresh
            </Button>
          )}
        </div>
      </div>
    </div>
  );
};

export default Subscription;
