import { useState, useMemo } from 'react';
import { Link } from 'react-router-dom';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Input } from '@/components/ui/input';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Progress } from '@/components/ui/progress';
import { ScrollArea } from '@/components/ui/scroll-area';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { 
  ArrowLeft, 
  Plus, 
  X, 
  Search, 
  Scale, 
  Utensils, 
  Sparkles,
  Skull,
  Flame,
  HeartPulse,
  Zap,
  Pill,
  AlertTriangle,
  Check,
  Trophy
} from 'lucide-react';
import { foodDatabase, FoodItem, TobinScore as FoodTobinScore, getTobinRating } from '@/data/foodDatabase';
import { cosmeticsDatabase, CosmeticItem, TobinScore as CosmeticTobinScore } from '@/data/cosmeticsDatabase';
import { useHealthProfile } from '@/hooks/useHealthProfile';
import { useAuth } from '@/hooks/useAuth';

type ComparisonItem = {
  id: string;
  name: string;
  emoji: string;
  type: 'food' | 'cosmetic';
  tobinScore: FoodTobinScore | CosmeticTobinScore;
  category: string;
};

const categories = [
  { key: 'T', label: 'Toxins', icon: Skull, color: 'text-red-500', bgColor: 'bg-red-500' },
  { key: 'O', label: 'Oxidative', icon: Flame, color: 'text-orange-500', bgColor: 'bg-orange-500' },
  { key: 'B', label: 'Biological', icon: HeartPulse, color: 'text-purple-500', bgColor: 'bg-purple-500' },
  { key: 'I', label: 'Inflammatory', icon: Zap, color: 'text-yellow-600', bgColor: 'bg-yellow-500' },
  { key: 'N', label: 'Nutrient', icon: Pill, color: 'text-blue-500', bgColor: 'bg-blue-500' },
] as const;

const TobinComparison = () => {
  const { user } = useAuth();
  const { profile } = useHealthProfile();
  const [selectedItems, setSelectedItems] = useState<ComparisonItem[]>([]);
  const [searchQuery, setSearchQuery] = useState('');
  const [activeTab, setActiveTab] = useState<'food' | 'cosmetic'>('food');

  // Get items with TOBIN scores
  const availableFoods = useMemo(() => 
    foodDatabase
      .filter(f => f.tobinScore)
      .map(f => ({
        id: f.id,
        name: f.name,
        emoji: f.emoji,
        type: 'food' as const,
        tobinScore: f.tobinScore!,
        category: f.categoryName,
      })),
    []
  );

  const availableCosmetics = useMemo(() => 
    cosmeticsDatabase
      .filter(c => c.tobinScore)
      .map(c => ({
        id: c.id,
        name: c.name,
        emoji: c.emoji,
        type: 'cosmetic' as const,
        tobinScore: c.tobinScore!,
        category: c.category,
      })),
    []
  );

  const currentItems = activeTab === 'food' ? availableFoods : availableCosmetics;
  
  const filteredItems = useMemo(() => 
    currentItems.filter(item => 
      item.name.toLowerCase().includes(searchQuery.toLowerCase()) &&
      !selectedItems.some(s => s.id === item.id)
    ),
    [currentItems, searchQuery, selectedItems]
  );

  const addItem = (item: ComparisonItem) => {
    if (selectedItems.length < 5) {
      setSelectedItems([...selectedItems, item]);
      setSearchQuery('');
    }
  };

  const removeItem = (id: string) => {
    setSelectedItems(selectedItems.filter(item => item.id !== id));
  };

  const getThresholdForCategory = (key: string): number => {
    switch (key) {
      case 'T': return profile.toxin_threshold;
      case 'O': return profile.oxidative_threshold;
      case 'B': return profile.biological_threshold;
      case 'I': return profile.inflammatory_threshold;
      case 'N': return profile.nutrient_threshold;
      default: return 5;
    }
  };

  const exceedsThreshold = (score: number, key: string): boolean => {
    return score > getThresholdForCategory(key);
  };

  const getBestItem = (): ComparisonItem | null => {
    if (selectedItems.length === 0) return null;
    return selectedItems.reduce((best, current) => 
      current.tobinScore.total < best.tobinScore.total ? current : best
    );
  };

  const bestItem = getBestItem();

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      <div className="container mx-auto px-4 py-8 max-w-7xl">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <div className="flex items-center gap-4">
            <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
              <ArrowLeft className="h-4 w-4" />
              Back
            </Link>
            <div>
              <h1 className="text-3xl font-bold flex items-center gap-3">
                <Scale className="h-8 w-8 text-primary" />
                TOBIN Comparison Tool
              </h1>
              <p className="text-muted-foreground">Compare items side-by-side against your health thresholds</p>
            </div>
          </div>
        </div>

        {/* Item Selector */}
        <Card className="mb-6">
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Plus className="h-5 w-5" />
              Add Items to Compare
            </CardTitle>
            <CardDescription>
              Select up to 5 items to compare their TOBIN scores
            </CardDescription>
          </CardHeader>
          <CardContent>
            <Tabs value={activeTab} onValueChange={(v) => setActiveTab(v as 'food' | 'cosmetic')}>
              <TabsList className="mb-4">
                <TabsTrigger value="food" className="gap-2">
                  <Utensils className="h-4 w-4" />
                  Foods
                </TabsTrigger>
                <TabsTrigger value="cosmetic" className="gap-2">
                  <Sparkles className="h-4 w-4" />
                  Cosmetics
                </TabsTrigger>
              </TabsList>
              
              <div className="relative mb-4">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder={`Search ${activeTab === 'food' ? 'foods' : 'cosmetics'}...`}
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-10"
                />
              </div>

              <ScrollArea className="h-48">
                <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-2">
                  {filteredItems.slice(0, 20).map((item) => (
                    <button
                      key={item.id}
                      onClick={() => addItem(item)}
                      disabled={selectedItems.length >= 5}
                      className="flex items-center gap-2 p-2 rounded-lg border bg-card hover:bg-muted transition-colors text-left disabled:opacity-50 disabled:cursor-not-allowed"
                    >
                      <span className="text-xl">{item.emoji}</span>
                      <div className="flex-1 min-w-0">
                        <p className="text-sm font-medium truncate">{item.name}</p>
                        <p className="text-xs text-muted-foreground">Score: {item.tobinScore.total}</p>
                      </div>
                      <Plus className="h-4 w-4 text-muted-foreground flex-shrink-0" />
                    </button>
                  ))}
                </div>
              </ScrollArea>
            </Tabs>
          </CardContent>
        </Card>

        {/* Selected Items */}
        {selectedItems.length > 0 && (
          <div className="mb-6">
            <div className="flex flex-wrap gap-2">
              {selectedItems.map((item) => (
                <Badge
                  key={item.id}
                  variant="secondary"
                  className="px-3 py-2 text-sm flex items-center gap-2"
                >
                  <span>{item.emoji}</span>
                  <span>{item.name}</span>
                  {bestItem?.id === item.id && (
                    <Trophy className="h-3 w-3 text-yellow-500" />
                  )}
                  <button
                    onClick={() => removeItem(item.id)}
                    className="ml-1 hover:text-destructive"
                  >
                    <X className="h-3 w-3" />
                  </button>
                </Badge>
              ))}
            </div>
          </div>
        )}

        {/* Personal Thresholds Info */}
        {user && selectedItems.length > 0 && (
          <Alert className="mb-6">
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              Scores exceeding your personal thresholds are highlighted in red. Your total threshold is {profile.total_threshold}/50.
            </AlertDescription>
          </Alert>
        )}

        {/* Comparison Grid */}
        {selectedItems.length > 0 ? (
          <div className="space-y-6">
            {/* Total Score Comparison */}
            <Card>
              <CardHeader>
                <CardTitle>Overall TOBIN Score</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid gap-4" style={{ gridTemplateColumns: `repeat(${selectedItems.length}, 1fr)` }}>
                  {selectedItems.map((item) => {
                    const rating = getTobinRating(item.tobinScore.total);
                    const exceedsTotal = item.tobinScore.total > profile.total_threshold;
                    
                    return (
                      <div 
                        key={item.id} 
                        className={`text-center p-4 rounded-xl border-2 ${
                          bestItem?.id === item.id 
                            ? 'border-primary bg-primary/5 ring-2 ring-primary/20' 
                            : exceedsTotal
                              ? 'border-red-300 bg-red-50 dark:bg-red-950/20'
                              : 'border-border'
                        }`}
                      >
                        {bestItem?.id === item.id && (
                          <Badge className="mb-2 bg-primary">
                            <Trophy className="h-3 w-3 mr-1" />
                            Best Choice
                          </Badge>
                        )}
                        <div className="text-3xl mb-2">{item.emoji}</div>
                        <p className="font-semibold text-sm mb-2 truncate">{item.name}</p>
                        <div className={`text-4xl font-bold ${rating.color}`}>
                          {item.tobinScore.total}
                        </div>
                        <p className="text-xs text-muted-foreground">/50</p>
                        <Badge className={`mt-2 ${rating.bgColor} ${rating.color} border-0`}>
                          {rating.label}
                        </Badge>
                        {exceedsTotal && (
                          <p className="text-xs text-red-600 mt-2 flex items-center justify-center gap-1">
                            <AlertTriangle className="h-3 w-3" />
                            Exceeds threshold
                          </p>
                        )}
                      </div>
                    );
                  })}
                </div>
              </CardContent>
            </Card>

            {/* Category Breakdown */}
            <Card>
              <CardHeader>
                <CardTitle>Category Breakdown</CardTitle>
                <CardDescription>Compare each TOBIN category across items</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                {categories.map(({ key, label, icon: Icon, color, bgColor }) => {
                  const threshold = getThresholdForCategory(key);
                  const scores = selectedItems.map(item => ({
                    item,
                    score: item.tobinScore[key as keyof typeof item.tobinScore] as number
                  }));
                  const bestScore = Math.min(...scores.map(s => s.score));

                  return (
                    <div key={key} className="space-y-3">
                      <div className="flex items-center gap-2">
                        <Icon className={`h-5 w-5 ${color}`} />
                        <span className="font-semibold">{label}</span>
                        {user && (
                          <Badge variant="outline" className="ml-auto text-xs">
                            Your threshold: {threshold}
                          </Badge>
                        )}
                      </div>
                      
                      <div className="grid gap-2" style={{ gridTemplateColumns: `repeat(${selectedItems.length}, 1fr)` }}>
                        {scores.map(({ item, score }) => {
                          const isBest = score === bestScore;
                          const exceeds = exceedsThreshold(score, key);
                          
                          return (
                            <div 
                              key={item.id}
                              className={`p-3 rounded-lg border ${
                                isBest 
                                  ? 'border-primary bg-primary/5' 
                                  : exceeds 
                                    ? 'border-red-300 bg-red-50 dark:bg-red-950/20' 
                                    : 'border-border bg-muted/30'
                              }`}
                            >
                              <div className="flex items-center justify-between mb-2">
                                <span className="text-sm truncate">{item.name}</span>
                                <div className="flex items-center gap-1">
                                  {isBest && <Check className="h-3 w-3 text-primary" />}
                                  <span className={`font-bold ${exceeds ? 'text-red-600' : ''}`}>
                                    {score}
                                  </span>
                                </div>
                              </div>
                              <div className="h-2 bg-muted rounded-full overflow-hidden">
                                <div 
                                  className={`h-full rounded-full transition-all duration-500 ${
                                    exceeds ? 'bg-red-500' : bgColor
                                  }`}
                                  style={{ width: `${(score / 10) * 100}%` }}
                                />
                              </div>
                            </div>
                          );
                        })}
                      </div>
                    </div>
                  );
                })}
              </CardContent>
            </Card>

            {/* Summary */}
            {bestItem && (
              <Card className="border-primary">
                <CardContent className="p-6">
                  <div className="flex items-center gap-4">
                    <div className="h-16 w-16 rounded-full bg-primary/10 flex items-center justify-center">
                      <Trophy className="h-8 w-8 text-primary" />
                    </div>
                    <div>
                      <h3 className="text-xl font-bold">Recommendation</h3>
                      <p className="text-muted-foreground">
                        Based on overall TOBIN scores, <strong>{bestItem.emoji} {bestItem.name}</strong> is the healthiest choice
                        with a total score of <strong>{bestItem.tobinScore.total}/50</strong>.
                      </p>
                    </div>
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        ) : (
          <Card>
            <CardContent className="p-12 text-center">
              <Scale className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
              <h3 className="text-xl font-semibold mb-2">No Items Selected</h3>
              <p className="text-muted-foreground">
                Search and add items above to start comparing their TOBIN health scores.
              </p>
            </CardContent>
          </Card>
        )}
      </div>
    </div>
  );
};

export default TobinComparison;
