import { useMemo } from 'react';
import { Link } from 'react-router-dom';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Progress } from '@/components/ui/progress';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  ArrowLeft, 
  Activity, 
  TrendingUp, 
  TrendingDown,
  Skull,
  Flame,
  HeartPulse,
  Zap,
  Pill,
  Calendar,
  ShoppingCart,
  AlertTriangle,
  CheckCircle,
  BarChart3,
  Loader2
} from 'lucide-react';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer, PieChart, Pie, Cell, Legend, RadarChart, PolarGrid, PolarAngleAxis, PolarRadiusAxis, Radar } from 'recharts';
import { useShoppingList } from '@/contexts/ShoppingListContext';
import { useHealthProfile } from '@/hooks/useHealthProfile';
import { useAuth } from '@/hooks/useAuth';
import { foodDatabase, TobinScore as FoodTobinScore } from '@/data/foodDatabase';
import { cosmeticsDatabase, TobinScore as CosmeticTobinScore } from '@/data/cosmeticsDatabase';
import { format, subDays, isAfter, startOfDay, parseISO } from 'date-fns';

type TobinScore = FoodTobinScore | CosmeticTobinScore;

interface ExposureData {
  date: string;
  total: number;
  T: number;
  O: number;
  B: number;
  I: number;
  N: number;
  itemCount: number;
}

const categories = [
  { key: 'T', label: 'Toxins', fullLabel: 'Toxins & Carcinogens', icon: Skull, color: '#ef4444' },
  { key: 'O', label: 'Oxidative', fullLabel: 'Oxidative Stress', icon: Flame, color: '#f97316' },
  { key: 'B', label: 'Biological', fullLabel: 'Biological/Hormonal', icon: HeartPulse, color: '#a855f7' },
  { key: 'I', label: 'Inflammatory', fullLabel: 'Inflammatory', icon: Zap, color: '#eab308' },
  { key: 'N', label: 'Nutrient', fullLabel: 'Nutrient Depletion', icon: Pill, color: '#3b82f6' },
];

const COLORS = ['#ef4444', '#f97316', '#a855f7', '#eab308', '#3b82f6'];

const TobinExposureTracker = () => {
  const { user, loading: authLoading } = useAuth();
  const { items, loading: listLoading } = useShoppingList();
  const { profile } = useHealthProfile();

  // Get TOBIN scores for shopping list items
  const itemsWithScores = useMemo(() => {
    return items.map(item => {
      if (item.item_type === 'food') {
        const food = foodDatabase.find(f => f.id === item.item_id);
        return { ...item, tobinScore: food?.tobinScore, details: food };
      } else {
        const cosmetic = cosmeticsDatabase.find(c => c.id === item.item_id);
        return { ...item, tobinScore: cosmetic?.tobinScore, details: cosmetic };
      }
    }).filter(item => item.tobinScore);
  }, [items]);

  // Calculate daily exposure data (last 7 days)
  const dailyExposure = useMemo(() => {
    const days: ExposureData[] = [];
    
    for (let i = 6; i >= 0; i--) {
      const date = subDays(new Date(), i);
      const dayStart = startOfDay(date);
      const dayEnd = startOfDay(subDays(date, -1));
      
      const dayItems = itemsWithScores.filter(item => {
        const itemDate = parseISO(item.created_at);
        return isAfter(itemDate, dayStart) && !isAfter(itemDate, dayEnd);
      });
      
      const totals = dayItems.reduce((acc, item) => {
        const score = item.tobinScore!;
        const qty = item.quantity || 1;
        return {
          total: acc.total + (score.total * qty),
          T: acc.T + (score.T * qty),
          O: acc.O + (score.O * qty),
          B: acc.B + (score.B * qty),
          I: acc.I + (score.I * qty),
          N: acc.N + (score.N * qty),
          itemCount: acc.itemCount + qty,
        };
      }, { total: 0, T: 0, O: 0, B: 0, I: 0, N: 0, itemCount: 0 });
      
      days.push({
        date: format(date, 'EEE'),
        ...totals,
      });
    }
    
    return days;
  }, [itemsWithScores]);

  // Calculate total exposure
  const totalExposure = useMemo(() => {
    return itemsWithScores.reduce((acc, item) => {
      const score = item.tobinScore!;
      const qty = item.quantity || 1;
      return {
        total: acc.total + (score.total * qty),
        T: acc.T + (score.T * qty),
        O: acc.O + (score.O * qty),
        B: acc.B + (score.B * qty),
        I: acc.I + (score.I * qty),
        N: acc.N + (score.N * qty),
        itemCount: acc.itemCount + qty,
      };
    }, { total: 0, T: 0, O: 0, B: 0, I: 0, N: 0, itemCount: 0 });
  }, [itemsWithScores]);

  // Category breakdown for pie chart
  const categoryBreakdown = useMemo(() => {
    return categories.map(cat => ({
      name: cat.label,
      value: totalExposure[cat.key as keyof typeof totalExposure] as number,
      color: cat.color,
    }));
  }, [totalExposure]);

  // Radar data for category comparison
  const radarData = useMemo(() => {
    return categories.map(cat => ({
      category: cat.label,
      exposure: totalExposure[cat.key as keyof typeof totalExposure] as number,
      threshold: profile[`${cat.key.toLowerCase() === 't' ? 'toxin' : cat.key.toLowerCase() === 'o' ? 'oxidative' : cat.key.toLowerCase() === 'b' ? 'biological' : cat.key.toLowerCase() === 'i' ? 'inflammatory' : 'nutrient'}_threshold` as keyof typeof profile] as number * totalExposure.itemCount,
    }));
  }, [totalExposure, profile]);

  // Items exceeding thresholds
  const itemsExceedingThresholds = useMemo(() => {
    return itemsWithScores.filter(item => {
      const score = item.tobinScore!;
      return score.total > profile.total_threshold ||
        score.T > profile.toxin_threshold ||
        score.O > profile.oxidative_threshold ||
        score.B > profile.biological_threshold ||
        score.I > profile.inflammatory_threshold ||
        score.N > profile.nutrient_threshold;
    });
  }, [itemsWithScores, profile]);

  // Highest risk items
  const highestRiskItems = useMemo(() => {
    return [...itemsWithScores]
      .sort((a, b) => (b.tobinScore?.total || 0) - (a.tobinScore?.total || 0))
      .slice(0, 5);
  }, [itemsWithScores]);

  if (authLoading || listLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  if (!user) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        <Card className="max-w-md">
          <CardContent className="p-6 text-center">
            <Activity className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
            <h2 className="text-xl font-semibold mb-2">Sign In Required</h2>
            <p className="text-muted-foreground mb-4">Sign in to track your TOBIN exposure from your shopping list.</p>
            <Link to="/auth">
              <Button>Sign In</Button>
            </Link>
          </CardContent>
        </Card>
      </div>
    );
  }

  const weeklyTotal = dailyExposure.reduce((sum, day) => sum + day.total, 0);
  const avgDaily = weeklyTotal / 7;

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      <div className="container mx-auto px-4 py-8 max-w-7xl">
        {/* Header */}
        <div className="flex items-center justify-between mb-8">
          <div className="flex items-center gap-4">
            <Link to="/" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors">
              <ArrowLeft className="h-4 w-4" />
              Back
            </Link>
            <div>
              <h1 className="text-3xl font-bold flex items-center gap-3">
                <Activity className="h-8 w-8 text-primary" />
                TOBIN Exposure Tracker
              </h1>
              <p className="text-muted-foreground">Monitor your health exposure from shopping list items</p>
            </div>
          </div>
          <Link to="/shopping-list">
            <Button variant="outline" className="gap-2">
              <ShoppingCart className="h-4 w-4" />
              View Shopping List
            </Button>
          </Link>
        </div>

        {itemsWithScores.length === 0 ? (
          <Card>
            <CardContent className="p-12 text-center">
              <ShoppingCart className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
              <h3 className="text-xl font-semibold mb-2">No Items to Track</h3>
              <p className="text-muted-foreground mb-4">
                Add items with TOBIN scores to your shopping list to start tracking your exposure.
              </p>
              <Link to="/">
                <Button>Browse Foods</Button>
              </Link>
            </CardContent>
          </Card>
        ) : (
          <div className="space-y-6">
            {/* Summary Cards */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-muted-foreground">Total Exposure</p>
                      <p className="text-3xl font-bold">{totalExposure.total}</p>
                    </div>
                    <BarChart3 className="h-8 w-8 text-primary" />
                  </div>
                </CardContent>
              </Card>
              
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-muted-foreground">Items Tracked</p>
                      <p className="text-3xl font-bold">{totalExposure.itemCount}</p>
                    </div>
                    <ShoppingCart className="h-8 w-8 text-blue-500" />
                  </div>
                </CardContent>
              </Card>
              
              <Card>
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-muted-foreground">Avg per Item</p>
                      <p className="text-3xl font-bold">
                        {totalExposure.itemCount > 0 
                          ? (totalExposure.total / totalExposure.itemCount).toFixed(1) 
                          : 0}
                      </p>
                    </div>
                    <TrendingUp className="h-8 w-8 text-orange-500" />
                  </div>
                </CardContent>
              </Card>
              
              <Card className={itemsExceedingThresholds.length > 0 ? 'border-red-300 bg-red-50 dark:bg-red-950/20' : 'border-green-300 bg-green-50 dark:bg-green-950/20'}>
                <CardContent className="p-4">
                  <div className="flex items-center justify-between">
                    <div>
                      <p className="text-sm text-muted-foreground">Threshold Alerts</p>
                      <p className="text-3xl font-bold">{itemsExceedingThresholds.length}</p>
                    </div>
                    {itemsExceedingThresholds.length > 0 ? (
                      <AlertTriangle className="h-8 w-8 text-red-500" />
                    ) : (
                      <CheckCircle className="h-8 w-8 text-green-500" />
                    )}
                  </div>
                </CardContent>
              </Card>
            </div>

            {/* Charts */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Weekly Trend */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Calendar className="h-5 w-5" />
                    Weekly Exposure Trend
                  </CardTitle>
                  <CardDescription>Daily TOBIN exposure over the past 7 days</CardDescription>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={250}>
                    <BarChart data={dailyExposure}>
                      <CartesianGrid strokeDasharray="3 3" className="stroke-muted" />
                      <XAxis dataKey="date" className="text-xs" />
                      <YAxis className="text-xs" />
                      <Tooltip 
                        contentStyle={{ 
                          backgroundColor: 'hsl(var(--card))',
                          border: '1px solid hsl(var(--border))',
                          borderRadius: '8px'
                        }}
                      />
                      <Bar dataKey="total" fill="hsl(var(--primary))" radius={[4, 4, 0, 0]} />
                    </BarChart>
                  </ResponsiveContainer>
                </CardContent>
              </Card>

              {/* Category Breakdown */}
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <Activity className="h-5 w-5" />
                    Category Distribution
                  </CardTitle>
                  <CardDescription>Breakdown by TOBIN category</CardDescription>
                </CardHeader>
                <CardContent>
                  <ResponsiveContainer width="100%" height={250}>
                    <PieChart>
                      <Pie
                        data={categoryBreakdown}
                        cx="50%"
                        cy="50%"
                        innerRadius={60}
                        outerRadius={80}
                        paddingAngle={5}
                        dataKey="value"
                      >
                        {categoryBreakdown.map((entry, index) => (
                          <Cell key={`cell-${index}`} fill={entry.color} />
                        ))}
                      </Pie>
                      <Tooltip />
                      <Legend />
                    </PieChart>
                  </ResponsiveContainer>
                </CardContent>
              </Card>
            </div>

            {/* Category Details */}
            <Card>
              <CardHeader>
                <CardTitle>Category Breakdown</CardTitle>
                <CardDescription>Detailed exposure by TOBIN category</CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                {categories.map(({ key, label, fullLabel, icon: Icon, color }) => {
                  const value = totalExposure[key as keyof typeof totalExposure] as number;
                  const maxPossible = totalExposure.itemCount * 10;
                  const percentage = maxPossible > 0 ? (value / maxPossible) * 100 : 0;
                  
                  return (
                    <div key={key} className="space-y-2">
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-2">
                          <Icon className="h-5 w-5" style={{ color }} />
                          <span className="font-medium">{fullLabel}</span>
                        </div>
                        <span className="font-bold">{value}</span>
                      </div>
                      <div className="h-2 bg-muted rounded-full overflow-hidden">
                        <div 
                          className="h-full rounded-full transition-all duration-500"
                          style={{ width: `${Math.min(percentage, 100)}%`, backgroundColor: color }}
                        />
                      </div>
                    </div>
                  );
                })}
              </CardContent>
            </Card>

            {/* Highest Risk Items */}
            {highestRiskItems.length > 0 && (
              <Card>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    <AlertTriangle className="h-5 w-5 text-orange-500" />
                    Highest Risk Items
                  </CardTitle>
                  <CardDescription>Items with the highest TOBIN scores in your list</CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="space-y-3">
                    {highestRiskItems.map((item, index) => (
                      <div 
                        key={item.id} 
                        className={`flex items-center gap-4 p-3 rounded-lg border ${
                          item.tobinScore!.total > profile.total_threshold 
                            ? 'border-red-300 bg-red-50 dark:bg-red-950/20' 
                            : 'bg-muted/50'
                        }`}
                      >
                        <span className="text-lg font-bold text-muted-foreground w-6">
                          #{index + 1}
                        </span>
                        <span className="text-2xl">
                          {(item.details as any)?.emoji || '📦'}
                        </span>
                        <div className="flex-1">
                          <p className="font-medium">{(item.details as any)?.name || 'Unknown Item'}</p>
                          <p className="text-sm text-muted-foreground">
                            Quantity: {item.quantity}
                          </p>
                        </div>
                        <div className="text-right">
                          <p className={`text-2xl font-bold ${
                            item.tobinScore!.total > 30 ? 'text-red-600' :
                            item.tobinScore!.total > 20 ? 'text-orange-600' : 'text-yellow-600'
                          }`}>
                            {item.tobinScore!.total}
                          </p>
                          <p className="text-xs text-muted-foreground">/50</p>
                        </div>
                        {item.tobinScore!.total > profile.total_threshold && (
                          <Badge variant="destructive" className="ml-2">
                            Exceeds threshold
                          </Badge>
                        )}
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {/* Threshold Alerts */}
            {itemsExceedingThresholds.length > 0 && (
              <Card className="border-red-300">
                <CardHeader>
                  <CardTitle className="flex items-center gap-2 text-red-600">
                    <AlertTriangle className="h-5 w-5" />
                    Items Exceeding Your Thresholds
                  </CardTitle>
                  <CardDescription>
                    These items exceed one or more of your personal health thresholds
                  </CardDescription>
                </CardHeader>
                <CardContent>
                  <div className="grid gap-2">
                    {itemsExceedingThresholds.map((item) => (
                      <div key={item.id} className="flex items-center gap-3 p-2 rounded-lg bg-red-50 dark:bg-red-950/20">
                        <span className="text-xl">
                          {(item.details as any)?.emoji || '📦'}
                        </span>
                        <span className="flex-1 font-medium">
                          {(item.details as any)?.name || 'Unknown Item'}
                        </span>
                        <Badge variant="destructive">
                          Score: {item.tobinScore!.total}
                        </Badge>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}
          </div>
        )}
      </div>
    </div>
  );
};

export default TobinExposureTracker;
