import { useState, useEffect } from 'react';
import { useParams, Link } from 'react-router-dom';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';
import { 
  ArrowLeft, 
  MessageSquare, 
  Heart,
  Trophy,
  Lightbulb,
  Sparkles,
  Loader2,
  Calendar,
  TrendingUp,
  Users
} from 'lucide-react';
import { formatDistanceToNow, format } from 'date-fns';
import { Footer } from '@/components/Footer';
import { FollowButton } from '@/components/FollowButton';
import { MessageButton } from '@/components/MessagesDrawer';
import { useFollows } from '@/hooks/useFollows';
import { userPublicApi, communityApi, achievementsApi, API_BASE } from '@/lib/apiServices';

interface UserStats {
  totalPosts: number;
  totalReplies: number;
  totalLikesReceived: number;
  joinedDate: string | null;
}

interface UserProfileData {
  display_name: string | null;
  bio: string | null;
  avatar_url: string | null;
}

interface UserPost {
  id: string;
  content: string;
  category: string;
  likes_count: number;
  replies_count: number;
  created_at: string;
}

interface UserReply {
  id: string;
  content: string;
  created_at: string;
  post_id: string;
}

interface Achievement {
  id: string;
  achievement_id: string;
  earned_at: string;
  progress: number;
}

const achievementDefinitions: Record<string, { name: string; description: string; icon: string }> = {
  first_scan: { name: 'First Scan', description: 'Scanned your first product', icon: '🔍' },
  health_conscious: { name: 'Health Conscious', description: 'Kept TOBIN score below threshold', icon: '💚' },
  community_contributor: { name: 'Community Contributor', description: 'Made your first post', icon: '✍️' },
  helpful_member: { name: 'Helpful Member', description: 'Received 10 likes on posts', icon: '⭐' },
  tip_master: { name: 'Tip Master', description: 'Shared 5 health tips', icon: '💡' },
  engaged_reader: { name: 'Engaged Reader', description: 'Replied to 10 posts', icon: '💬' },
  popular_poster: { name: 'Popular Poster', description: 'Received 50 likes total', icon: '🏆' },
  week_streak: { name: 'Week Streak', description: '7 days of healthy choices', icon: '🔥' },
};

const categoryIcons: Record<string, React.ElementType> = {
  tip: Lightbulb,
  question: MessageSquare,
  achievement: Trophy,
  discovery: Sparkles,
};

const UserProfile = () => {
  const { userId } = useParams<{ userId: string }>();
  const { stats: followStats } = useFollows(userId);
  const [loading, setLoading] = useState(true);
  const [userName, setUserName] = useState<string>('');
  const [profileData, setProfileData] = useState<UserProfileData>({
    display_name: null,
    bio: null,
    avatar_url: null,
  });
  const [stats, setStats] = useState<UserStats>({
    totalPosts: 0,
    totalReplies: 0,
    totalLikesReceived: 0,
    joinedDate: null,
  });
  const [posts, setPosts] = useState<UserPost[]>([]);
  const [replies, setReplies] = useState<UserReply[]>([]);
  const [achievements, setAchievements] = useState<Achievement[]>([]);

  useEffect(() => {
    if (userId) {
      fetchUserData();
    }
  }, [userId]);

  const fetchUserData = async () => {
    if (!userId) return;
    setLoading(true);
    try {
      const [profileInfo, postsData, repliesData, achievementsData] = await Promise.all([
        userPublicApi.getProfile(userId).catch(() => null),
        communityApi.getPosts(undefined, 100, 0, userId),
        communityApi.getRepliesByUser(userId),
        achievementsApi.getAll(userId),
      ]);

      if (profileInfo) {
        setProfileData({
          display_name: profileInfo.display_name,
          bio: profileInfo.bio,
          avatar_url: profileInfo.avatar_url,
        });
        if (profileInfo.display_name) setUserName(profileInfo.display_name);
      }

      if (postsData && postsData.length > 0) {
        if (!userName && !profileInfo?.display_name) setUserName(postsData[0].author_name);
        setPosts(postsData);
        const totalLikes = postsData.reduce((sum: number, post: UserPost) => sum + (post.likes_count || 0), 0);
        setStats(prev => ({
          ...prev,
          totalPosts: postsData.length,
          totalLikesReceived: totalLikes,
          joinedDate: postsData[postsData.length - 1].created_at,
        }));
      }

      if (repliesData && repliesData.length > 0) {
        if (!userName) setUserName(repliesData[0].author_name);
        setReplies(repliesData);
        setStats(prev => ({ ...prev, totalReplies: repliesData.length }));
      }

      if (achievementsData) setAchievements(achievementsData);
    } catch (e) {
      console.error('Error fetching user data:', e);
    }
    setLoading(false);
  };

  const formatTimeAgo = (dateString: string) => {
    try {
      return formatDistanceToNow(new Date(dateString), { addSuffix: true });
    } catch {
      return 'recently';
    }
  };

  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  if (!userName) {
    return (
      <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
        <div className="container mx-auto px-4 py-8 max-w-4xl">
          <Link to="/community" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors mb-8">
            <ArrowLeft className="h-4 w-4" />
            Back to Community
          </Link>
          <Card>
            <CardContent className="p-12 text-center">
              <p className="text-muted-foreground">User not found or has no activity yet.</p>
            </CardContent>
          </Card>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-gradient-to-br from-emerald-50 via-white to-amber-50 dark:from-gray-900 dark:via-gray-800 dark:to-gray-900">
      <div className="container mx-auto px-4 py-8 max-w-4xl">
        {/* Header */}
        <Link to="/community" className="flex items-center gap-2 text-muted-foreground hover:text-foreground transition-colors mb-8">
          <ArrowLeft className="h-4 w-4" />
          Back to Community
        </Link>

        {/* Profile Header */}
        <Card className="mb-6">
          <CardContent className="p-6">
            <div className="flex items-start gap-4">
              <Avatar className="h-20 w-20">
                <AvatarImage src={profileData.avatar_url?.startsWith('/') ? `${API_BASE}${profileData.avatar_url}` : (profileData.avatar_url || undefined)} alt={userName} />
                <AvatarFallback className="text-2xl bg-primary/10 text-primary">
                  {userName.substring(0, 2).toUpperCase()}
                </AvatarFallback>
              </Avatar>
              
              <div className="flex-1">
                <div className="flex items-start justify-between gap-4">
                  <div>
                    <h1 className="text-2xl font-bold">{profileData.display_name || userName}</h1>
                    {profileData.bio && (
                      <p className="text-sm text-muted-foreground mt-1 mb-2">{profileData.bio}</p>
                    )}
                    {stats.joinedDate && (
                      <p className="text-xs text-muted-foreground flex items-center gap-1">
                        <Calendar className="h-3 w-3" />
                        Member since {format(new Date(stats.joinedDate), 'MMMM yyyy')}
                      </p>
                    )}
                  </div>
                  {userId && (
                    <div className="flex gap-2">
                      <FollowButton userId={userId} />
                      <MessageButton userId={userId} />
                    </div>
                  )}
                </div>
                
                {/* Follower Stats */}
                <div className="flex items-center gap-4 mt-3 text-sm">
                  <span>
                    <strong>{followStats.followersCount}</strong>{' '}
                    <span className="text-muted-foreground">followers</span>
                  </span>
                  <span>
                    <strong>{followStats.followingCount}</strong>{' '}
                    <span className="text-muted-foreground">following</span>
                  </span>
                </div>
              </div>
            </div>

            {/* Stats Grid */}
            <div className="grid grid-cols-4 gap-3 mt-6">
              <div className="text-center p-3 rounded-lg bg-muted/50">
                <MessageSquare className="h-5 w-5 mx-auto text-blue-500 mb-1" />
                <p className="text-xl font-bold">{stats.totalPosts}</p>
                <p className="text-xs text-muted-foreground">Posts</p>
              </div>
              <div className="text-center p-3 rounded-lg bg-muted/50">
                <MessageSquare className="h-5 w-5 mx-auto text-purple-500 mb-1" />
                <p className="text-xl font-bold">{stats.totalReplies}</p>
                <p className="text-xs text-muted-foreground">Replies</p>
              </div>
              <div className="text-center p-3 rounded-lg bg-muted/50">
                <Heart className="h-5 w-5 mx-auto text-red-500 mb-1" />
                <p className="text-xl font-bold">{stats.totalLikesReceived}</p>
                <p className="text-xs text-muted-foreground">Likes</p>
              </div>
              <div className="text-center p-3 rounded-lg bg-muted/50">
                <Users className="h-5 w-5 mx-auto text-emerald-500 mb-1" />
                <p className="text-xl font-bold">{followStats.followersCount}</p>
                <p className="text-xs text-muted-foreground">Followers</p>
              </div>
            </div>
          </CardContent>
        </Card>
        {/* Content Tabs */}
        <Tabs defaultValue="posts" className="space-y-4">
          <TabsList className="grid w-full grid-cols-3">
            <TabsTrigger value="posts" className="gap-2">
              <MessageSquare className="h-4 w-4" />
              Posts ({posts.length})
            </TabsTrigger>
            <TabsTrigger value="replies" className="gap-2">
              <TrendingUp className="h-4 w-4" />
              Activity ({replies.length})
            </TabsTrigger>
            <TabsTrigger value="achievements" className="gap-2">
              <Trophy className="h-4 w-4" />
              Badges ({achievements.length})
            </TabsTrigger>
          </TabsList>

          {/* Posts Tab */}
          <TabsContent value="posts" className="space-y-3">
            {posts.length === 0 ? (
              <Card>
                <CardContent className="p-8 text-center">
                  <MessageSquare className="h-10 w-10 mx-auto text-muted-foreground mb-2" />
                  <p className="text-muted-foreground">No posts yet</p>
                </CardContent>
              </Card>
            ) : (
              posts.map(post => {
                const CategoryIcon = categoryIcons[post.category] || MessageSquare;
                return (
                  <Card key={post.id}>
                    <CardContent className="p-4">
                      <div className="flex items-start gap-3">
                        <Badge variant="outline" className="text-xs gap-1">
                          <CategoryIcon className="h-3 w-3" />
                          {post.category}
                        </Badge>
                        <div className="flex-1">
                          <p className="text-sm mb-2">{post.content}</p>
                          <div className="flex items-center gap-4 text-xs text-muted-foreground">
                            <span className="flex items-center gap-1">
                              <Heart className="h-3 w-3" />
                              {post.likes_count} likes
                            </span>
                            <span className="flex items-center gap-1">
                              <MessageSquare className="h-3 w-3" />
                              {post.replies_count} replies
                            </span>
                            <span>{formatTimeAgo(post.created_at)}</span>
                          </div>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                );
              })
            )}
          </TabsContent>

          {/* Activity Tab */}
          <TabsContent value="replies" className="space-y-3">
            {replies.length === 0 ? (
              <Card>
                <CardContent className="p-8 text-center">
                  <TrendingUp className="h-10 w-10 mx-auto text-muted-foreground mb-2" />
                  <p className="text-muted-foreground">No activity yet</p>
                </CardContent>
              </Card>
            ) : (
              replies.slice(0, 20).map(reply => (
                <Card key={reply.id}>
                  <CardContent className="p-4">
                    <p className="text-sm mb-2">{reply.content}</p>
                    <p className="text-xs text-muted-foreground">
                      Replied {formatTimeAgo(reply.created_at)}
                    </p>
                  </CardContent>
                </Card>
              ))
            )}
          </TabsContent>

          {/* Achievements Tab */}
          <TabsContent value="achievements">
            {achievements.length === 0 ? (
              <Card>
                <CardContent className="p-8 text-center">
                  <Trophy className="h-10 w-10 mx-auto text-muted-foreground mb-2" />
                  <p className="text-muted-foreground">No badges earned yet</p>
                </CardContent>
              </Card>
            ) : (
              <div className="grid gap-3 sm:grid-cols-2">
                {achievements.map(achievement => {
                  const def = achievementDefinitions[achievement.achievement_id] || {
                    name: achievement.achievement_id,
                    description: 'Achievement unlocked',
                    icon: '🏅',
                  };
                  return (
                    <Card key={achievement.id}>
                      <CardContent className="p-4 flex items-center gap-3">
                        <span className="text-3xl">{def.icon}</span>
                        <div>
                          <p className="font-medium">{def.name}</p>
                          <p className="text-xs text-muted-foreground">{def.description}</p>
                          <p className="text-xs text-muted-foreground mt-1">
                            Earned {formatTimeAgo(achievement.earned_at)}
                          </p>
                        </div>
                      </CardContent>
                    </Card>
                  );
                })}
              </div>
            )}
          </TabsContent>
        </Tabs>
      </div>
      
      <Footer />
    </div>
  );
};

export default UserProfile;
