/**
 * Glycogenic Index (GI*) and Lipogenic Index (LI) from nutrition per 100g.
 * Formulas aligned with dashboard: EquationDisplay (GI* and LI equations).
 * Used as the reliable source when nutrition data exists; AI values are fallback only.
 */

export interface SimpleNutrition {
  carbs?: number | null;
  fiber?: number | null;
  protein?: number | null;
  fat?: number | null;
  sugar?: number | null;
}

function toNum(v: number | null | undefined): number {
  const n = Number(v);
  return Number.isFinite(n) ? n : 0;
}

/** True if we have enough nutrition to compute indices (at least carbs or fat present). */
export function hasEnoughNutritionForIndices(nutrition: SimpleNutrition | null | undefined): boolean {
  if (!nutrition) return false;
  const carbs = toNum(nutrition.carbs);
  const fat = toNum(nutrition.fat);
  const protein = toNum(nutrition.protein);
  return carbs > 0 || fat > 0 || protein > 0;
}

/**
 * GI* = (Cnet × kabs × Isec) / (Ffiber + Pbuffer)
 * Cnet = net digestible carbs, Pbuffer = protein × 0.5. Same structure as dashboard.
 */
export function estimateGlycogenicIndex(nutrition: SimpleNutrition): number | null {
  const carbs = toNum(nutrition.carbs);
  const fiber = toNum(nutrition.fiber);
  const protein = toNum(nutrition.protein);
  const netCarbs = Math.max(0, carbs - fiber);
  const denominator = fiber + protein * 0.5 + 0.1;
  if (denominator <= 0) return null;
  const kabs = 0.85;
  const sugar = toNum(nutrition.sugar);
  const Isec = carbs > 0 && sugar >= carbs * 0.5 ? 0.95 : 0.75;
  const raw = (netCarbs * kabs * Isec) / denominator;
  const gi = Math.max(0, Math.min(3, raw));
  return Math.round(gi * 100) / 100;
}

/**
 * LI = ((Isec × Cnet) + (Fcomb × kDNL)) / (AMPKscore + Meff)
 * Same structure as dashboard; estimated factors when not available.
 */
export function estimateLipogenicIndex(nutrition: SimpleNutrition): number | null {
  const carbs = toNum(nutrition.carbs);
  const fiber = toNum(nutrition.fiber);
  const protein = toNum(nutrition.protein);
  const fat = toNum(nutrition.fat);
  const netCarbs = Math.max(0, carbs - fiber);
  const Isec = 0.75;
  const Fcomb = fat > 15 ? 1.4 : fat > 8 ? 1.1 : 0.8;
  const kDNL = 0.4;
  const numerator = Isec * netCarbs + Fcomb * fat * (kDNL / 10);
  const ampkScore = (fiber + protein) / 20 + 0.5;
  const meff = 0.5;
  const denominator = ampkScore + meff + 0.1;
  if (denominator <= 0) return null;
  const raw = numerator / denominator;
  const li = Math.max(0, Math.min(3, raw));
  return Math.round(li * 100) / 100;
}

/**
 * TOBIN total (0-50) from GI* and LI when we don't have full TOBIN breakdown.
 * Consistent mapping so calculated stats are reliable.
 */
export function estimateTobinFromIndices(gi: number | null, li: number | null): number | null {
  if (gi == null && li == null) return null;
  const g = gi ?? 0.5;
  const l = li ?? 0.5;
  const score = Math.min(50, Math.round(g * 15 + l * 20));
  return Math.max(0, score);
}
