import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import { phytoestrogensDatabase, PhytoestrogensData } from "@/data/chemicalDatabase";

const potencyOrder = { strong: 0, moderate: 1, weak: 2 };
const classLabels: Record<string, string> = {
  isoflavone: "Isoflavones",
  lignan: "Lignans",
  coumestan: "Coumestans",
  stilbene: "Stilbenes",
  flavonoid: "Flavonoids",
  mycoestrogen: "Mycoestrogens",
  other: "Other Compounds",
};

export function generatePhytoestrogenPdf() {
  const doc = new jsPDF();
  const pageWidth = doc.internal.pageSize.getWidth();
  let yPos = 20;

  // Title
  doc.setFontSize(24);
  doc.setTextColor(220, 38, 38); // Red color
  doc.text("PHYTOESTROGEN AVOIDANCE GUIDE", pageWidth / 2, yPos, { align: "center" });
  
  yPos += 10;
  doc.setFontSize(12);
  doc.setTextColor(100);
  doc.text("Complete List of Estrogenic Compounds in Foods", pageWidth / 2, yPos, { align: "center" });
  
  yPos += 8;
  doc.setFontSize(10);
  doc.setTextColor(150);
  doc.text(`Generated: ${new Date().toLocaleDateString()}`, pageWidth / 2, yPos, { align: "center" });

  // Warning box
  yPos += 12;
  doc.setFillColor(254, 242, 242);
  doc.setDrawColor(220, 38, 38);
  doc.roundedRect(14, yPos, pageWidth - 28, 20, 3, 3, "FD");
  
  doc.setFontSize(10);
  doc.setTextColor(153, 27, 27);
  doc.text("⚠️ RECOMMENDATION: AVOID ALL LISTED COMPOUNDS", pageWidth / 2, yPos + 8, { align: "center" });
  doc.setFontSize(8);
  doc.setTextColor(100);
  doc.text("Phytoestrogens mimic estrogen and may disrupt hormonal balance", pageWidth / 2, yPos + 14, { align: "center" });

  yPos += 28;

  // Summary statistics
  const totalCompounds = phytoestrogensDatabase.length;
  const strongCount = phytoestrogensDatabase.filter(p => p.potency === "strong").length;
  const moderateCount = phytoestrogensDatabase.filter(p => p.potency === "moderate").length;
  const weakCount = phytoestrogensDatabase.filter(p => p.potency === "weak").length;

  doc.setFontSize(11);
  doc.setTextColor(50);
  doc.text("Summary:", 14, yPos);
  yPos += 6;
  doc.setFontSize(9);
  doc.text(`Total Compounds: ${totalCompounds} | Strong Potency: ${strongCount} | Moderate: ${moderateCount} | Weak: ${weakCount}`, 14, yPos);
  
  yPos += 10;

  // Group by class and sort by potency
  const groupedByClass = phytoestrogensDatabase.reduce((acc, compound) => {
    if (!acc[compound.class]) acc[compound.class] = [];
    acc[compound.class].push(compound);
    return acc;
  }, {} as Record<string, PhytoestrogensData[]>);

  // Sort each class by potency
  Object.keys(groupedByClass).forEach(cls => {
    groupedByClass[cls].sort((a, b) => potencyOrder[a.potency] - potencyOrder[b.potency]);
  });

  // Order classes
  const classOrder = ["isoflavone", "lignan", "coumestan", "stilbene", "flavonoid", "mycoestrogen", "other"];

  classOrder.forEach(cls => {
    const compounds = groupedByClass[cls];
    if (!compounds || compounds.length === 0) return;

    // Check if we need a new page
    if (yPos > 250) {
      doc.addPage();
      yPos = 20;
    }

    // Class header
    doc.setFontSize(14);
    doc.setTextColor(50);
    doc.text(`${classLabels[cls]} (${compounds.length})`, 14, yPos);
    yPos += 4;

    // Table for this class
    const tableData = compounds.map(compound => [
      compound.name,
      compound.potency.toUpperCase(),
      compound.foodSources.slice(0, 5).join(", ") + (compound.foodSources.length > 5 ? "..." : ""),
      compound.healthConcerns.slice(0, 2).join("; ") + (compound.healthConcerns.length > 2 ? "..." : ""),
    ]);

    autoTable(doc, {
      startY: yPos,
      head: [["Compound", "Potency", "Food Sources", "Health Concerns"]],
      body: tableData,
      theme: "striped",
      headStyles: {
        fillColor: [220, 38, 38],
        textColor: 255,
        fontSize: 9,
        fontStyle: "bold",
      },
      bodyStyles: {
        fontSize: 8,
        textColor: 50,
      },
      columnStyles: {
        0: { cellWidth: 35, fontStyle: "bold" },
        1: { cellWidth: 22, halign: "center" },
        2: { cellWidth: 60 },
        3: { cellWidth: 60 },
      },
      margin: { left: 14, right: 14 },
      didParseCell: (data) => {
        if (data.section === "body" && data.column.index === 1) {
          const potency = data.cell.raw?.toString().toLowerCase();
          if (potency === "strong") {
            data.cell.styles.textColor = [185, 28, 28];
            data.cell.styles.fontStyle = "bold";
          } else if (potency === "moderate") {
            data.cell.styles.textColor = [194, 65, 12];
          } else {
            data.cell.styles.textColor = [161, 98, 7];
          }
        }
      },
    });

    yPos = (doc as any).lastAutoTable.finalY + 10;
  });

  // Add detailed appendix
  doc.addPage();
  yPos = 20;

  doc.setFontSize(18);
  doc.setTextColor(50);
  doc.text("DETAILED COMPOUND REFERENCE", pageWidth / 2, yPos, { align: "center" });
  yPos += 15;

  // Full details for each compound
  phytoestrogensDatabase
    .sort((a, b) => potencyOrder[a.potency] - potencyOrder[b.potency])
    .forEach(compound => {
      if (yPos > 250) {
        doc.addPage();
        yPos = 20;
      }

      // Compound name with potency badge
      doc.setFontSize(11);
      doc.setTextColor(50);
      doc.text(`${compound.name}`, 14, yPos);
      
      // Potency indicator
      const potencyColors: Record<string, number[]> = {
        strong: [220, 38, 38],
        moderate: [234, 88, 12],
        weak: [202, 138, 4],
      };
      const color = potencyColors[compound.potency];
      doc.setFontSize(8);
      doc.setTextColor(color[0], color[1], color[2]);
      doc.text(`[${compound.potency.toUpperCase()}]`, 14 + doc.getTextWidth(compound.name) + 3, yPos);
      
      yPos += 5;
      
      // Class
      doc.setFontSize(8);
      doc.setTextColor(100);
      doc.text(`Class: ${classLabels[compound.class]}`, 14, yPos);
      yPos += 4;

      // Food sources
      doc.setTextColor(70);
      const foodSourcesText = `Foods: ${compound.foodSources.join(", ")}`;
      const splitFoods = doc.splitTextToSize(foodSourcesText, pageWidth - 28);
      doc.text(splitFoods, 14, yPos);
      yPos += splitFoods.length * 4;

      // Health concerns
      doc.setTextColor(153, 27, 27);
      const concernsText = `Concerns: ${compound.healthConcerns.join("; ")}`;
      const splitConcerns = doc.splitTextToSize(concernsText, pageWidth - 28);
      doc.text(splitConcerns, 14, yPos);
      yPos += splitConcerns.length * 4;

      // Notes if present
      if (compound.notes) {
        doc.setTextColor(100);
        const notesText = `Note: ${compound.notes}`;
        const splitNotes = doc.splitTextToSize(notesText, pageWidth - 28);
        doc.text(splitNotes, 14, yPos);
        yPos += splitNotes.length * 4;
      }

      yPos += 6;
    });

  // Footer on last page
  const pageCount = doc.getNumberOfPages();
  for (let i = 1; i <= pageCount; i++) {
    doc.setPage(i);
    doc.setFontSize(8);
    doc.setTextColor(150);
    doc.text(
      `Page ${i} of ${pageCount} | Metabolic Food Scanner - Phytoestrogen Avoidance Guide`,
      pageWidth / 2,
      doc.internal.pageSize.getHeight() - 10,
      { align: "center" }
    );
  }

  // Save the PDF
  doc.save("phytoestrogen-avoidance-guide.pdf");
}
